/*-----------------------------------------------------------------
 *	$Id: x2sys_get.c,v 1.9 2004/09/23 00:26:11 pwessel Exp $
 *
 *      Copyright (c) 1999-2004 by P. Wessel
 *      See COPYING file for copying and redistribution conditions.
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; version 2 of the License.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      Contact info: gmt.soest.hawaii.edu
 *--------------------------------------------------------------------*/
/* x2sys_get will read the track index database and report all the tracks
 * that matches the specified geographical or data-type criteria given
 * on the command line.
 *
 * Author:	Paul Wessel
 * Date:	14-JUN-2004
 * Version:	1.1, based on the spirit of the old mgg code
 *
 */

#include "x2sys.h"

main (int argc, char **argv)
{
	char *TAG = CNULL, *fflags = CNULL, *mark;

	struct X2SYS_INFO *s;
	struct X2SYS_BIX B;
	struct X2SYS_BIX_TRACK *track;
	
	BOOLEAN error = FALSE, no_suffix = FALSE;
	
	int i, j, k, combo, ij, n_tracks_found, n_tracks, bit, start_j, start_i, stop_j, stop_i;
	
	double west = 0.0, east = 0.0, south = 0.0, north = 0.0;
	
	GMT_begin (argc, argv);
	for (i = strlen(argv[0]); i >= 0 && argv[0][i] != '/'; i--);
	X2SYS_program = &argv[0][i+1];	/* Name without full path */
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				case 'V':
				case 'R':
				case '\0':
					error += GMT_get_common_args (argv[i], &west, &east, &south, &north);
					break;
				case 'F':
					fflags = &argv[i][2];
					break;
				case 'S':
					no_suffix = TRUE;
					break;
				case 'T':
					TAG = &argv[i][2];
					break;
				default:
					error = TRUE;
					fprintf (stderr, "%s: Unrecognized option -%c\n", GMT_program, argv[i][1]);
					break;
			}
		}
	}
	
	if (argc == 1 || error || GMT_quick) {
		fprintf (stderr, "x2sys_get %s - Get track listing from track index database\n\n", X2SYS_VERSION);
		fprintf (stderr, "usage: x2sys_get -T<TAG> [-F<fflags>] [-R<region>] [-V]\n\n");

		if (GMT_quick) exit (EXIT_FAILURE);

		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "	-F is comma-separated list of column names that must be present [Default is any field]\n");
		GMT_explain_option ('R');
		fprintf (stderr, "	-S reports tracks files without suffiz [Default is file name]\n");
		fprintf (stderr, "	[Default region is the entire data domain]\n");
		GMT_explain_option ('V');
		exit (EXIT_FAILURE);
	}
	
	x2sys_set_system (TAG, &s, &B, &GMT_io);

	if (west == east && south == north) {	/* Set default region */
		west = B.x_min;		east = B.x_max;
		south = B.y_min;	north = B.y_max;
	}
	
	if (fflags) x2sys_pick_fields (fflags, s);
	for (i = combo = 0; i < s->n_out_columns; i++) combo |= (1 << s->out_order[i]);

	x2sys_bix_init (&B, FALSE);

	/* Read existing track-information from <ID>_tracks.d file */
	
	n_tracks = x2sys_bix_read_tracks (TAG, &B, 1);
	
	mark = (char *) GMT_memory (VNULL, (size_t)n_tracks, sizeof (char), GMT_program);

	/* Read geographical track-info from <ID>_index.b file */
	
	x2sys_bix_read_index (TAG, &B);
	
	/* Ok, now we can start finding the tracks requested */
	
	(void) x2sys_bix_get_ij (west, south, &start_i, &start_j, &B);
	(void) x2sys_bix_get_ij (east, north, &stop_i, &stop_j, &B);
	
	for (j = start_j, n_tracks_found = 0; j <= stop_j; j++) {
		for (i = start_i; i <= stop_i; i++) {
			ij = j * B.nx_bin + i;
			if (B.base[ij].n_tracks == 0) continue;
			
			for (k = 0, track = B.base[ij].first_track->next_track; k < B.base[ij].n_tracks; k++, track = track->next_track) {
				if ((!fflags || (track->track_flag & combo) == combo) && mark[track->track_id] == 0) {
					mark[track->track_id] = 1;
					n_tracks_found++;
				}
			}
		}
	}
	
	if (n_tracks_found) {
		if (gmtdefs.verbose ) fprintf (stderr, "%s: Found %d tracks\n", GMT_program, n_tracks_found);
	
		printf ("# Search command: %s", GMT_program);
		for (i = 1; i < argc; i++) printf (" %s", argv[i]);
		printf ("\n# trackname\t\t");
		for (i = 0; i < (s->n_fields-1); i++) printf ("%s\t", s->info[i].name);
		printf ("%s\n", s->info[s->n_fields-1].name);
		for (k = 0; k < n_tracks; k++) {
			if (mark[k] == 0) continue;
			if (no_suffix) {
				for (i = strlen (B.head[k].trackname) - 1; i > 0 && B.head[k].trackname[i] != '.'; i--);
				B.head[k].trackname[i] = '\0';
				printf ("%s\t", B.head[k].trackname);
			}
			else
				printf ("%s\t", B.head[k].trackname);
			for (i = 0, bit = 1; i < s->n_fields; i++, bit <<= 1) {
				(B.head[k].flag & bit) ? printf ("\tY") : printf ("\tN");
			}
			printf ("\n");
		}
	}
	else
		if (gmtdefs.verbose ) fprintf (stderr, "%s: Search found no tracks\n", GMT_program);
		
	GMT_free ((void *)B.base);
	GMT_free ((void *)B.head);
	GMT_free ((void *)mark);
	x2sys_free_info (s);
	
	if (gmtdefs.verbose) fprintf (stderr, "%s completed successfully\n", GMT_program);
	
	GMT_end (argc, argv);
}
