-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-connection.ads,v $
--  Description     : Ada Database Objects - Connection object               --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/02/16 22:08:49 $
--  Version         : $Revision: 1.2 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This is the connection object. The connection object combines all        --
--  objects such as statements and the underlying driver.                    --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Finalization;                  use Ada.Finalization;

--* ADO
with GNU.DB.ADBC.Driver;                use GNU.DB.ADBC.Driver;
use  GNU.DB.ADBC;

package GNU.DB.ADBC.Connection is

   type Object( Db : Driver.Handle ) is new Controlled with private;
   type Handle is access all Object;

   Null_ID : constant Connection_ID := Connection_ID'First;
   ---------------------------------------------------------------------------
   --# Description    :
   --#    Connect to the data base as is described by the data base object
   --#    This method will invoke the finaly the Execute_Connect of the
   --#    underlying data base.
   --# Preconditions  :
   --#    R.1 - All data required by the underlying dbcs are set.
   --# Postconditions :
   --#    P.1 - Connection to the data base established and the connection s
   --#          tatus has been updated
   --#    P.2 - The connection status is updated according to the connection
   --#          result.
   --# Exceptions     :
   --#    Exceptions are gnerated by the implementation based upon the
   --#    database behaviour.
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Connect(
      This       : in out Object;
      Name       : in String;
      Password   : in String := Default_Password;
      Database   : in String := Default_Database;
      Parameters : in String := "" );


   Not_Connected : exception;

   function Driver_Handle(
      This : in Object ) return Driver.Handle;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Disconnect from the data base
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --#    P.1 - The connection status not_defined
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   procedure Disconnect(
      This : in out Object );

   ---------------------------------------------------------------------------
   --# Description    :
   --#    Prepare a statement for the data base.
   --#    Foe example ths statement
   --#       select * from employees where empno = :my_empno
   --#
   --#     is split into a usable set of strings.
   --#
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --# Postconditions :
   --#    P.1 - A statement id is allocated to the statement.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   function Prepare(
      This      : in Object;
      Statement : in String ) return Statement_ID;

   Overflow : exception;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    This method executes a prepared statement.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --#    R.2 - the statement has been prepared unless a direct SQL statement
   --# Postconditions :
   --#    P.1 - A statement id is allocated to the statement.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   function Execute(
      This      : in Object;
      Stmt      : in Statement_ID ) return Resultset_ID;

   ---------------------------------------------------------------------------
   --# Description    :
   --#    This method executes a prepared statement.
   --# Preconditions  :
   --#    R.1 - The data base is connected
   --#    R.2 - the statement has been prepared unless a direct SQL statement
   --# Postconditions :
   --#    P.1 - A statement id is allocated to the statement.
   --# Exceptions     :
   --#    R.1 violated - Usage_Error
   --# Note           :
   ---------------------------------------------------------------------------
   function Execute(
      This      : in Object;
      Statement : in String ) return Resultset_ID;


   function Driver_Handle(
      Id        : Connection_ID ) return Driver.Handle;

   procedure Add_Statement(
      Con  : in Connection_ID;
      Stmt : in Statement_ID );

   procedure Delete_Statement(
      Con  : in Connection_ID;
      Stmt : in Statement_ID );


   ---=====================================================================---
private

   type Object_Data;
   type Object_Data_Access is access Object_Data;

   type Object( Db : Driver.Handle ) is new Controlled with record
         Data : Object_Data_Access := null;
      end record;

   procedure Initialize(
      This : in out Object );

   procedure Finalize(
      This : in out Object );

   function Self(
      This : in Object ) return Handle;

end GNU.DB.ADBC.Connection;

