------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2006 Maxim Reznik.                                        --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

with Ada.Text_IO;
with Ada.Command_Line;

with OCI.Lib;
with OCI.Thick.Object;
with OCI.Thick.Statements;
with OCI.Thick.Connections;
with OCI.Thick.String_Bounded;
with OCI.Thick.Number_Functions;

procedure Object_Test is
   use OCI.Thick;
   use type OCI.Lib.Sb2;

   --  Execute this in database:
   --  create or replace type Test as object (Left Integer, Right Integer);

   type Object_Node is record
      Left, Right : OCINumber;
   end record;
   --  Corresponding Ada type

   type Indicator is record
      Whole       : OCI.Lib.Sb2;    --  object is null
      Left, Right : OCI.Lib.Sb2;    --  field is null
   end record;
   --  Corresponding nullity information

   package Test_Type is
      new Object (Object_Node, Indicator, Type_Name => "TEST");

   Session : constant Connections.Connection :=
     Connections.Logon (Ada.Command_Line.Argument (1));

   Stmt    : Statements.Statement;
   Text    : String_Bounded.Variable (10);
   Var     : Test_Type.Variable;

begin
   --  Use object as IN parameter

   Test_Type.Create (Session, Var);

   Test_Type.Get_Value (Var).all.Left
     := Number_Functions.To_Number ("1", "99");
   Test_Type.Get_Value (Var).all.Right
     := Number_Functions.To_Number ("2", "99");
   Test_Type.Get_Indicator (Var).all
     := (others => OCI.Lib.Not_Null_Indicator);

   Stmt := Statements.Prepare
     (Session,
      "declare X Test := :Param; begin :Res := X.Left + X.Right; end;");

   Test_Type.Bind (Stmt, Var, 1);
   String_Bounded.Bind (Stmt, Text, 2);
   Statements.Execute (Stmt);
   Ada.Text_IO.Put_Line ("1 + 2 = " & String_Bounded.Get_Value (Text));

   --  Use object as OUT parameter

   Stmt := Statements.Prepare (Session, "begin :Result := Test(2, 1); end;");

   Test_Type.Bind (Stmt, Var, 1);
   Statements.Execute (Stmt);

   declare
      Result : constant Object_Node := Test_Type.Get_Value (Var);
      Ind    : constant Indicator   := Test_Type.Get_Indicator (Var);
   begin
      if Ind.Whole = OCI.Lib.Null_Indicator then
         Ada.Text_IO.Put_Line ("This is null object");
      else
         if Ind.Whole = OCI.Lib.Null_Indicator then
            Ada.Text_IO.Put_Line ("Result.Left is null");
         else
            Ada.Text_IO.Put_Line
              ("Result.Left="
               & Number_Functions.To_String (Result.Left, "99"));
         end if;

         if Ind.Whole = OCI.Lib.Null_Indicator then
            Ada.Text_IO.Put_Line ("Result.Right is null");
         else
            Ada.Text_IO.Put_Line
              ("Result.Right="
               & Number_Functions.To_String (Result.Right, "99"));
         end if;
      end if;
   end;

   Test_Type.Free (Var);
end Object_Test;
