/* Low level interface for debugging FSU threads for GDB, the GNU debugger.
   Copyright 1998 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_SYS_PROCFS_H
#include <sys/procfs.h>
#endif

#include "defs.h"
#include "target.h"
#include "inferior.h"
#include "gdbcore.h"
#include "gdb_proc_service.h"

#ifdef __linux__

/*
setjmp.h gives us

#define JB_BX   0
#define JB_SI   1
#define JB_DI   2
#define JB_BP   3
#define JB_SP   4
#define JB_PC   5
*/

#include <asm/ptrace.h>
/* used for the following defines:
#define EBX 0
#define UESP 15
#define EBP 5
#define ESI 3
#define EDI 4
#define EIP 12
*/

static int registers_map [] =
  { /* JB_BX => */ EBX,
    /* JB_SI => */ ESI,
    /* JB_DI => */ EDI,
    /* JB_BP => */ EBP,
    /* JB_SP => */ UESP,
    /* JB_PC => */ EIP};
#endif /* __linux__ */

extern unsigned long current_thread;
extern CORE_ADDR pthread_kern_addr;

gdb_gregset_t *jmpbuf2regset (int *regs)
{
  static gdb_gregset_t ret;
  int j;

  for (j = 0; j < sizeof (registers_map) / sizeof (int); j++)
    ret [registers_map [j]] = regs [j];

  return &ret;
}

int
fsu_thread_fetch_registers ()
{
  int tcb [sizeof (registers_map) / sizeof (int)];

  /* Get the general regs.
     We rely on the fact that the registers are stored in the first bytes
     of the tcb (true as long as tcb.context is the first field in pthread */

  if (current_thread == 0)
    return -1;

  read_memory ((CORE_ADDR) current_thread, (char*) &tcb, sizeof (tcb));

  supply_gregset (jmpbuf2regset (tcb));
  return 0;
}

unsigned long fsu_thread_get_current_thread ()
{
  unsigned long thread;

  if (pthread_kern_addr == 0)
    return 0L;

  read_memory (pthread_kern_addr, (char*) &thread, sizeof (thread));
  return thread;
}
