/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// BinauralBeat.h: interface for the CBinauralBeat class.
// Author: Bret Logan
/////////////////////////////////////////////////////////////////////////////// 

#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#define BB_PRECISION_TYPE double
#define BB_INTEGRATION_REZ 441  //larger the number, less I integrate freq. 441 = every .01 sec.
#define BB_VOLUMETONERANGE 16384

const BB_PRECISION_TYPE BB_TWO_PI = 3.1415926535897932384626433832795 * 2.0;
const BB_PRECISION_TYPE BB_SAMPLE_FACTOR = ((3.1415926535897932384626433832795 * 2.0) / 44100.0);       // == 0.000142476
//const BB_PRECISION_TYPE SAMPLE_FACTOR = 0.0000226757369614512471655328798185941f; // 1.0/44100.0

const unsigned int BB_COMPLETED = 1;
const unsigned int BB_NEWLOOP = 2;
const unsigned int BB_NEWENTRY = 4;
//const unsigned int BB_INIT = 8;
//const unsigned int BB_ERROR = 16;
//const unsigned int BB_NEWFREQ = 32;
//const unsigned int BB_INFO = 64;

class CItinerary
{
public:
 BB_PRECISION_TYPE FreqREnd;
 BB_PRECISION_TYPE FreqLEnd;
 BB_PRECISION_TYPE FreqLStart;
 BB_PRECISION_TYPE FreqRStart;
 BB_PRECISION_TYPE Duration;
 BB_PRECISION_TYPE FreqLSpread;
 BB_PRECISION_TYPE FreqRSpread;
 unsigned int AbsoluteEndTime_100;      //real time in schedule, in 100ths of a sec.
 unsigned int PrevAbsoluteEndTime_100;  //real time in schedule, in 100ths of a sec.

// START CItinerary class
 CItinerary ()
 {
  FreqLStart = 0.f;
  FreqRStart = 0.f;
  FreqLEnd = 0.f;
  FreqREnd = 0.f;
  Duration = 10.f;
  FreqLSpread = 0.f;
  FreqRSpread = 0.f;
  AbsoluteEndTime_100 = 0;
  PrevAbsoluteEndTime_100 = 0;
 }

 CItinerary (BB_PRECISION_TYPE lfrequency, BB_PRECISION_TYPE rfrequency,
             BB_PRECISION_TYPE duration, BB_PRECISION_TYPE spreadl,
             BB_PRECISION_TYPE spreadr)
 {
  FreqLStart = lfrequency;
  FreqRStart = rfrequency;
  Duration = duration;
  FreqLSpread = spreadl;
  FreqRSpread = spreadr;
 }
};

// END CItinerary class

//START CBinauralBeat class:
class CBinauralBeat
{
public:
 CBinauralBeat (char *sched_filename = DefaultSchedFilename);
//the only Constructor
//SchedFilenameToSchedule() opens whatever user has set SchedFilename
//to in Constructor, dumps it in to SchedBuff, then uses SchedBuffSchedFilenameToSchedule()
//to fill Schedule with the contents of SchedBuff
 int SchedFilenameToSchedule ();
//SchedBuffToSchedule uses a SchedBuff (NOT SchedFilename) to create a Schedule, and
//then (important to remember) writes the schedule to a file indicated by SchedFilename,
//essentially either reformatting a user's existing schedule file or creating a new one if none.
//It is separate from SchedFilenameToSchedule because I have an internal SchedBuff 
//called DefaultSchedBuff
 void SchedBuffToSchedule (char *str);
 int SchedFilenameToSchedule_OriginalFileFormat ();
 char *SchedFilename;
 bool WriteWAVFile (char *szFilename);
 bool WriteWAVFileToStream (FILE * stream = NULL);
 bool WriteWAVFileToSTDOUT ();
 void SetDefaultValues ();
 void WriteScheduleToFile_oldformat (char *file_name);  //handles original file format
 void WriteScheduleToFile (char *file_name);    //handles new Gnaural2 writing
 void WriteDefaultScheduleToFile ();
 void Schedule_CalculateIntervals ();   //essential precalc of all values implicit in provided schedule data
 int Schedule_Allot (int entrycount);

//User directly sets FreqBase to set base frequency:
 BB_PRECISION_TYPE FreqBase;
//set ManualFreqOffsetControl true to let next ManualFreqLOffset and ManualFreqROffset 
//control beat frequency without heeding Schedule:
 bool ManualFreqOffsetControl;
 BB_PRECISION_TYPE ManualFreqLOffset;
 BB_PRECISION_TYPE ManualFreqROffset;

 int StereoSwap;                //set true to swap left and right stereo channels
 BB_PRECISION_TYPE VolumeOverall_left;  //set between 0 to 1 equally attenutate both Noise and Tone on left side
 BB_PRECISION_TYPE VolumeOverall_right; //set between 0 to 1 equally attenutate both Noise and Tone on right side
 BB_PRECISION_TYPE Volume_Tone; // range: int 0 to 1 (later gets multiplied by BB_VOLUMETONERANGE)
 BB_PRECISION_TYPE Volume_Noiz; //range: BB_PRECISION_TYPE 0 to 1
 int StereoNoiz;

 unsigned int InfoFlag;         //BB uses this only to send messages to the user, who must reset the messages.
 static void UserSoundProc (void *pSoundBuffer, long bufferLenInBytes);
 void Reset ();
 unsigned int FileByteCount;
 bool WriteStop;
 void Mute (bool act);
 void ParseCmd (FILE * stream);
 bool FileFlag;
 void next ();
 void previous ();
 void MainLoop (void *pSoundBuffer, long bufferLen);
 //CBinauralBeat(BB_PRECISION_TYPE basefreq, char *tmpbuff, int _loop);
 static CBinauralBeat *me;
 virtual ~ CBinauralBeat ();
 BB_PRECISION_TYPE FreqL;
 BB_PRECISION_TYPE FreqR;
 int loops;                     //hold total number of loops intended to be done (0 means infinite)
  int loopcount;//keeps track of current loop number
 int ScheduleEntriesCount;      //always holds the total number of entries in *Schedule
 int ScheduleCount;             //indicates which entry BinauralBeat is currently at in *Schedule
 BB_PRECISION_TYPE ScheduleTime;        //total projected time-length of schedule in seconds
 //NOTE: Confusion between SchedFilename and SchedBuff has made for some serious bugs.
 //SchedFilename is the actual string containing the filename (with prepended path),
 //while SchedBuff is really just a gigantic string representation of the kind of data actually
 //found *inside* a valid schedule file (look at DefaultSchedBuff for an example).
 void InitBinauralBeat ();
 unsigned int TotalSampleCount; //sample count/100 -- basically translates 100 = 1 second
 unsigned long TotalSampleCountLooped;  //sum of all TotalSampleCount values per loop
 CItinerary *Schedule;          //this holds all the schedule values
 char *SchedBuff;               //users would not normally set this; it exists separately because users don't always have schedule files
 static char DefaultSchedBuff[];
 static char DefaultSchedFilename[];

//### pseudorandom number generator funcs and vars:
// int Seed; // I think int is OK too
 void SeedRand (unsigned int i1, unsigned int i2 = -1);
 unsigned long mcgn, srgn;
 int rand ();
//### end pseudorandom number generator funcs and vars:  

//these globals are needed only for XML file reading:
 int BB_XMLEventDataParser (const gchar * DataType, const gchar * Value);
 int BB_XMLReadFile (char *filename);
 void BB_XMLParser (const gchar * CurrentElement,       //this must always be a valid string
                    const gchar * Attribute,    //beware: this will equal NULL if there are none
                    const gchar * Value);       //beware: this will equal NULL to announce end of CurrentElement
//the next function only passes it's contents on to BB_XMLParser
 static void BB_XMLParser_callback (const gchar * CurrentElement,
                                    const gchar * Attribute,
                                    const gchar * Value);
 int BB_ParserXML_voicecount;
 int BB_ParserXML_curvoice;
 int BB_ParserXML_curentry;

/*
Waveform Audio File Structure (Type - Size -	Value):
First "Chunk" ID 	4 bytes (char [4]) 	"RIFF"
File Size 	4 bytes (long) 	Size of the file in bytes
Form Type ID 	4 bytes (char [4]) 	"WAVE"
Second "Chunk" ID 	4 bytes (char [4]) 	"fmt "
Wave Format Size 	4 bytes (long) 	Size in bytes of the Wave Format Info
Wave Format Info 	Wave Format Size 	Information on the format of the PCM data.
Third "Chunk" ID 	4 bytes (char [4]) 	"data"
Data Size 	4 bytes (long) 	Size in bytes of the PCM data
Data 	Data Size 	The actual PCM data
*/
//typedef unsigned short    word;                       //16 bits
//  typedef unsigned int    dword;                      //32 bits 
 struct WAVheader
 {
  char ckID[4];                 // chunk id 'RIFF'            
  unsigned int ckSize;          // chunk size [it is total filesize-8]
  char wave_ckID[4];            // wave chunk id 'WAVE'       
  char fmt_ckID[4];             // format chunk id 'fmt '     
  unsigned int fmt_ckSize;      // format chunk size          
  unsigned short formatTag;     // format tag currently pcm   
  unsigned short nChannels;     // number of channels         
  unsigned int nSamplesPerSec;  // sample rate in hz          
  unsigned int nAvgBytesPerSec; // average bytes per second
  unsigned short nBlockAlign;   // number of bytes per sample 
  unsigned short nBitsPerSample;        // number of bits in a sample 
  char data_ckID[4];            // data chunk id 'data'       
  unsigned int data_ckSize;     // length of data chunk [byte count of actual data (not descriptors)]
 } wavh;

 void WAVheader_fill ();        //NOTE: uses whatever value is in FileByteCount for size values
};

//END CBinauralBeat class:
