/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
	
////////////////////////////////////////////////////////////
//	ScheduleGUI.cpp, by Bret Logan (c) 2006 
//A user-manipulable plot for creating Gnaural Schedule files
//To compile ScheduleGUI for standalone:
//g++ ScheduleGUI.cpp -o ScheduleGUI -D SCHEDULEGUI_FREESTANDING `pkg-config --cflags --libs gtk+-2.0`
//To use with another project, include ScheduleGUI.h and rename main();
////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <sys/time.h> //needed only for utility Stopwatch()
#include <time.h>
#include <string.h> //needed for memcpy()
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h> //needed for keyboard translations




//PUT STUFF SPECIFIC TO CALLING PROGRAM HERE:
//These are required only for Gnaural project (not for freestanding):
#ifndef SCHEDULEGUI_FREESTANDING
#include "BinauralBeat.h"
#include "gnaural.h"
#define DBGOUT(a,b)    if (cmdline_d==true)  fprintf(stderr,"Gnaural ScheduleGUI: %s %d\n",a, b);
//#define ERROUT(a)   fprintf(stderr,"ScheduleGUI: #Error# %s\n",a) 
//#define DBGBEEP  putc('\a',stderr);
#else
#define DBGOUT(a,b)    fprintf(stderr,"ScheduleGUI: %s %d\n",a, b);
#endif




//TODO: 
// - make it so I don't have to send InsertDatapoint() a summed number (it is just getting diff anyway)
//- implement schedule time-stretching by pulling rightmost point
//- multiple undos
// - allow user to set font/colors for graph and a menu heading to reset graph to the current (real) data BB is using
// - hovering mouse over a datapoint should give a tooltip with the point's info
// - allow for space around graph so user can grab points easier
// - when user deletes rightmost point on graph it gets replaced by second to rightmost (actual last DataPoint) values -- that it doesn't now is non-obvious for user.
// - make it so that when user right-clicks (or anyclick, I guess) a datapoint it changes color or in some way let's user know it is selected
// - create a zoom function, also scrolling ability (via side scrollbars)
// - provide a function that deletes redundant points (points in which hz doesn't change over three points)
// - allow user to choose colors, get default background from configuration

//BUGS:
//- deleting the remaining visible point does strange stuff, but apparently not destructive, so keep it for later
//- totalscheduleduration can't be 0 because divide-by-0 errors will crash app (fudging it for now by never letting it)




//keep squaresize even (I guess):
#define SCHEDULEGUI_SQUARESIZE 6
#define SCHEDULEGUI_GRAPHBOUNDARY 16
#define SCHEDULEGUI_GRIDY_MINSTEP  12
#define SCHEDULEGUI_GRIDY_MAXSTEP 32
#define SCHEDULEGUI_GRIDY_UNIT        1
#define SCHEDULEGUI_GRIDX_MINSTEP  32
#define SCHEDULEGUI_GRIDX_MAXSTEP 64
#define SCHEDULEGUI_GRIDX_UNIT        60
#define SCHEDULEGUI_FONTSIZE  6
//#define SCHEDULEGUI_XTEXTOFFSET (SCHEDULEGUI_FONTSIZE+2)
#define SCHEDULEGUI_XTEXTOFFSET 8
#define SCHEDULEGUI_SELECTED  1
#define SCHEDULEGUI_UNSELECTED  0


//DO NOT assume memory layout of DataPoint:
struct DataPoint {
 double x;
 double y;
 double duration;
 double hz;
	int state; //masks: SCHEDULEGUI_SELECTED
 DataPoint *NextDataPoint;
 DataPoint *PrevDataPoint;	//this is needed only to contain x-axis of mouse movement
} * ScheduleGUI_FirstDataPoint=NULL, * ScheduleGUI_CurrentDataPoint=NULL, * ScheduleGUI_FirstBackupDataPoint=NULL;

struct SelectionBox {
	int status; //live=1, dead=0
	int startX;
	int startY;
	int endX;
	int endY;
} ScheduleGUI_SelectionBox;
	
struct CopyPaste {
	int count;
 int OrigWidth;
 int OrigHeight;
 double OrigScheduleGUI_TotalScheduleDuration;
 double OrigScheduleGUI_MaxScheduleBeatfrequency;
	DataPoint * data;
} ScheduleGUI_CopyPaste;

 void ScheduleGUI_Cleanup();
 gboolean ScheduleGUI_delete_event(GtkWidget* window, GdkEvent* e, gpointer data);
 gboolean ScheduleGUI_button_release_event( GtkWidget  *widget, GdkEventButton *event);
 gboolean ScheduleGUI_button_press_event( GtkWidget *widget, GdkEventButton *event);
 gboolean ScheduleGUI_expose_event( GtkWidget *widget, GdkEventExpose *event);
 gboolean ScheduleGUI_configure_event( GtkWidget  *widget, GdkEventConfigure *event);
 gboolean ScheduleGUI_configure_event( GtkWidget  *widget, GdkEventConfigure *event);
 gboolean ScheduleGUI_motion_notify_event( GtkWidget *widget, GdkEventMotion *event);
 void ScheduleGUI_SetupDataPoints();
 void ScheduleGUI_DeselectDataPoints();
 void ScheduleGUI_SelectAllDataPoints();
 void ScheduleGUI_SelectIntervalAllDataPoints(int interval, gboolean deselectothers);
 void ScheduleGUI_InvertSelectionAllDataPoints();
 void ScheduleGUI_SelectDataPoints(int startX,int startY,int endX,int endY, gboolean deselect=FALSE);
 void ScheduleGUI_DrawGraph( GtkWidget *widget);
 void ScheduleGUI_CleanupDataPoints(DataPoint * firstDataPoint);
 DataPoint * ScheduleGUI_AddNewDataPointToEnd(double  duration, double hz);
 DataPoint * ScheduleGUI_InsertNewDataPointXY(GtkWidget *widget, double  x, double  y); //creates a point and puts it in the right llist order according to x
// DataPoint * ScheduleGUI_InsertNewDataPointDurHz(GtkWidget *widget, double  dur, double  hz); //creates a point and puts it in the right llist order according to duration
 void ScheduleGUI_GetScheduleLimits();//fills maxduration and maxbeatfrequency global variables
 void ScheduleGUI_ConvertDurHzToXY(GtkWidget *widget);//this recalibrates scale, finds totalscheduleduration, etc.
 void ScheduleGUI_ConvertXYToDurHZ(GtkWidget *widget,  DataPoint * currentDataPoint);
	void ScheduleGUI_ConvertXToDuration_AllPoints(GtkWidget *widget);
	void ScheduleGUI_ConvertYToHz_AllPoints(GtkWidget *widget); 
void		ScheduleGUI_MoveDataPoint(GtkWidget *widget, DataPoint * curDP, double newx, double newy);
void ScheduleGUI_MoveSelectedDataPoints(GtkWidget * widget, double moveX, double moveY);
gboolean ScheduleGUI_SwapLinks(DataPoint * curDP);//swaps datapoint with its left neighbor
gboolean ScheduleGUI_SwapLinks(DataPoint * curDP1, DataPoint * curDP2); //swaps any two existing links
void ScheduleGUI_DeleteSelectedPoints(GtkWidget * widget, gboolean DeleteTime);
void ScheduleGUI_DeleteDataPoint(DataPoint * curDP, gboolean DeleteTime=FALSE);
 void ScheduleGUI_BackupDataPoints();
void ScheduleGUI_RestoreDataPoints(GtkWidget * widget);
void ScheduleGUI_CopySelectedDataPoints(GtkWidget * widget);
void ScheduleGUI_PasteSelectedDataPoints(GtkWidget * widget, gboolean predeselect = TRUE);
void ScheduleGUI_DeleteDuplicateDataPoints(GtkWidget * widget, double threshold);
void ScheduleGUI_SelectNeighboringDataPoints(gboolean next, gboolean deselect) ;
gboolean ScheduleGUI_key_press_event( GtkWidget * widget, GdkEventKey * event );




double ScheduleGUI_TotalScheduleDuration = 1;
double ScheduleGUI_MaxScheduleBeatfrequency = 1;
gboolean ScheduleGUI_ProgressIndicatorFlag = TRUE; //to disallow progress indication if graph has been modified (out of sync with BB)


//main drawing area:
 GdkPixmap *ScheduleGUI_pixmap = NULL;
 GdkGC *ScheduleGUI_gc=NULL;
 PangoLayout * ScheduleGUI_layout=NULL;	


//NOTE: this function was def'd out because gettimeofday() isn't Windows 
//friendly, and I only use this function for optimizing/debugging
#ifdef SCHEDULEGUI_FREESTANDING
/////////////////////////////////////////////////////
//this function measures time using very first call to 
//it as 0, making it similar to the non-ansi uclock().
//USE: taskflag: 0 means start timer and return total amount 
//of time since first call, !0 means return time elapsed since 
//start timer
unsigned int Stopwatch(int taskflag)
{
 struct timeval tv;
 static unsigned int  init_sec = 0;
 static unsigned int lastcheck=0;

  gettimeofday(&tv, 0);

if (taskflag==0)
{
  if (init_sec == 0) init_sec = tv.tv_sec;
  lastcheck=(tv.tv_sec - init_sec) * 1000000 + tv.tv_usec;
  return lastcheck; // this gives user total length of time since this function was called)
  }
  return ((tv.tv_sec - init_sec) * 1000000 + tv.tv_usec)-lastcheck;
}
#endif


/////////////////////////////////////////////////////
 gboolean ScheduleGUI_realize( GtkWidget  *widget,
                                 GdkEventConfigure *event )
{
	return FALSE;
}


/////////////////////////////////////////////////////
// Initializing or resizing, so create a new backing pixmap of the appropriate size
 gboolean ScheduleGUI_configure_event( GtkWidget  *widget,
                                 GdkEventConfigure *event )
{
  if (ScheduleGUI_pixmap!=NULL)   g_object_unref (ScheduleGUI_pixmap);

  ScheduleGUI_pixmap = gdk_pixmap_new (widget->window,
			   widget->allocation.width,
			   widget->allocation.height,
			   -1);

  //APPARENTLY THIS HAS BECOME ESSENTIAL TO GNAURAL. 
  //ScheduleGUI_SetupDataPoints() must get called before program can run,
  //so this was where it happened (discovered when I commented-it out).
  if (ScheduleGUI_FirstDataPoint==NULL)  ScheduleGUI_SetupDataPoints();
  
		ScheduleGUI_ConvertDurHzToXY(widget);
		ScheduleGUI_DrawGraph(widget);
  return TRUE;
}


/////////////////////////////////////////////////////
// This is the repaint signal, so redraw the screen from the backing pixmap
 gboolean ScheduleGUI_expose_event( GtkWidget      *widget,
                              GdkEventExpose *event )
{
	/*
	The GtkDrawingArea widget is used for creating custom user interface elements. 
	It's essentially a blank widget; you can draw on widget->window. After creating a 
	drawing area, the application may want to connect to:

-      Mouse and button press signals to respond to input from the user. 
	(Use gtk_widget_add_events() to enable events you wish to receive.)

 -     The "realize" signal to take any necessary actions when the widget is instantiated 
	on a particular display. (Create GDK resources in response to this signal.)

 -     The "ScheduleGUI_configure_event" signal to take any necessary actions when the widget changes size.

  -    The "expose_event" signal to handle redrawing the contents of the widget.

GDK automatically clears the exposed area to the background color before sending the 
	expose event, and that drawing is implicitly clipped to the exposed area. 
	*/
  gdk_draw_drawable (widget->window,
		     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		     ScheduleGUI_pixmap,
		     event->area.x, event->area.y,
		     event->area.x, event->area.y,
		     event->area.width, event->area.height);
							
  return FALSE;
}




/////////////////////////////////////////////////////
//this is slow as hell -- for repetitious things, inlined this code
void draw_text(GtkWidget *widget, gint x, gint y,gchar *s_utf8)
{
 PangoLayout *layout;
 layout = pango_layout_new(gdk_pango_context_get());
 pango_layout_set_text(layout, s_utf8, -1);

 PangoFontDescription *	fontdesc = pango_font_description_copy(widget->style->font_desc);
	//I can set the font here, but seems safer to work with what I know user has:
 //pango_font_description_set_family (fontdesc, "Sans Bold Italic");
 pango_font_description_set_size (fontdesc,PANGO_SCALE*6);
	//pango_layout_set_alignment(layout,PANGO_ALIGN_RIGHT);
 pango_layout_set_font_description (layout, fontdesc);
 pango_font_description_free (fontdesc);

	gdk_draw_layout(ScheduleGUI_pixmap, widget->style->black_gc, x, y, layout);

 g_object_unref(layout);
}



/////////////////////////////////////////////////////
//If there is one function I should speed-up,
//this is it. Among other things, I could split
//it up in to separate functions, some of which
//could be left-out during high refresh periods
////
// Draw graph on the screen
/////////////////////////////////////////////////////
#define SCHEDULEGUI_TEXTYOFFSET 8
void ScheduleGUI_DrawGraph( GtkWidget *widget)
{
int x,y;
static char graphtext[32];//this is used for all text rendering
	

//	Stopwatch(0);//REMOVE FOR FINISHED VERSION!!!


	// Create a GC to draw with if it isn't already created:
 if (ScheduleGUI_gc==NULL) 	//you delete something like this with g_object_unref(ScheduleGUI_gc);
{
  ScheduleGUI_gc = gdk_gc_new(widget->window);
}

//Create Pango layout and set it up to write text with if it isn't already created:
 if (ScheduleGUI_layout==NULL) 	//you delete something like this with g_object_unref(gc);
{
//prepare for Text writing:
 ScheduleGUI_layout = pango_layout_new(gdk_pango_context_get());

 PangoFontDescription *	fontdesc = pango_font_description_copy(widget->style->font_desc);
	//I can set the font here, but seems safer to work with what I know user has:
 //pango_font_description_set_family (fontdesc, "Sans Bold Italic");
 pango_font_description_set_size (fontdesc,PANGO_SCALE*SCHEDULEGUI_FONTSIZE);
	//pango_layout_set_alignment(layout,PANGO_ALIGN_RIGHT);
 pango_layout_set_font_description (ScheduleGUI_layout, fontdesc);
 pango_font_description_free (fontdesc);
	}
	
//First paint the background:
//NOTE: colors are between 0 and ~65000 in GDK
  GdkColor color;
  color.red   = color.green = color.blue  = 65000;
  gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
		gdk_draw_rectangle (ScheduleGUI_pixmap,
//		      widget->style->white_gc,
        ScheduleGUI_gc,
		      TRUE,
		      0, 0,
		      widget->allocation.width,
		      widget->allocation.height);
	
  color.red   = color.green = color.blue  = 55555;
  gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  

	
//NOW draw grid:
bool SimpleGrid=FALSE;
int xtextoffset=widget->allocation.height-SCHEDULEGUI_XTEXTOFFSET;
//Try to draw the friendly (slow) grid:
if (SimpleGrid==FALSE || ScheduleGUI_MaxScheduleBeatfrequency>0 || ScheduleGUI_TotalScheduleDuration>0)
{
//Instead draw "friendly" (and extremely computationally intensive) horizontal grid lines:
 double index=widget->allocation.height;
 double textindex=0;
	double textstep=SCHEDULEGUI_GRIDY_UNIT;//this is the basis of one "unit" of vertical climb
 double step=index/ScheduleGUI_MaxScheduleBeatfrequency;//BUG: THIS WILL CRASH IF ScheduleGUI_MaxScheduleBeatfrequency IS ZERO
//these next two uglies could be speeded up in a number of ways, no?
	if (step<SCHEDULEGUI_GRIDY_MINSTEP) {
		do { step*=2; textstep*=2; } while (step <SCHEDULEGUI_GRIDY_MINSTEP );
	}
 else if (step>SCHEDULEGUI_GRIDY_MAXSTEP) {
		do { step*=.5; textstep*=.5; } while (step > SCHEDULEGUI_GRIDY_MAXSTEP);
	}
	int ax;
while ((ax=(int)((index-=step)+.5))>-1) 
	{
	gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, 0,ax, widget->allocation.width, ax);
		textindex+=textstep;
  sprintf(graphtext,"%gHz",textindex);
 pango_layout_set_text(ScheduleGUI_layout, graphtext, -1);
	gdk_draw_layout(ScheduleGUI_pixmap, widget->style->black_gc, 0, ax, ScheduleGUI_layout);
}

//Instead draw "friendly" (and extremely computationally intensive) vertical grid lines:
 index=0;
 textindex=0;
	textstep=SCHEDULEGUI_GRIDX_UNIT;//this is the basis of one "unit" of horizontal movement
 step=(SCHEDULEGUI_GRIDX_UNIT*widget->allocation.width)/ScheduleGUI_TotalScheduleDuration;//BUG: THIS WILL CRASH IF ScheduleGUI_TotalScheduleDuration IS ZERO
int minutes, seconds;
//these next two uglies could be speeded up in a number of ways, no?
	if (step<SCHEDULEGUI_GRIDX_MINSTEP) {
		do { step*=2; textstep*=2; } while (step < SCHEDULEGUI_GRIDX_MINSTEP );
	}
 else if (step> SCHEDULEGUI_GRIDX_MAXSTEP) {
		do { step*=.5; textstep*=.5; } while (step > SCHEDULEGUI_GRIDX_MAXSTEP);
	}
while ((ax=(int)((index+=step)+.5)) <widget->allocation.width) 
{
	gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, ax,0,ax, widget->allocation.height);
	textindex+=textstep;
	//first handle if there are no minutes at all:
	if ((minutes=((int)(textindex+.5))/SCHEDULEGUI_GRIDX_UNIT)<1)  sprintf(graphtext,"%ds",((int)(textindex+.5)));
	else if ((seconds= ((int)(textindex+.5))%SCHEDULEGUI_GRIDX_UNIT)==0) sprintf(graphtext,"%dm",minutes);
	else sprintf(graphtext,"%dm%ds",minutes, seconds);
 pango_layout_set_text(ScheduleGUI_layout, graphtext, -1);
	gdk_draw_layout(ScheduleGUI_pixmap, widget->style->black_gc, ax,xtextoffset, ScheduleGUI_layout);
}

}

else
{
//First draw the horizontal marker lines:
#define SCHEDULEGUI_YSTEP 24
int index=widget->allocation.height;
double textstep=(SCHEDULEGUI_YSTEP*ScheduleGUI_MaxScheduleBeatfrequency/widget->allocation.height);
double textindex=0;
while ((index-=SCHEDULEGUI_YSTEP)>-1) 
	{
	gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, 0,index, widget->allocation.width, index);
		textindex+=textstep;
  sprintf(graphtext,"%6.5gHz",textindex);
 pango_layout_set_text(ScheduleGUI_layout, graphtext, -1);
	gdk_draw_layout(ScheduleGUI_pixmap, widget->style->black_gc, 0, index, ScheduleGUI_layout);
}

//Next draw vertical lines:
#define SCHEDULEGUI_XSTEP  64
index=0;
textstep=(SCHEDULEGUI_XSTEP*ScheduleGUI_TotalScheduleDuration/widget->allocation.width);
textindex=0;
while ((index+=SCHEDULEGUI_XSTEP)<widget->allocation.width) 
{
	gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, index,0,index, widget->allocation.height);
	textindex+=textstep;
 sprintf(graphtext,"%dm %ds",((int)(textindex+.5))/60, ((int)(textindex+.5))%60);
 pango_layout_set_text(ScheduleGUI_layout, graphtext, -1);
	gdk_draw_layout(ScheduleGUI_pixmap, widget->style->black_gc, index,xtextoffset, ScheduleGUI_layout);
}
}
	//==
//FINALLY, done with the grid, now present the data:
 //==
#define SG_ROUNDER +.5
//#define SG_ROUNDER 
int x_next=0,y_next=0;
//First, connect the dots:
DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
//first do all the easy points (where I have a valid current and next):
color.red = 32768; color.green = 32768; color.blue = 65535;
gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
while (currentDataPoint->NextDataPoint != NULL)
{
	x=(int)(currentDataPoint->x SG_ROUNDER);
	y=(int)(currentDataPoint->y SG_ROUNDER);
	x_next=(int)(currentDataPoint->NextDataPoint->x SG_ROUNDER);
	y_next=(int)(currentDataPoint->NextDataPoint->y SG_ROUNDER);
	
 gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, x, y, x_next, y_next);

	//advance to next point:
 currentDataPoint=currentDataPoint->NextDataPoint;
};//end of main connect the dots loop
//======

//Now draw last line from final graphpoint tranlated from the first datapoint:
//x=(int)(ScheduleGUI_FirstDataPoint->x+widget->allocation.width+-1.5);
x=(int)(ScheduleGUI_FirstDataPoint->x+widget->allocation.width);
y=(int)(ScheduleGUI_FirstDataPoint->y SG_ROUNDER);
gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, x_next, y_next,x,y);	

//Now start making the marker rectangles: 
//First do square for last graphpoint, since it is a special case and 
//the preparatory math already got done above:
	//decide last graphpoint square color (will be same as first datapoint's):
 if (ScheduleGUI_FirstDataPoint->state==SCHEDULEGUI_UNSELECTED) color.red   = color.green = color.blue  = 0;
	else { color.red = 65535; color.green = 32768; color.blue = 32768; }
	gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
gdk_draw_rectangle (ScheduleGUI_pixmap, ScheduleGUI_gc, FALSE,
  x-(SCHEDULEGUI_SQUARESIZE>>1), 
  y-(SCHEDULEGUI_SQUARESIZE>>1), 
  SCHEDULEGUI_SQUARESIZE, SCHEDULEGUI_SQUARESIZE);	

// Now do the rest of the marker rectangles:
//split all the loops on 20060414 to solve problem of pasted selected DPs hiding behind
//unselected 	parent DPs:
	//First do unselected:
color.red = color.green = color.blue = 0;
gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
currentDataPoint=ScheduleGUI_FirstDataPoint;
do  {
	if (currentDataPoint->state == SCHEDULEGUI_UNSELECTED) 
   gdk_draw_rectangle (ScheduleGUI_pixmap, ScheduleGUI_gc, FALSE,
     (int)(currentDataPoint->x SG_ROUNDER)-(SCHEDULEGUI_SQUARESIZE>>1), 
     (int)(currentDataPoint->y SG_ROUNDER)-(SCHEDULEGUI_SQUARESIZE>>1), 
     SCHEDULEGUI_SQUARESIZE, SCHEDULEGUI_SQUARESIZE);	
	//advance to next point:
 currentDataPoint=currentDataPoint->NextDataPoint;
} while (currentDataPoint != NULL);

//Now do selected and draw highlighted line for BB's current entry:
color.red = 65535; color.green = 32768; color.blue = 32768;
	gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
currentDataPoint=ScheduleGUI_FirstDataPoint;
#ifndef SCHEDULEGUI_FREESTANDING
int count=0;
#endif
do  {
 //first check to see if progress line should be drawn:
 //START of highlighted line indicating schedule progress
	#ifndef SCHEDULEGUI_FREESTANDING
 if (ScheduleGUI_ProgressIndicatorFlag==TRUE && count++ == bb->ScheduleCount) {
 color.red = 65535>>2; color.green = 65535; color.blue = 65535>>2;
	gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
 if (currentDataPoint->NextDataPoint != NULL) {
    	x=(int)(currentDataPoint->x SG_ROUNDER);
	    y=(int)(currentDataPoint->y SG_ROUNDER);
	    x_next=(int)(currentDataPoint->NextDataPoint->x SG_ROUNDER);
	    y_next=(int)(currentDataPoint->NextDataPoint->y SG_ROUNDER);
  } else {
     x=(int)(ScheduleGUI_FirstDataPoint->x+widget->allocation.width);
     y=(int)(ScheduleGUI_FirstDataPoint->y SG_ROUNDER);
     x_next=(int)(currentDataPoint->x SG_ROUNDER);
     y_next=(int)(currentDataPoint->y SG_ROUNDER);
  }
 gdk_draw_line (ScheduleGUI_pixmap, ScheduleGUI_gc, x, y, x_next, y_next);
 color.red = 65535; color.green = 32768; color.blue = 32768;
	gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
  }//END of highlighted line indicating schedule progress
		#endif
 //now draw squares:
	if (currentDataPoint->state == SCHEDULEGUI_SELECTED) 
   gdk_draw_rectangle (ScheduleGUI_pixmap, ScheduleGUI_gc, FALSE,
     (int)(currentDataPoint->x SG_ROUNDER)-(SCHEDULEGUI_SQUARESIZE>>1), 
     (int)(currentDataPoint->y SG_ROUNDER)-(SCHEDULEGUI_SQUARESIZE>>1), 
     SCHEDULEGUI_SQUARESIZE, SCHEDULEGUI_SQUARESIZE);	
	//advance to next point:
 currentDataPoint=currentDataPoint->NextDataPoint;
} while (currentDataPoint != NULL);

//fprintf(stderr, "%g\n",1000000.0/Stopwatch(1));//REMOVE - FOR DEBUGGING ONLY

//all done, ask manager to put it out there:
gtk_widget_queue_draw_area (widget,0,0,widget->allocation.width, widget->allocation.height);
}













/////////////////////////////////////////////////////
//Tests if a mouse-click landed on a square
DataPoint * ScheduleGUI_TestXYForDataPoint(gdouble x,gdouble y, int graphwidth)
{
	//make all points undraggable:
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
	//first check to see if last square is clicked (since it is REALLY the first one):
	if ( (x-SCHEDULEGUI_SQUARESIZE) < (currentDataPoint->x+graphwidth) && 
		    (x+SCHEDULEGUI_SQUARESIZE) > (currentDataPoint->x+graphwidth) &&
      (y-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->y && 
	     (y+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->y)   	return currentDataPoint;

//wasn't last square, so now trudge through all of them:	
do{
	if ((x-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->x && 
		    (x+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->x &&
      (y-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->y && 
	     (y+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->y)  	return currentDataPoint;
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
//########
//If it got here, user missed all existant points -- therefore it is
//probably a NEW DATA POINT (unless it had the same x as an existant point).
//########
return NULL;
}








/////////////////////////////////////////////////////
//now I figure out which data point got clicked, if any:
gboolean ScheduleGUI_button_press_event( GtkWidget * widget, GdkEventButton * event )
{
	if (ScheduleGUI_pixmap==NULL) return TRUE;
		
//############################
//Check for button 1: [Select point or create point]
//############################
if (event->button == 1)
{
 //Button 1 means either select a point (making it and other selected draggable) or create new point:
	//Test if mouse is clicked an existing point, so make point selected:
	ScheduleGUI_CurrentDataPoint=ScheduleGUI_TestXYForDataPoint(event->x,event->y,widget->allocation.width);
	
	//If !NULL, it found a suitable point:
	if (ScheduleGUI_CurrentDataPoint!=NULL) 
	{
		//OVERVIEW: If point is already selected, see if Ctrl key is pressed to unselect it. If Ctrl isn't pressed,
		//just leave the selected point alone so that motion_notify_event can be allowed to move a clump of 
		//points. But If point is not selected, deselect all points then select this one.
		if (ScheduleGUI_CurrentDataPoint->state==SCHEDULEGUI_SELECTED) {
			if ((event->state & GDK_CONTROL_MASK) != 0) ScheduleGUI_CurrentDataPoint->state=SCHEDULEGUI_UNSELECTED;
			return TRUE;
		}
 	if ((event->state & GDK_SHIFT_MASK)==0) ScheduleGUI_DeselectDataPoints(); 				
  ScheduleGUI_CurrentDataPoint->state=SCHEDULEGUI_SELECTED;
		ScheduleGUI_DrawGraph(widget);
		return TRUE;
	}

	//must check for SHIFT and CTRL keys again in this context,
 //just in case use is dragging a selection box:
	if ((event->state & GDK_SHIFT_MASK)==0 && 
		    (event->state & GDK_CONTROL_MASK) == 0) ScheduleGUI_DeselectDataPoints(); 				

		//Mouse clicked a non-existant point, so first see if it was a double-click (create a new point):
	if (event->type==GDK_2BUTTON_PRESS) 
	{
	ScheduleGUI_CurrentDataPoint=ScheduleGUI_InsertNewDataPointXY(widget, event->x,event->y+.5);
	if (ScheduleGUI_CurrentDataPoint != NULL) 
	{
  ScheduleGUI_ProgressIndicatorFlag=FALSE;
	//Point exists, so convert it's XY to schedule values:
//	ScheduleGUI_CurrentDataPoint->state=SCHEDULEGUI_SELECTED;
// ScheduleGUI_ConvertXYToDurHZ(widget, ScheduleGUI_CurrentDataPoint);
 ScheduleGUI_DrawGraph(widget);
	return TRUE;
	}
 }
//if it got here, use XY as start of a selection bounding box:
ScheduleGUI_SelectionBox.status=1;
ScheduleGUI_SelectionBox.startX=(int)(event->x);
ScheduleGUI_SelectionBox.startY=(int)(event->y);
ScheduleGUI_DrawGraph(widget);
 return TRUE;
}
//############################
//Check for button 2 [delete point]:
//############################
else if (event->button ==2) 
{	
		DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	if ((event->x-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->x && 
		   (event->x+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->x &&
     (event->y-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->y && 
	    (event->y+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->y)
{
 ScheduleGUI_ProgressIndicatorFlag=FALSE;
	if ((event->state & GDK_SHIFT_MASK)==0) ScheduleGUI_DeleteDataPoint(currentDataPoint);
 else ScheduleGUI_DeleteDataPoint(currentDataPoint, TRUE);
	ScheduleGUI_ConvertDurHzToXY(widget);
	ScheduleGUI_DrawGraph(widget);	
	return TRUE;
}
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);

	//If I got here, it must be the last square (which is REALLY the first DataPoint):
 currentDataPoint=ScheduleGUI_FirstDataPoint;
	if ((event->x+SCHEDULEGUI_SQUARESIZE) > (currentDataPoint->x+widget->allocation.width) &&
      (event->y-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->y && 
	     (event->y+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->y &&
	      ScheduleGUI_FirstDataPoint->NextDataPoint!=NULL)//to be sure we don't end up trying to delete the ONLY point
{
//	if ((event->state & GDK_SHIFT_MASK)==0) ScheduleGUI_DeleteDataPoint(currentDataPoint);
// else ScheduleGUI_DeleteDataPoint(currentDataPoint, TRUE);
 ScheduleGUI_DeleteDataPoint(currentDataPoint);
	ScheduleGUI_ConvertDurHzToXY(widget);
	ScheduleGUI_DrawGraph(widget);	
	return TRUE;
}	
//shouldn't actually ever get here...
return TRUE;
}
//############################
//Not button 1 or 2, so check for button 3 [bring-up a properties box for the DataPoint]:
//############################
else if (event->button ==3) 
{
		DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
//run through all the DataPoints to find the one clicked (if any):
	//NOTE: we ignore trying to click on last visible datapoint because it isn't a real one (rather, 'tis the first one)
	do {
	if ((event->x-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->x && (event->x+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->x &&
      (event->y-SCHEDULEGUI_SQUARESIZE) < currentDataPoint->y && (event->y+SCHEDULEGUI_SQUARESIZE) > currentDataPoint->y)
{
//	printf("Properties: Duration %g, Hz, %g\n", currentDataPoint->duration, currentDataPoint->hz);
//	printf("Height: %d Width: %d\n", widget->allocation.height, widget->allocation.width);
	  GtkWidget *dialog = gtk_dialog_new_with_buttons ("Event Properties",
          NULL,
						   //(GtkWindow*) widget,
						   (GtkDialogFlags)
						   (GTK_DIALOG_MODAL |
						    GTK_DIALOG_DESTROY_WITH_PARENT),
						   GTK_STOCK_OK,
						   GTK_RESPONSE_ACCEPT,
						   GTK_STOCK_CANCEL,
						   GTK_RESPONSE_REJECT,
						   NULL);

//add some stuff:
  GtkWidget *label_dur = gtk_label_new ("Event Duration:");
  GtkWidget *entry_Input_dur = gtk_entry_new ();
  GtkWidget *label_hz = gtk_label_new ("Starting Frequency:");
  GtkWidget *entry_Input_hz = gtk_entry_new ();

 char tmpstring[256];
  sprintf (tmpstring, "%g",currentDataPoint->duration);
  gtk_entry_set_text (GTK_ENTRY (entry_Input_dur), tmpstring);
  sprintf (tmpstring, "%g",currentDataPoint->hz);
  gtk_entry_set_text (GTK_ENTRY (entry_Input_hz), tmpstring);


  // Add the label, and show everything we've added to the dialog. 
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_dur);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_dur);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_hz);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), entry_Input_hz);

  double times =currentDataPoint->x*(ScheduleGUI_TotalScheduleDuration/widget->allocation.width);
  sprintf (tmpstring, "Start time: %dmin %dsec",((int)(times+.5))/60,((int)(times+.5))%60);
  GtkWidget *label_start = gtk_label_new (tmpstring);
  sprintf (tmpstring, "End time: %dmin %dsec",((int)(times+currentDataPoint->duration+.5))/60,((int)(times+currentDataPoint->duration+.5))%60);
  GtkWidget *label_end = gtk_label_new (tmpstring);
  sprintf (tmpstring, "\nX,Y:  %d, %d",(int)(currentDataPoint->x+.5),
     (int)((widget->allocation.height-currentDataPoint->y)+.5));
  GtkWidget *label_xy = gtk_label_new (tmpstring);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_start);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_end);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label_xy);

  gtk_widget_show_all (dialog);

  //block until I get a response:
  gint result = gtk_dialog_run (GTK_DIALOG (dialog));
  switch (result)
    {
    case GTK_RESPONSE_ACCEPT:
      {
	double tmpd;
	tmpd = atof (gtk_entry_get_text (GTK_ENTRY (entry_Input_dur)));
	if (tmpd > 0 || tmpd < 60000)		currentDataPoint->duration=tmpd;
	tmpd = atof (gtk_entry_get_text (GTK_ENTRY (entry_Input_hz)));
	if (tmpd > 0 || tmpd < 10000)		currentDataPoint->hz=tmpd;
 ScheduleGUI_ProgressIndicatorFlag=FALSE;
 ScheduleGUI_ConvertDurHzToXY(widget);
	ScheduleGUI_DrawGraph(widget);
	break;
	
	default:
    break;
    }
			}
  gtk_widget_destroy (dialog);
	return TRUE;
}
currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
return TRUE;
}

return TRUE;
}



/////////////////////////////////////////////////////
gboolean ScheduleGUI_button_release_event( GtkWidget * widget, GdkEventButton *event)
{
  if (event->button != 1 ||  ScheduleGUI_pixmap == NULL) return TRUE;

  //this was added 20060414. It is sort of arbitrary that it is here, so keep aware of it:
  	ScheduleGUI_DeleteDuplicateDataPoints(widget, 0.0);

		//=============
 //first see if we're doing bounding box selecting:
	 if (ScheduleGUI_SelectionBox.status!=0)
		{
			ScheduleGUI_SelectionBox.status=0;//tell renderer not to draw box
			ScheduleGUI_DrawGraph(widget);//erases last rendering of box
			return TRUE; //don't do any dp processing
		}		
		
// if (ScheduleGUI_CurrentDataPoint!=NULL && event->button == 1 && ScheduleGUI_pixmap != NULL) 
if (ScheduleGUI_CurrentDataPoint!=NULL) 
{
//First the easy one: convert all Y values to Hz:
ScheduleGUI_ConvertYToHz_AllPoints(widget);
	
 //Now convert all X values to Duration:
 //Can't move first data point along X axis://NOTE: I think it is no longer necessary to do this here (other functions are safe on this point). But safest to just leave it...
	if (ScheduleGUI_CurrentDataPoint!=ScheduleGUI_FirstDataPoint)
	{
 //Now the hard one: convert ScheduleGUI_CurrentDataPoint's x to duration value. Hard because 
 //I also need to change PrevDataPoint's duration:
 if (ScheduleGUI_TotalScheduleDuration==0 || widget->allocation.width==0) return TRUE;
	ScheduleGUI_ConvertXToDuration_AllPoints(widget);
	}
	
	ScheduleGUI_CurrentDataPoint=NULL;
 ScheduleGUI_ConvertDurHzToXY(widget);// NOTE: I need to call this both to set limits and to bring XY's outside of graph back in
 ScheduleGUI_DrawGraph(widget);
	}
  return TRUE;
}





//////////////////////////////////////////
//this inserts srcDP to the left of destDP. 
gboolean ScheduleGUI_InsertLink(DataPoint * srcDP, DataPoint * destDP)
{//swap two links:
	if (srcDP==NULL || destDP==NULL) return FALSE;

					DataPoint * destprevDP=destDP->PrevDataPoint;
					DataPoint * srcprevDP=srcDP->PrevDataPoint;
					DataPoint * srcnextDP=srcDP->NextDataPoint;
//					DataPoint * destnextDP=destDP->NextDataPoint;
	
	//curDP is FirstDataPoint; there is nothing logical to swap-left with, so:
	if (destprevDP==NULL)		return FALSE;

	destDP->PrevDataPoint=srcDP;
	destprevDP->NextDataPoint=srcDP;
 srcDP->NextDataPoint=destDP;
	srcDP->PrevDataPoint=destprevDP;
	srcprevDP->NextDataPoint=srcnextDP;
//deal with special case where	srcDP is LastDataPoint, in which case 2nd-to-last becomes last:
 if (srcnextDP!=NULL) srcnextDP->PrevDataPoint=srcprevDP;
		
	return TRUE;
}



//////////////////////////////////////////
//this swaps link you pass to it with neighbor to it's "left"
//(so if you want to want to swap something with it's
//neighbor to the right, pass the neighbor instead)
//NOTE: this works well for the visual needs of the graph, but 
//is unsuitable for really determining final values for data points
//because it can't compensate dur/hz values losing their
//original reference to neighbors. Use "InsertDatapoint"
//after mouse-click is released to get real final values from
//the x,y 
gboolean ScheduleGUI_SwapLinks(DataPoint * curDP)
{//swap two links:
	if (curDP==NULL) return FALSE;
					DataPoint * prevDP=curDP->PrevDataPoint;
					DataPoint * nextDP=curDP->NextDataPoint;
	if (prevDP==NULL)	{ //curDP is FirstDataPoint; there is nothing logical to swap-left with, so:
		return FALSE;
	}	
					DataPoint * prevprevDP=curDP->PrevDataPoint->PrevDataPoint;
	if (prevprevDP==NULL) return FALSE;
					curDP->NextDataPoint=prevDP;
					curDP->PrevDataPoint=prevprevDP;
					prevDP->NextDataPoint=nextDP;
					prevDP->PrevDataPoint=curDP;
//deal with special case where	curDP is LastDataPoint, in which case 2nd-to-last becomes last:
					if (nextDP!=NULL) nextDP->PrevDataPoint=prevDP;
					prevprevDP->NextDataPoint=curDP;
	
	return TRUE;
}
/*
//////////////////////////////////////////
//THIS IS ONLY HERE AS AN EXAMPLE OF THE INVERSE
//OF THE ABOVE SWAP FUNCTION:
//this swaps link you pass to it with neighbor to it's "right"
//(so if you want to want to swap something with it's
//neighbor to the left, you could pass the neighbor instead)
gboolean ScheduleGUI_SwapLinks_right(DataPoint * curDP)
{//swap two links:
	if (curDP==NULL) return FALSE;
					DataPoint * prevDP=curDP->PrevDataPoint;
					DataPoint * nextDP=curDP->NextDataPoint;
	if (nextDP==NULL || prevDP==NULL) return FALSE;
					DataPoint * nextnextDP=curDP->NextDataPoint->NextDataPoint;
		if (nextnextDP==NULL) return FALSE;
					curDP->NextDataPoint=nextnextDP;
					curDP->PrevDataPoint=nextDP;
					nextDP->PrevDataPoint=prevDP;
					nextDP->NextDataPoint=curDP;
					prevDP->NextDataPoint=nextDP;
					nextnextDP->PrevDataPoint=curDP;
		return TRUE;
}
*/




//TODO: Figure out why I must convert the values to ints for "is_hint"
/////////////////////////////////////////////////////
 gboolean ScheduleGUI_motion_notify_event( GtkWidget *widget, GdkEventMotion *event)
{
  int x, y;
  GdkModifierType state;

  if (event->is_hint)  gdk_window_get_pointer (event->window, &x, &y, &state);
  else
    {
      x = (int)(event->x+.5);
      y = (int)(event->y+.5);
      state = (GdkModifierType) event->state;
    }
    
  if ( (state&GDK_BUTTON1_MASK)==0 ||  ScheduleGUI_pixmap == NULL) return TRUE;
			
		//=============
		//see whether to do bounding box or drag point(s):
		//first, is is a bounding box?:
	 if (ScheduleGUI_SelectionBox.status!=0)
		{
			//first draw graph (to erase last bounding box draw):
			ScheduleGUI_DrawGraph(widget);
			//now draw a box over it:
  GdkColor color;
  color.red = color.green = color.blue = 0;
  gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  
		gdk_gc_set_line_attributes (ScheduleGUI_gc, 1, GDK_LINE_ON_OFF_DASH, (GdkCapStyle) 0, (GdkJoinStyle) 0);
		int startX=ScheduleGUI_SelectionBox.startX;
		int startY=ScheduleGUI_SelectionBox.startY;
		int endX=x-startX;
		int endY=y-startY;
		if (endX<0) { startX=x; endX=ScheduleGUI_SelectionBox.startX-x; }
		if (endY<0) { startY=y; endY=ScheduleGUI_SelectionBox.startY-y; }
  gdk_draw_rectangle (ScheduleGUI_pixmap, ScheduleGUI_gc, FALSE,	startX, startY, endX,endY);	
  if ((event->state & GDK_CONTROL_MASK) == 0)  ScheduleGUI_SelectDataPoints(startX,startY,startX+endX,startY+endY);
		else 	ScheduleGUI_SelectDataPoints(startX,startY,startX+endX,startY+endY, TRUE);
		//all done, put it out there:
		//NOTE: I don't need to do this, for some reason:
//		gtk_widget_queue_draw_area (widget,startX,startY,endX, endY);
//		gtk_widget_queue_draw_area (widget,0,0,widget->allocation.width, widget->allocation.height);
		gdk_gc_set_line_attributes (ScheduleGUI_gc, 1, GDK_LINE_SOLID, (GdkCapStyle) 0, (GdkJoinStyle) 0);
		}
		//=============
		//Not bounding selection box, so deal with possibility that user was trying to move some points:
		//the user must always have one live ScheduleGUI_CurrentDataPoint to move any points, and it must
		//be selected, because it's movement will be the reference for all other other selected points to move
		else	 if (ScheduleGUI_CurrentDataPoint!=NULL && ScheduleGUI_CurrentDataPoint->state==SCHEDULEGUI_SELECTED)
		{
   ScheduleGUI_ProgressIndicatorFlag=FALSE;
			//First determine how far X and Y have moved:			
			double moveX=x-ScheduleGUI_CurrentDataPoint->x;
			double moveY=y-ScheduleGUI_CurrentDataPoint->y;

//Now go through all selected points and move them accordingly:
DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;	
do {
	if (currentDataPoint->state == SCHEDULEGUI_SELECTED)
//	if (FALSE == ScheduleGUI_MoveDataPoint(widget, currentDataPoint, currentDataPoint->x+moveX,currentDataPoint->y+moveY)) break;
	ScheduleGUI_MoveDataPoint(widget, currentDataPoint, currentDataPoint->x+moveX,currentDataPoint->y+moveY);
	//this must be in every linked list loop:
	currentDataPoint=currentDataPoint->NextDataPoint;
} while (currentDataPoint != NULL);
	


			
/*
			//THIS PART WORKS. It was just commented out so that I could implement freedom to move points past neighbors.
			//##The following mess LIMITS x SO THAT USER CAN'T MOVE PAST ADJACENT DataPoints##
			//First visible datapoint should only be allowed to move up and down, not across time. Last visible one (which is also the real 
			//first DataPoint) would ideally be able to move, though, to expland schedule time... but not for now:
			if (ScheduleGUI_CurrentDataPoint!=ScheduleGUI_FirstDataPoint) 
   {
   //Must treat the last real DataPoint differently, so first do all others:
				if (ScheduleGUI_CurrentDataPoint->NextDataPoint!=NULL)  {
    if (x > ScheduleGUI_CurrentDataPoint->PrevDataPoint->x  && x < ScheduleGUI_CurrentDataPoint->NextDataPoint->x)			ScheduleGUI_CurrentDataPoint->x=x;
    }
   else  if (x > ScheduleGUI_CurrentDataPoint->PrevDataPoint->x  && x < widget->allocation.width)			ScheduleGUI_CurrentDataPoint->x=x;
  }
*/

//Now do X:
//##NOTE: The following mess (moved to ScheduleGUI_MoveDataPointX) ALLOWS x USER TO MOVE PAST 
//ADJACENT DataPoints##
//Must treat first and last DataPoints differently, since either their prev or next is NULL 
//(making swaping a segfault certainScheduleGUI_MoveDataPointXty). First visible datapoint/graphpoint should 
//only be allowed to move up and down, not across time. Last visible graphpoint 
//(which is actually also the first DataPoint) would ideally be able to move, though,
//to expland schedule time... but not for now:
// if (FALSE==ScheduleGUI_MoveDataPointX(widget, ScheduleGUI_CurrentDataPoint, x)) return TRUE;
 //ScheduleGUI_MoveDataPoint(widget, ScheduleGUI_CurrentDataPoint, x,y);

 //Finally, put the graphical result out there (save translating in to dur/hz for mouse release; way too slow):
 ScheduleGUI_DrawGraph(widget);
	}
  return TRUE;
}


/*
/////////////////////////////////////////////////////
//THIS ONE ONLY WORKS DISREGARDING SELECTED/UNSELECTED STATUS
//give the existing datapoint and the desired new x, this will
//move it to the proper place on graph AND in the linked-list.
//returns FALSE if it something illegal is being asked-for:
gboolean		ScheduleGUI_MoveDataPoint_simple(GtkWidget *widget, DataPoint * curDP, double newx, double newy)
{
	//first do y:
		if (newy > widget->allocation.height)  newy = widget->allocation.height;
		curDP->y=newy;	

//now do x:		
if (curDP!=ScheduleGUI_FirstDataPoint) 
	{
		//first limit-check x axis:
		if (newx >= 0 && newx < widget->allocation.width) curDP->x=newx;
 	else return FALSE;
		
		//now see if point moved past a neighbor:
		do {
   if (newx < curDP->PrevDataPoint->x) 	ScheduleGUI_SwapLinks(curDP);
  //Must treat the last real DataPoint differently, for two reasons: 
   //1) it can't be pulled further than right-edge of graph
		 //2) it's NextDataPoint equals NULL, so can't swap-right
			else 	if (curDP->NextDataPoint==NULL) break;//last DP can't swap with anything to its right
		 if (newx > curDP->NextDataPoint->x)			ScheduleGUI_SwapLinks(curDP->NextDataPoint);
		} while (curDP->NextDataPoint!=NULL &&
                			(newx < curDP->PrevDataPoint->x    || 
			                 newx > curDP->NextDataPoint->x));
	 }	else return FALSE;
		return TRUE;
}
*/

/////////////////////////////////////////////////////
//THIS ONE TAKES IN TO CONSIDERATION SELECTED/UNSELECTED STATUS
//give the existing datapoint and the desired new x, this will
//move it to the proper place on graph AND in the linked-list.
//NOTE: this function quadrupled in complexity once it had to
//differentiate selected and unselected points and allow user to 
//expand length of schedule by dragging past right edge
void		ScheduleGUI_MoveDataPoint(GtkWidget *widget, DataPoint * curDP, double newx, double newy)
{
	//first do y: 
	if (newy > widget->allocation.height) newy = widget->allocation.height;//this means all Hz below 0 become 0
		
	curDP->y=newy;	

//now do x:
	//first, make sure it isn't the first data point, since that can't move sideways:
 if (curDP==ScheduleGUI_FirstDataPoint) return;

//first limit-check x axis:
//	if (newx < 0 || newx >= widget->allocation.width) return FALSE;
if (newx < 0) newx=0;
//else if (newx >= widget->allocation.width) newx=widget->allocation.width; //TODO: eventually, I should allow user to pull past final point

//now get difference between new point and old, then assign new value:
 double diff=curDP->x - newx;
	curDP->x=newx;
//	fprintf(stderr,"diff: %g\n",diff);
	
 DataPoint * tempDP;
	gboolean swapflag;
		//Positive diff means test if point moved past an unselected neighbor to the left:
  if (diff>0) 	do {
			//First try to find an unselected neighbor:
			tempDP=curDP;
			swapflag=FALSE;
			do {
				tempDP=tempDP->PrevDataPoint;
				if (tempDP==ScheduleGUI_FirstDataPoint) return;//no unselected neighbor to switch with
			}while (tempDP->state==SCHEDULEGUI_SELECTED);
   if (newx < tempDP->x) {				ScheduleGUI_InsertLink(curDP, tempDP); swapflag=TRUE; 			}
//   if (newx < tempDP->x) {				ScheduleGUI_SwapLinks(curDP); swapflag=TRUE; 			}
		} while (swapflag==TRUE);

		//Negative diff means test if point moved past an unselected neighbor to the right:
		else  if (diff<0 && curDP->NextDataPoint!=NULL)  do  { //remember, last DP can't swap with anything to its right) 
			//First try to find an unselected neighbor:
			tempDP=curDP;
			swapflag=FALSE;
			do {
				tempDP=tempDP->NextDataPoint;
				if (tempDP==NULL) return;//no unselected neighbor to switch with
			}while (tempDP->state==SCHEDULEGUI_SELECTED);
   if (newx > tempDP->x) { ScheduleGUI_InsertLink(tempDP,curDP); swapflag=TRUE; }
//   if (newx > tempDP->x) { ScheduleGUI_SwapLinks(curDP->NextDataPoint); swapflag=TRUE; }
		} while (swapflag==TRUE);
		else {
   //=====
   //here comes the most complicated set of possibilities: dealing with the last DP:
   //First the easy one: if I am merely moving the last data point within the current graph size,
   //there is nothing to do:
   if (curDP->x <= widget->allocation.width)  return;
   //But if I got here, it means that the user is pulling 
   //last DP beyond size of graph, which turns out to be quite a complex 
   //situation:

   //Because of some apparently untangle-able de-syncs between duration vs. x for 2 DPs
   //in any given move (the problem: I can't know one of the two), I need to translate x in to dur for 
   //all DPs, increment ScheduleGUI_TotalScheduleDuration, then translate the DPs back:
   //First, x to duration conversion:
 double conversion_factor=ScheduleGUI_TotalScheduleDuration/widget->allocation.width;
  tempDP=ScheduleGUI_FirstDataPoint;
 while (tempDP->NextDataPoint != NULL) {
  	tempDP->duration =	(tempDP->NextDataPoint->x-tempDP->x)*conversion_factor;		   
    tempDP=tempDP->NextDataPoint;
   }
  	//do last datapoint:
	  tempDP->duration = 0;//zero because it reached the edge of the graph
   //Now increment ScheduleGUI_TotalScheduleDuration:
   ScheduleGUI_TotalScheduleDuration+=(curDP->x - widget->allocation.width) *conversion_factor;
   
 //Now, convert them back (dur to x conversion):
 conversion_factor=widget->allocation.width/ScheduleGUI_TotalScheduleDuration;
	double currentx=0;
	tempDP=ScheduleGUI_FirstDataPoint;	
//this is necessary because each DP's duration is now out-of-correct proportion with it's X
//Durations are correct, while X has become arbitrary -- so here I make X agree with duration:
	do{
	tempDP->x=currentx;
	currentx+=tempDP->duration*conversion_factor;
	tempDP=tempDP->NextDataPoint;
}while (tempDP != NULL);
//now get new limits (not necessary because I incremented ScheduleGUI_TotalScheduleDuration directly):
//    ScheduleGUI_GetScheduleLimits();
  }
		return;
}









/////////////////////////////////////////////////////
void quit ()
{
  exit (0);
}


/////////////////////////////////////////////////////
//recalculates durations for all datapoints according to their x position on the graph
void  ScheduleGUI_ConvertXToDuration_AllPoints(GtkWidget *widget)
{
 double secs_per_pixel=ScheduleGUI_TotalScheduleDuration/widget->allocation.width;
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;	
 while (currentDataPoint->NextDataPoint != NULL) {
 	currentDataPoint->duration =	(currentDataPoint->NextDataPoint->x - currentDataPoint->x)*secs_per_pixel;		

//	DBGOUT("Dur:",(int)(currentDataPoint->duration+.5));

//this must be in every linked list loop:
		currentDataPoint=currentDataPoint->NextDataPoint;
 };
	
	//do last datapoint:
	currentDataPoint->duration = ScheduleGUI_TotalScheduleDuration - ((currentDataPoint->x)*secs_per_pixel);
//	DBGOUT("LastDur:",(int)(currentDataPoint->duration+.5));
}


/////////////////////////////////////////////////////
//recalculates durations for all datapoints according to their x position on the graph
void  ScheduleGUI_ConvertYToHz_AllPoints(GtkWidget *widget)
{
//double y_scaling_factor=widget->allocation.height/ScheduleGUI_MaxScheduleBeatfrequency;
//	ScheduleGUI_CurrentDataPoint->duration=ScheduleGUI_CurrentDataPoint->x_interval/x_scaling_factor;
	//First the easy one: convert ScheduleGUI_CurrentDataPoint's y to Hz value:
//ScheduleGUI_CurrentDataPoint->hz =	((widget->allocation.height-ScheduleGUI_CurrentDataPoint->y)*ScheduleGUI_MaxScheduleBeatfrequency)/widget->allocation.height;
double hz_per_pixel=ScheduleGUI_MaxScheduleBeatfrequency/widget->allocation.height;
DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
 if	(currentDataPoint->state==SCHEDULEGUI_SELECTED)
	{
		currentDataPoint->hz=(widget->allocation.height - currentDataPoint->y) * hz_per_pixel;
	}

	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}


/////////////////////////////////////////////////////
//NOTE: should change name of this function; it actually requires user 
//to know duration of a point if it is going to be the new last point.
//This translates the X of an existing datapoint in to the correct
//duration in relationship to it's neighbors:
//at some point before calling this, ScheduleGUI_GetScheduleLimits() 
//should have been called once to correctly set two key vars:
//ScheduleGUI_TotalScheduleDuration and
//ScheduleGUI_MaxScheduleBeatfrequency
void ScheduleGUI_ConvertXYToDurHZ(GtkWidget *widget,  DataPoint * currentDataPoint)
{
 //First, convert Y to Hz:
 //NOTE: might want to figure out which point had the highest value again before doing this (lastDataPoint=ScheduleGUI_GetScheduleLimits())
 //currentDataPoint->hz = (widget->allocation.height-currentDataPoint->y)*(ScheduleGUI_MaxScheduleBeatfrequency/widget->allocation.height);
 currentDataPoint->hz = ((widget->allocation.height-currentDataPoint->y)*ScheduleGUI_MaxScheduleBeatfrequency)/widget->allocation.height;

 //Now convert X to Duration:
//The hard part: figuring out new time relationships between neighboring pixels:
//NOTE: rounding errors and pixel resolution have made getting this right a bit of a nightmare,
//propagating/summing small errors along total duration...
 if (currentDataPoint->PrevDataPoint==NULL) return;
	//NEW WAY (20060409): 
 //Benefit: this one doesn't introduce NEARLY as much rounding-error/x-rez  noise
 double oldprevdur=currentDataPoint->PrevDataPoint->duration;
	currentDataPoint->PrevDataPoint->duration = (currentDataPoint->x - currentDataPoint->PrevDataPoint->x) * (ScheduleGUI_TotalScheduleDuration/(double)widget->allocation.width);
 currentDataPoint->duration=oldprevdur - currentDataPoint->PrevDataPoint->duration;
}



/////////////////////////////////////////////////////
//this creates and adds a new DataPoint to the existing linked list based on X Y mouse event,
//assigning a duration relative to neighbor-proximity or (in the case of a point placed after
//formerly last point) distance from end as related to Total Duration (if x was within graph width).
//IMPORTANT: the duration it returns will be assigned 0 if datapoint x was beyond graph width. This
//is why the calling function must be responsible for checking if an x sent here is beyond the graph 
//width, and thus will need to give the DP returned a valid duration (and then probably then do a 
//convertDurHzToXY() to rescale graph to contain last-point's duration).
//NEW 20060412: now user can create data points with same x ("0 duration" points). Became
//a moot point as dragging points past each other created 0 duration points.
DataPoint * ScheduleGUI_InsertNewDataPointXY(GtkWidget *widget, double   x, double   y)
{
	DataPoint * NewDataPoint=NULL;
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
	
	while (currentDataPoint->NextDataPoint!=NULL)
	{
		if (currentDataPoint->x <= x && x <= currentDataPoint->NextDataPoint->x)
		{
			//allot the memory for the new point:
			NewDataPoint=(DataPoint *)malloc(sizeof(DataPoint));
			
			//insert it between the earlier and later points:
			NewDataPoint->NextDataPoint=currentDataPoint->NextDataPoint;
			NewDataPoint->NextDataPoint->PrevDataPoint=NewDataPoint;
			NewDataPoint->PrevDataPoint=currentDataPoint;
			currentDataPoint->NextDataPoint=NewDataPoint;

			//fill it's data. Because it is new, I can reference it to it's neighbor's x 
   //without introducing any (or at least minimal) resolution/rounding errors:
 		NewDataPoint->x=x;
			NewDataPoint->y=y;
			NewDataPoint->state=SCHEDULEGUI_SELECTED;
			NewDataPoint->hz=((widget->allocation.height-y)*ScheduleGUI_MaxScheduleBeatfrequency)/widget->allocation.height;//taken from  ScheduleGUI_ConvertXYToDurHZ()
   //now for the hard one, duration:
   double oldprevdur=NewDataPoint->PrevDataPoint->duration;
	  NewDataPoint->PrevDataPoint->duration = (NewDataPoint->x - NewDataPoint->PrevDataPoint->x) * 
                   (ScheduleGUI_TotalScheduleDuration/(double)widget->allocation.width);
   NewDataPoint->duration=oldprevdur - NewDataPoint->PrevDataPoint->duration;
			
			//found the point's position and filled all its data: all done:
			return NewDataPoint;
		}
		//putc('n',stderr);
		currentDataPoint=currentDataPoint->NextDataPoint;
	}
	//if it got here, the new DP's x was greater than the formerly-last DP's x (second-to-last graphpoint).
	//So, need to deal with two possibilities: whether new DP's x is within graph width (easy), or bigger than graph
 // width (hard).
//		if (currentDataPoint->x <= x && x <= widget->allocation.width) //second conditional is not needed?
		if (currentDataPoint->x <= x)
		{
			//allot the memory for the new point:
			NewDataPoint=(DataPoint *)malloc(sizeof(DataPoint));
			
			//put it after the last real datapoint and link it up to the first:
			NewDataPoint->NextDataPoint=NULL;//it is the real "last one", so NULL NextDataPoint
			currentDataPoint->NextDataPoint=NewDataPoint;
			NewDataPoint->PrevDataPoint=currentDataPoint;
			
			//fill it's data:
 		NewDataPoint->x=x;
			NewDataPoint->y=y;
			NewDataPoint->state=SCHEDULEGUI_SELECTED;
			NewDataPoint->hz=((widget->allocation.height-y)*ScheduleGUI_MaxScheduleBeatfrequency)/widget->allocation.height;//taken from  ScheduleGUI_ConvertXYToDurHZ()
   //now for the hard one, duration:
   double oldprevdur=NewDataPoint->PrevDataPoint->duration;
	  NewDataPoint->PrevDataPoint->duration = (NewDataPoint->x - NewDataPoint->PrevDataPoint->x) * 
                   (ScheduleGUI_TotalScheduleDuration/(double)widget->allocation.width);
   //this next "if" is a bit disconcerting in one way: it says that the new DP will be assigned a zero-duration
   //if it is larger than graph width. This means it is up to the calling function to recognize that the
   //x went past, and then give it a more appropriate duration (if necessary). Importantly, if the calling
   //function is adding multiple points to the end, it only needs to assign a duration and convert it for the very 
   //last point added, since the previously added  "last" ones (initially given a 0 duration) get valid durations 
   //once they become 2nd to last:
   if (x <= widget->allocation.width)    NewDataPoint->duration=oldprevdur - NewDataPoint->PrevDataPoint->duration;
   else {
//    DBGOUT("x is bigger than window-width",(int)NewDataPoint->x);
    NewDataPoint->duration=0;
   }
  	return NewDataPoint;
		}
 //Shouldn't ever get here; will be NULL if it does (which will at least give me a bug to locate the problem)
	return NewDataPoint; 
}








/////////////////////////////////////////////////////
//this creates and adds a new DataPoint to the end of the existing linked list,
//or starts new one if first element doesn't exist. It is obviously pretty slow
//because it has to search for the end every time it is called; but for lists less
//than 1000 elements long it isn't a problem. It it ever gets to be a problem,
//I just come up with a global LastDataPoint -- anyway, just plotting
//100,000 DataPoints takes forever (I tried it).
 DataPoint * ScheduleGUI_AddNewDataPointToEnd(double  duration, double hz)
{
	//always must have a ScheduleGUI_FirstDataPoint:
	if (ScheduleGUI_FirstDataPoint==NULL) 	{
	ScheduleGUI_FirstDataPoint=(DataPoint *) malloc(sizeof(DataPoint));
	ScheduleGUI_FirstDataPoint->x=0;//this is all we know so far graphically
	ScheduleGUI_FirstDataPoint->y=0;
	ScheduleGUI_FirstDataPoint->duration=duration;
	ScheduleGUI_FirstDataPoint->hz=hz;
	ScheduleGUI_FirstDataPoint->NextDataPoint=NULL;
	ScheduleGUI_FirstDataPoint->PrevDataPoint=NULL;
		return ScheduleGUI_FirstDataPoint;
	}

	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;	

	//find end of list:
	while (currentDataPoint->NextDataPoint!=NULL)
	{
			currentDataPoint=currentDataPoint->NextDataPoint;
	}

	currentDataPoint->NextDataPoint=(DataPoint *)malloc(sizeof(DataPoint));
 currentDataPoint->NextDataPoint->PrevDataPoint=currentDataPoint;
 currentDataPoint->NextDataPoint->duration=duration;
 currentDataPoint->NextDataPoint->hz=hz;
	currentDataPoint->NextDataPoint->x=0;
	currentDataPoint->NextDataPoint->y=0;
	currentDataPoint->NextDataPoint->NextDataPoint=NULL;

	return currentDataPoint->NextDataPoint;
}



/////////////////////////////////////////////////////
//20060324
//Creates a duplicate of the main linked-list
//Implemented as a rudimentary one-step, toggle "Undo" feature
//when used with a ScheduleGUI_RestoreDataPoints().
void ScheduleGUI_BackupDataPoints()
{
 //first free any previous backup:
 ScheduleGUI_CleanupDataPoints(ScheduleGUI_FirstBackupDataPoint);
 ScheduleGUI_FirstBackupDataPoint=NULL;//Added 20060501 
  
 //allot memory for first backup data point, so there is a valid starting point for linked list:
 ScheduleGUI_FirstBackupDataPoint=(DataPoint *)malloc(sizeof(DataPoint));
 if (ScheduleGUI_FirstBackupDataPoint==NULL) return;
	DataPoint * currentBackupDataPoint=ScheduleGUI_FirstBackupDataPoint;
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
 DataPoint * lastBackupDataPoint=NULL;//MUST equal NULL!
  
 int count=1;
 //duplicate the list:
  while (1)
  {
   memcpy(currentBackupDataPoint,	currentDataPoint,sizeof(DataPoint));
   currentBackupDataPoint->PrevDataPoint=lastBackupDataPoint;
   lastBackupDataPoint=currentBackupDataPoint;
   if ((currentDataPoint=currentDataPoint->NextDataPoint)==NULL) break;
   else currentBackupDataPoint->NextDataPoint=(DataPoint *)malloc(sizeof(DataPoint));
   currentBackupDataPoint=currentBackupDataPoint->NextDataPoint;
   ++count;
  };

  DBGOUT("Undo DataPoints Created:",count);  
  
  //make sure the last NextDataPoint equals NULL:
  currentBackupDataPoint->NextDataPoint=NULL;
}//END ScheduleGUI_BackupDataPoints()


/////////////////////////////////////////////////////
//20060324
//Restores a backup (if existant) of main linked-list
//Implemented as a rudimentary one-step,toggle "Undo" feature
//when used with a ScheduleGUI_BackupDataPoints()
void ScheduleGUI_RestoreDataPoints(GtkWidget * widget)
{
 //first validate ScheduleGUI_FirstBackupDataPoint:
 if (ScheduleGUI_FirstBackupDataPoint==NULL) return;
 
 //delete old graph data:
 //ScheduleGUI_CleanupDataPoints(ScheduleGUI_FirstDataPoint);
 //20060328 -- instead of deleting old graph data (above), smarter to make 
 //it the "Backup" data. But it is up to implementation to make clear that
 //restoring this "Undo" is actually a "Redo" after an "Undo":
 DataPoint * tmpDP=ScheduleGUI_FirstDataPoint;
 ScheduleGUI_FirstDataPoint=ScheduleGUI_FirstBackupDataPoint;
 ScheduleGUI_FirstBackupDataPoint=tmpDP;
 ScheduleGUI_CurrentDataPoint=NULL; 
 ScheduleGUI_ConvertDurHzToXY(widget);
 ScheduleGUI_DrawGraph(widget); 
}



/////////////////////////////////////////////////////
//20060412
//Copies any selected points to a simple array to create
//a paste-able mass when used with a ScheduleGUI_PasteDataPoints().
void ScheduleGUI_CopySelectedDataPoints(GtkWidget * widget)
{
	DataPoint * currentDataPoint;	
 
//First count how many DPs are selected. One reason to do this first is to be sure
//there really are selected DPs before throwing away the old buffer:
int count=0;
currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED) ++count;//++ScheduleGUI_CopyPaste.count;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);

//if there are no selected DPs, don't do anything:
if (count<1) return;

 //Now it is OK to throw away old buffer:
	if (ScheduleGUI_CopyPaste.data !=NULL) {
		free(ScheduleGUI_CopyPaste.data);
		ScheduleGUI_CopyPaste.data=NULL;
	}

	//Now get scaling vars required to correctly paste data regardless of how 
 //drawingarea dimensions may have changed:
 //Calling ScheduleGUI_GetScheduleLimits() just to be safe (it fills the DurHz vars below):
 ScheduleGUI_GetScheduleLimits();
 ScheduleGUI_CopyPaste.OrigScheduleGUI_TotalScheduleDuration = ScheduleGUI_TotalScheduleDuration;
 ScheduleGUI_CopyPaste.OrigScheduleGUI_MaxScheduleBeatfrequency = ScheduleGUI_MaxScheduleBeatfrequency;
	ScheduleGUI_CopyPaste.OrigWidth=widget->allocation.width;
	ScheduleGUI_CopyPaste.OrigHeight=widget->allocation.height;
	ScheduleGUI_CopyPaste.count=count;

 //allot the memory:
 ScheduleGUI_CopyPaste.data  = (DataPoint *) calloc(ScheduleGUI_CopyPaste.count,sizeof(DataPoint));
 if (ScheduleGUI_CopyPaste.data == NULL) return;
 
//	Now put all the selected DPs in to the copy buffer:
currentDataPoint=ScheduleGUI_FirstDataPoint;
ScheduleGUI_CopyPaste.count=0;
do {
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED) {
		memcpy(&(ScheduleGUI_CopyPaste.data[ScheduleGUI_CopyPaste.count]), currentDataPoint,sizeof(DataPoint));
		++ScheduleGUI_CopyPaste.count;
	}
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}//END ScheduleGUI_CopySelectedDataPoints()


/////////////////////////////////////////////////////
//20060412
//Pastes any previously selected points copied with ScheduleGUI_PasteDataPoints().
void ScheduleGUI_PasteSelectedDataPoints(GtkWidget * widget, gboolean predeselect)
{
	if (ScheduleGUI_CopyPaste.data == NULL || ScheduleGUI_CopyPaste.count < 1) return;
	ScheduleGUI_ProgressIndicatorFlag=FALSE;
 gboolean convertflag=FALSE;
	//Not sure what user would want:
	if (predeselect == TRUE) ScheduleGUI_DeselectDataPoints();

 double xscale=widget->allocation.width/(double)ScheduleGUI_CopyPaste.OrigWidth;
 double yscale=widget->allocation.height/(double)ScheduleGUI_CopyPaste.OrigHeight;
	int count = 0;
	DataPoint * curDP;

 //There are three possibilities that matter here: 
 // 1) Hz scale and drawingarea dimensions have stayed the same since copy (1:1 XY copy)
 // 2) drawingarea dimensions have changed, scale still the same (scaled XY copy)
 // 3) Hz scale has changed, must do complex translation of DurationHz to convert to XY
 
 //This satisfies possibilities 1 and 2:
 if (ScheduleGUI_TotalScheduleDuration == ScheduleGUI_CopyPaste.OrigScheduleGUI_TotalScheduleDuration &&
     ScheduleGUI_MaxScheduleBeatfrequency==ScheduleGUI_CopyPaste.OrigScheduleGUI_MaxScheduleBeatfrequency)
 {
//  DBGOUT("Paste Context Hasn't Changed",(int)(xscale*1000));
	do {
		curDP=ScheduleGUI_InsertNewDataPointXY(widget, 
    ScheduleGUI_CopyPaste.data[count].x * xscale,
    ScheduleGUI_CopyPaste.data[count].y * yscale);
		++count;
	} while (count <	ScheduleGUI_CopyPaste.count);
 } else
{ //This satisfies possibility 3: Needs a different approach, because the original XY context for the points 
   //is no gone. Easiest thing to do is just paste the mess with the first point at x=0, but it would be
   //really nice to try and paste it "where it was", time-wise, in case user really just changed a few details
 //Scale all x's appropriately and calculate appropriate durations:
 xscale *= ScheduleGUI_CopyPaste.OrigScheduleGUI_TotalScheduleDuration / ScheduleGUI_TotalScheduleDuration;
 DBGOUT("Paste context changed: scale*1000=",(int)(xscale*1000));
 count = 0;
	do {
		curDP = ScheduleGUI_InsertNewDataPointXY(widget, 
    ScheduleGUI_CopyPaste.data[count].x * xscale,
    0); //ScheduleGUI_CopyPaste.data[count].y);//NOTE: I MANUALLY SET THIS
  curDP->hz = ScheduleGUI_CopyPaste.data[count].hz;
//  if (curDP->hz > ScheduleGUI_MaxScheduleBeatfrequency) convertflag=TRUE;
  if (curDP->x > widget->allocation.width) {//deal with x's beyond current graph width:
   DBGOUT("Pasted point beyond graph width", (int)curDP->x);
//   convertflag=TRUE;
   curDP->duration = ScheduleGUI_CopyPaste.data[count].duration;
  }
  //DBGOUT("Original Dur:",(int) ScheduleGUI_CopyPaste.data[count].duration); DBGOUT("New Dur:",(int) curDP->duration);
 	++count;
	} while (count <	ScheduleGUI_CopyPaste.count);

 //20060421: I've observed that it is wise to NOT call this function unless absolutely 
 //necessary; it can introduce extremely subtle but insidiously accumulating
// rounding errors in some cases
// if (convertflag==TRUE) 
 ScheduleGUI_ConvertDurHzToXY(widget);
 }

 //draw it:
	ScheduleGUI_DrawGraph(widget);
}//END ScheduleGUI_CopySelectedDataPoints()

















/////////////////////////////////////////////////////
//this offers a means for the user to cull invisible duplicate DPs.
void ScheduleGUI_DeleteDuplicateDataPoints(GtkWidget * widget, double threshold=0.0)
{
	int count=0;
DataPoint * curDP;	
curDP=ScheduleGUI_FirstDataPoint->NextDataPoint;
if (curDP==NULL) return;
 
 if (threshold==0.0) while (curDP->NextDataPoint != NULL)
{
 if (curDP->x == curDP->NextDataPoint->x &&
		   curDP->y == curDP->NextDataPoint->y){
						ScheduleGUI_DeleteDataPoint(curDP->NextDataPoint);
						curDP=curDP->PrevDataPoint;
						++count;
						}
	//all list loops need this:
	curDP=curDP->NextDataPoint;
} else while (curDP->NextDataPoint != NULL)
{
 if (fabs(curDP->x - curDP->NextDataPoint->x) < threshold &&
		    fabs(curDP->y - curDP->NextDataPoint->y) < threshold) {
						ScheduleGUI_DeleteDataPoint(curDP->NextDataPoint);
						curDP=curDP->PrevDataPoint;
						++count;
						}
	//all list loops need this:
	curDP=curDP->NextDataPoint;
}

	if (count>0) 	DBGOUT("Duplicates deleted:",count);

	ScheduleGUI_DrawGraph(widget);
}



/*
/////////////////////////////////////////////////////
//literally just paints all regions with selected points under them
//red -- needed because selected DPs can hide behind unselected ones:
void ScheduleGUI_HighlightSelectedDataPoints(GtkWidget * widget) 
{	
 if (ScheduleGUI_FirstDataPoint == NULL) return;
	
	GdkColor color;
	color.red = 65535; color.green = 32768; color.blue = 32768;
	gdk_gc_set_rgb_fg_color(ScheduleGUI_gc, &color);  

	DataPoint * curDP=ScheduleGUI_FirstDataPoint;
do{
	if (curDP->state == SCHEDULEGUI_SELECTED) 
   gdk_draw_rectangle (ScheduleGUI_pixmap, ScheduleGUI_gc, FALSE,
     (int)(curDP->x+.5)-(SCHEDULEGUI_SQUARESIZE>>1), 
     (int)(curDP->y+.5)-(SCHEDULEGUI_SQUARESIZE>>1), 
     SCHEDULEGUI_SQUARESIZE, SCHEDULEGUI_SQUARESIZE);	

//all list loops need this:
	curDP=curDP->NextDataPoint;
}while (curDP != NULL);
//paint it:
gtk_widget_queue_draw_area (widget,0,0,widget->allocation.width, widget->allocation.height);
}
*/





/////////////////////////////////////////////////////
//20060326: this was made to be able to clean any linked list of
//data points (so both main and backup sets can use this func)
//NOTE: User must now make sure to NULL their own FirstDataPoint when done!
void ScheduleGUI_CleanupDataPoints(DataPoint * firstDataPoint)
{
 if (firstDataPoint==NULL) return;
		DataPoint * currentDataPoint=firstDataPoint;
		DataPoint * nextDataPoint;
	 unsigned int count=0;
	while (currentDataPoint->NextDataPoint!=NULL)
	{
		nextDataPoint=currentDataPoint->NextDataPoint;
		free(currentDataPoint);
		currentDataPoint=nextDataPoint;
  ++count;
		}

//still need to get the last one:		
  if (currentDataPoint!=NULL) 
		{
			free(currentDataPoint);
			++count;
		}

  if (firstDataPoint==ScheduleGUI_FirstDataPoint)  { DBGOUT("DataPoints Deleted:",count) }
  else { DBGOUT("Undo DataPoints Deleted:",count) }
		  
  //User must make sure to NULL their First Data Point!
//  firstDataPoint=NULL;
}



/////////////////////////////////////////////////////
//this fills global variables ScheduleGUI_MaxScheduleBeatfrequency
//and ScheduleGUI_TotalScheduleDuration. It can be
//called without a widget, making it a bit more general than
//related workhorse, ScheduleGUI_ConvertDurHzToXY()
void ScheduleGUI_GetScheduleLimits()
{
 //DBGOUT("Got to ScheduleGUI_GetScheduleLimits\n",0);
	ScheduleGUI_TotalScheduleDuration=0;
	ScheduleGUI_MaxScheduleBeatfrequency=0;
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	ScheduleGUI_TotalScheduleDuration+=currentDataPoint->duration;
	if (ScheduleGUI_MaxScheduleBeatfrequency<currentDataPoint->hz) ScheduleGUI_MaxScheduleBeatfrequency=currentDataPoint->hz;
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}


/////////////////////////////////////////////////////
//NOTE: This function turns out to be one of the most critical to the entire
//functionality of ScheduleGUI, being called virtually every time anything is
//done to the points. It is worth really studying this one to see where/if it 
//is introducing subtle but accumulating rounding-errors or graphpoint/datapoint 
//resolution errors (since for many operations, a DataPoint's x determined here
//gets used to recalibrate that DataPoint's duration).
////////
//This function first determines global duration and hz limits, then uses 
//that info in combination with drawingarea dimensions to give exact XY
//placement for DataPoints. Used after a drawingarea resize, or any time
//a change in graphical context invalidates datapoints' xy's
//NOTE: I inlined ScheduleGUI_GetScheduleLimits() here, since the 
//two almost always need to be called together.
void ScheduleGUI_ConvertDurHzToXY(GtkWidget *widget)
{
	ScheduleGUI_TotalScheduleDuration=0;
	ScheduleGUI_MaxScheduleBeatfrequency=0;
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
 //First get ScheduleGUI_MaxScheduleBeatfrequency (so I know Y scaling) and
 //ScheduleGUI_TotalScheduleDuration (so I know X scaling):
do{
	ScheduleGUI_TotalScheduleDuration+=currentDataPoint->duration;
	if (ScheduleGUI_MaxScheduleBeatfrequency<currentDataPoint->hz) ScheduleGUI_MaxScheduleBeatfrequency=currentDataPoint->hz;
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
  
//Now limit-check DurHz maximums:
//this is to keep our vertical scale within reasonable limits, and avoid divide-by-0 issues:
if (ScheduleGUI_MaxScheduleBeatfrequency<.01) ScheduleGUI_MaxScheduleBeatfrequency=.04;

//come up with suitable scaling factors 
 double x_scaling_factor=widget->allocation.width/ScheduleGUI_TotalScheduleDuration;
 double y_scaling_factor=widget->allocation.height/ScheduleGUI_MaxScheduleBeatfrequency;
	double currentx=0;

	
//this simply lays all the points along a running sum along the x axis proportional to the durations:
	currentDataPoint=ScheduleGUI_FirstDataPoint;
	do{
	currentDataPoint->x=currentx;
	currentx+=currentDataPoint->duration*x_scaling_factor;
	currentDataPoint->y=widget->allocation.height-(currentDataPoint->hz*y_scaling_factor);
// DBGOUT("Duration:", (int)currentDataPoint->duration);
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}




/////////////////////////////////////////////////////
void ScheduleGUI_DeselectDataPoints() 
{	
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}


/////////////////////////////////////////////////////
void ScheduleGUI_SelectAllDataPoints() 
{	
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	currentDataPoint->state=SCHEDULEGUI_SELECTED;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}



/////////////////////////////////////////////////////
void ScheduleGUI_InvertSelectionAllDataPoints() 
{	
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED) currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
 else currentDataPoint->state=SCHEDULEGUI_SELECTED;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}

/////////////////////////////////////////////////////
void ScheduleGUI_SelectIntervalAllDataPoints(int interval, gboolean deselectothers) 
{	
 if (interval<2) {
  ScheduleGUI_SelectAllDataPoints(); 
  return;
 }
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
 int count=0;
do{
	if (count==0) currentDataPoint->state=SCHEDULEGUI_SELECTED;
 else if (deselectothers==TRUE) currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
  ++count;
 if (count>=interval) count=0;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
}



/////////////////////////////////////////////////////
//if next is TRUE, selects neighbors to the right; FALSE means left.
//if deselect=TRUE, it deselects the original DataPoints when selecting their neighbors
void ScheduleGUI_SelectNeighboringDataPoints(gboolean next, gboolean deselect) 
{
 DataPoint *  currentDataPoint=ScheduleGUI_FirstDataPoint;
 
 if (next==FALSE) {
do{
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED 
     && currentDataPoint->PrevDataPoint != NULL)  {
      currentDataPoint->PrevDataPoint->state=SCHEDULEGUI_SELECTED;
     }
     if (deselect==TRUE) currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
//all list loops need this:
	currentDataPoint=currentDataPoint->NextDataPoint;
}while (currentDataPoint != NULL);
} else {
 //first find end:
 while (currentDataPoint->NextDataPoint !=NULL) currentDataPoint=currentDataPoint->NextDataPoint;
do{
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED 
     && currentDataPoint->NextDataPoint != NULL) {
      currentDataPoint->NextDataPoint->state=SCHEDULEGUI_SELECTED;
     }
     if (deselect==TRUE) currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
//all list loops need this:
	currentDataPoint=currentDataPoint->PrevDataPoint;
}while (currentDataPoint != NULL);
  
 
}




}



/////////////////////////////////////////////////////
// if DeleteTime==FALSE (default) this function adds time from deleted point
//to the previous point. TRUE deletes that time, making it essential
//to call ScheduleGUI_GetScheduleLimits() at some point before 
//rendering, etc. (advisable anyway).
void ScheduleGUI_DeleteDataPoint(DataPoint * curDP, gboolean DeleteTime)
{
	if (curDP==NULL) return;

	//to be sure we don't end up trying to delete the ONLY point:
 if (curDP->NextDataPoint==NULL && curDP->PrevDataPoint==NULL) return;
	
	DataPoint * prevDP=curDP->PrevDataPoint;
		
	//first check to see if ScheduleGUI_FirstDataPoint got deleted:
	if (prevDP==NULL) {
		ScheduleGUI_FirstDataPoint=ScheduleGUI_FirstDataPoint->NextDataPoint;
		ScheduleGUI_FirstDataPoint->PrevDataPoint=NULL;
	}
	else {
		//sum currentDataPoint's duration in to previous DataPoints:
		if (DeleteTime==FALSE)  prevDP->duration+=curDP->duration;
		//Make previous DataPoint now link to the next DataPoint after the one to be deleted:
		prevDP->NextDataPoint=curDP->NextDataPoint;
 if (curDP->NextDataPoint!=NULL) //i.e., if this point is not the last DataPoint in the llist:
 {
		//Make NextDataPoint now call the prevDataPoint PrevDataPoint:
		curDP->NextDataPoint->PrevDataPoint=prevDP;
	}
	}
	//Assuming I've linked everything remaining properly, can now free the memory:
	free(curDP);
	//recalibrate everything: [TOO SLOW! Let calling function do this, in case it is calling 8000 points]
//	ScheduleGUI_ConvertDurHzToXY(widget);
//	ScheduleGUI_DrawGraph(widget);	
}




/////////////////////////////////////////////////////
void ScheduleGUI_DeleteSelectedPoints(GtkWidget * widget, gboolean DeleteTime)
{
 ScheduleGUI_ProgressIndicatorFlag=FALSE;
	//DBGOUT("before ScheduleGUI_ShiftFlag",ScheduleGUI_ShiftFlag);
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;	
 while (currentDataPoint->NextDataPoint != NULL) {
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED) 
	 {
		//since I'm about to obliterate it, I need to advance now:
		 currentDataPoint=currentDataPoint->NextDataPoint;
	 	ScheduleGUI_DeleteDataPoint(currentDataPoint->PrevDataPoint, DeleteTime);
  }
  else currentDataPoint=currentDataPoint->NextDataPoint;
 }
	//now check last datapoint:
	if (currentDataPoint->state==SCHEDULEGUI_SELECTED)	 ScheduleGUI_DeleteDataPoint(currentDataPoint, DeleteTime);
	ScheduleGUI_ConvertDurHzToXY(widget);
	ScheduleGUI_DrawGraph(widget);	
	//DBGOUT("after ScheduleGUI_ShiftFlag",ScheduleGUI_ShiftFlag);
}



/////////////////////////////////////////////////////
//selects datapoint in a rectangular region of graph
void ScheduleGUI_SelectDataPoints(int startX,int startY,int endX,int endY, gboolean deselect) 
{	
	DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;
do{
	if (currentDataPoint->x > startX && currentDataPoint->x < endX && 
		   currentDataPoint->y > startY && currentDataPoint->y < endY) {
 	  		if (deselect==FALSE) currentDataPoint->state=SCHEDULEGUI_SELECTED;
						else currentDataPoint->state=SCHEDULEGUI_UNSELECTED;
					}
	 else {
			//No need to deselect points, because the initial button_press_event handler
			//will have already deselected them if SHIFT isn't pressed.
			//currentDataPoint->state=SCHEDULEGUI_UNSELECTED; 
		}
	 currentDataPoint=currentDataPoint->NextDataPoint;
 }while (currentDataPoint != NULL);
}


/////////////////////////////////////////////////////
//THIS IS WHERE EXTERNAL PROGRAMS SHOULD SETUP DATA WITH
//SCHEDULEGUI -- ONE TIME ONLY AT THE START
/////////////////////////////////////////////////////
void ScheduleGUI_SetupDataPoints() 
{
	ScheduleGUI_SelectionBox.status=0;
 ScheduleGUI_CopyPaste.count=0;
	ScheduleGUI_CopyPaste.data=NULL;
#ifdef SCHEDULEGUI_FREESTANDING
	//this line is just for testing the standalone; is replaced by a real feeder from app using ScheduleGUI:
	int i; 	for (i=0;i<1000;i++)  	if (NULL==ScheduleGUI_AddNewDataPointToEnd(30, .01*((rand()%700)+300))) exit(0);
#else
//#################	
//this is the line that connects data up to current BinauralBeat data: 
	ScheduleGUI_CleanupDataPoints(ScheduleGUI_FirstDataPoint); 
 ScheduleGUI_FirstDataPoint=NULL; //Must do this after calling ScheduleGUI_CleanupDataPoints!
 for (int i=0; i<bb->ScheduleEntriesCount; i++)
 {
  ScheduleGUI_AddNewDataPointToEnd(bb->Schedule[i].Duration, 
  fabs(bb->Schedule[i].FreqRStart-bb->Schedule[i].FreqLStart));
  }
  		//fprintf(stderr,"Created %d DataPoints\n", bb->ScheduleEntriesCount);
  DBGOUT("DataPoints Created:",bb->ScheduleEntriesCount);  
//#################	
#endif
//I do this to give ScheduleGUI_MaxScheduleBeatfrequency and 
//ScheduleGUI_TotalScheduleDuration some reasonable starting 
//points (they change whenever size of drawingarea changes):
 ScheduleGUI_GetScheduleLimits();	
 
 //just general housecleaning:  
	ScheduleGUI_DeselectDataPoints();
}


gboolean ScheduleGUI_key_press_event( GtkWidget * widget, GdkEventKey * event )
{
 //NOTE: ALMOST EVERYTHING IS GNAURAL IS HANDLED BY ACCELLERATORS,
 //therefore almost everything here is only needed for freestanding mode:
 #ifdef SCHEDULEGUI_FREESTANDING
 //delete points and delete points and time (Shift):
	if (event->keyval==GDK_Delete) ScheduleGUI_DeleteSelectedPoints(widget, (event->state & GDK_SHIFT_MASK));
	else if (event->state & GDK_CONTROL_MASK) {
 //select all:
	if (event->keyval==GDK_a) {
	   ScheduleGUI_SelectDataPoints(-8,-8,widget->allocation.width+8, widget->allocation.height+8);
 			ScheduleGUI_ConvertDurHzToXY(widget);
   	ScheduleGUI_DrawGraph(widget);	
		}
  //copy:
		else if (event->keyval==GDK_c) ScheduleGUI_CopySelectedDataPoints(widget);
  //paste:
		else if (event->keyval==GDK_v) ScheduleGUI_PasteSelectedDataPoints(widget, TRUE);
  //cut:
		else if (event->keyval==GDK_x) { 
   ScheduleGUI_BackupDataPoints();
   ScheduleGUI_CopySelectedDataPoints(widget); 
   ScheduleGUI_DeleteSelectedPoints(widget, FALSE); 
   }
  //NOTE: undo/redo are dealt with by accellerators because their workings are closely tied with menu states:
  else if (event->keyval==GDK_z || event->keyval==GDK_y) { ScheduleGUI_RestoreDataPoints(widget); }
  else 
  //cut (deleting time also):
  if (event->keyval==GDK_X) { 
    ScheduleGUI_BackupDataPoints();
   ScheduleGUI_CopySelectedDataPoints(widget); 
   ScheduleGUI_DeleteSelectedPoints(widget, TRUE); 
   }
   
	} else 

 if (event->state & GDK_SHIFT_MASK) {//I have to do this with Shift down because Glade/GTK steals Arrow keys:
		if (event->keyval==GDK_Up || event->keyval==GDK_Down) { 
   ScheduleGUI_MoveSelectedDataPoints(widget, 0, (event->keyval==GDK_Up) ? -1:1); 
   ScheduleGUI_ConvertYToHz_AllPoints(widget);
   ScheduleGUI_ConvertDurHzToXY(widget);// NOTE: I need to call this both to set limits and to bring XY's outside of graph back in
   ScheduleGUI_DrawGraph(widget);
   }
		else if (event->keyval==GDK_Left || event->keyval==GDK_Right) {
   ScheduleGUI_MoveSelectedDataPoints(widget, (event->keyval==GDK_Left) ? -1:1,0); 
   ScheduleGUI_ConvertXToDuration_AllPoints(widget);
   ScheduleGUI_ConvertDurHzToXY(widget);// NOTE: I need to call this both to set limits and to bring XY's outside of graph back in
   ScheduleGUI_DrawGraph(widget);
   }
 }
  #endif
	
  return FALSE;
}


/*
/////////////////////////////////////////////////////
gboolean ScheduleGUI_key_release_event (GtkWidget  *widget, GdkEventKey  *event)
{
  //if (event->state & GDK_CONTROL_MASK)  DBGOUT("KeyRelease",event->keyval);//g_print ("The control key is pressed\n");
//	DBGOUT("KeyRelease",event->keyval);
//	ScheduleGUI_ShiftFlag=(event->state & GDK_SHIFT_MASK);
//	DBGOUT("Shiftflag release:",event->state & GDK_SHIFT_MASK);
  return FALSE;
}
*/

/////////////////////////////////////////////////////
void ScheduleGUI_MoveSelectedDataPoints(GtkWidget * widget, double moveX, double moveY)
{
//Go through all selected points and move them accordingly:
DataPoint * currentDataPoint=ScheduleGUI_FirstDataPoint;	
do {
	if (currentDataPoint->state == SCHEDULEGUI_SELECTED) {
//	if (FALSE == ScheduleGUI_MoveDataPoint(widget, currentDataPoint, currentDataPoint->x+moveX,currentDataPoint->y+moveY)) break;
	 ScheduleGUI_MoveDataPoint(widget, currentDataPoint, currentDataPoint->x+moveX,currentDataPoint->y+moveY);
 }
	//this must be in every linked list loop:
	currentDataPoint=currentDataPoint->NextDataPoint;
} while (currentDataPoint != NULL);
}



/////////////////////////////////////////////////////
void ScheduleGUI_Cleanup()
{
//done writing text, so discard this:
 if (ScheduleGUI_layout!=NULL) g_object_unref(ScheduleGUI_layout);
	if (ScheduleGUI_gc!=NULL) g_object_unref(ScheduleGUI_gc);
 if (ScheduleGUI_pixmap!=NULL)   g_object_unref (ScheduleGUI_pixmap);
 ScheduleGUI_pixmap=NULL;
 ScheduleGUI_gc=NULL;
 ScheduleGUI_layout=NULL;
 ScheduleGUI_CleanupDataPoints(ScheduleGUI_FirstBackupDataPoint);
 ScheduleGUI_FirstBackupDataPoint=NULL;//ALWAYS REMEMBER TO DO THIS!
	ScheduleGUI_CleanupDataPoints(ScheduleGUI_FirstDataPoint);
 ScheduleGUI_FirstDataPoint=NULL;//ALWAYS REMEMBER TO DO THIS!
	if (ScheduleGUI_CopyPaste.data != NULL) free(ScheduleGUI_CopyPaste.data);
}

/////////////////////////////////////////////////////
//This is the formal form required if it is really going to be a callback (ie, ScheduleGUI is freestanding).
//If ScheduleGUI is used as a module, just call ScheduleGUI_Cleanup();
//gint ScheduleGUI_delete_event(GtkWidget* window, GdkEventAny* e, gpointer data)
gboolean ScheduleGUI_delete_event(GtkWidget* window, GdkEvent* e, gpointer data)
{
 ScheduleGUI_Cleanup();
 
#ifdef SCHEDULEGUI_FREESTANDING
 gtk_main_quit();
#endif
  return FALSE;
}



#ifdef SCHEDULEGUI_FREESTANDING
/////////////////////////////////////////////////////
int main( int   argc, char *argv[] )
{
 srand(time(NULL));
	ScheduleGUI_FirstDataPoint=NULL;
//			if (ScheduleGUI_FirstDataPoint!=NULL) ScheduleGUI_CleanupDataPoints();
 ScheduleGUI_SetupDataPoints();
	
  GtkWidget *window;
  GtkWidget *drawing_area;
  GtkWidget *vbox;

  gtk_init (&argc, &argv);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name (window, "ScheduleGUI");

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);

  g_signal_connect (G_OBJECT (window), "destroy", G_CALLBACK (quit), NULL);

  /* Create the drawing area */
  drawing_area = gtk_drawing_area_new ();
  gtk_widget_set_size_request (GTK_WIDGET (drawing_area), 800, 128);
  gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

  gtk_widget_show (drawing_area);

  GTK_WIDGET_SET_FLAGS (drawing_area, GTK_CAN_FOCUS);
//  gtk_widget_add_events (drawing_area, 
  gtk_widget_set_events (drawing_area, 
		    GDK_EXPOSURE_MASK 
				| GDK_POINTER_MOTION_MASK 
				| GDK_POINTER_MOTION_HINT_MASK 
				| GDK_BUTTON_PRESS_MASK 
				| GDK_BUTTON_RELEASE_MASK 
				| GDK_KEY_PRESS_MASK //    | GDK_KEY_RELEASE_MASK);
				| GDK_LEAVE_NOTIFY_MASK);


 // Signals used to handle backing pixmap
  g_signal_connect (G_OBJECT (drawing_area), "realize", G_CALLBACK (ScheduleGUI_realize), NULL);//not actually using this; docs too poor to figure it out
  g_signal_connect (G_OBJECT (drawing_area), "expose_event", G_CALLBACK (ScheduleGUI_expose_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area),"configure_event", G_CALLBACK (ScheduleGUI_configure_event), NULL);
		gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(ScheduleGUI_delete_event), NULL);

  // Event signals
  g_signal_connect (G_OBJECT (drawing_area), "motion_notify_event", G_CALLBACK (ScheduleGUI_motion_notify_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "button_press_event", G_CALLBACK (ScheduleGUI_button_press_event), NULL);
  g_signal_connect (G_OBJECT (drawing_area), "button_release_event", G_CALLBACK (ScheduleGUI_button_release_event), NULL);
		g_signal_connect (G_OBJECT (drawing_area), "key_press_event", G_CALLBACK (ScheduleGUI_key_press_event), NULL);
//  g_signal_connect (G_OBJECT (drawing_area), "key_release_event", G_CALLBACK (ScheduleGUI_key_release_event), NULL);
//  g_signal_connect ((gpointer) window, "key_press_event", G_CALLBACK (ScheduleGUI_key_press_event),  NULL);
//  g_signal_connect ((gpointer) window, "key_release_event", G_CALLBACK (ScheduleGUI_key_release_event),  NULL);

  gtk_widget_show (window);
 
	gtk_main ();

  return 0;
}
#endif
