/*
 * Copyright (C) 2004,2005 Olivier Rossiny <mrfreeze@tuxfamily.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define GLADE_HOOKUP_OBJECT(component,widget,name) \
  g_object_set_data_full (G_OBJECT (component), name, \
    gtk_widget_ref (widget), (GDestroyNotify) gtk_widget_unref)

#define GLADE_HOOKUP_OBJECT_NO_REF(component,widget,name) \
  g_object_set_data (G_OBJECT (component), name, widget)

int nb[2];	/* Nombre d'éléments dans chaque liste */
netswitch *netswitches;
int interface;
GtkWidget *mainW, *boite = NULL, *auto_menu = NULL;
GtkListStore *scanmodel;

enum {
	COLUMN_BSSID,
	COLUMN_ESSID,
	COLUMN_PROTOCOL,
	COLUMN_ENCRYPTION,
	COLUMN_MODE,
	COLUMN_LINK,
	COLUMN_NUMBER
};

netswitch *netswitch_lookup(char *iface)
{
  netswitch *n = netswitches;
  do {
   n = n->next;
  } while(strcmp(iface, n->iface));
  return n;
}

GtkWidget* create_interface_block(GtkWidget *parent, netswitch *n)
{
  GtkWidget *block = gtk_vbox_new(FALSE, 5);
  char *txt = malloc(80);
  GtkWidget *image;
  if (n->type == WIRELESS) {
    sprintf(txt, _("Wireless %s interface"), n->iface);
    image = gtk_image_new_from_pixbuf(create_pixbuf("wifi.png"));
  } else {
    sprintf(txt, _("Ethernet %s interface"), n->iface);
    image = gtk_image_new_from_pixbuf(create_pixbuf("wire.png"));
  }
  gtk_box_pack_start(GTK_BOX(block), image, FALSE, FALSE, 0);
  GtkWidget *title = gtk_label_new(txt);
  gtk_box_pack_start(GTK_BOX(block), title, FALSE, FALSE, 0);
  sprintf(txt, "image_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, image, txt);
  GtkWidget *active = gtk_label_new(NULL);
  sprintf(txt, "active_profile_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, active, txt);
  gtk_box_pack_start(GTK_BOX(block), active, FALSE, FALSE, 0);
  GtkListStore *tree = gtk_list_store_new(1, G_TYPE_STRING);
  GtkWidget *profiles = gtk_combo_box_new_with_model(GTK_TREE_MODEL(tree));
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(profiles), renderer, TRUE);
  gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(profiles), renderer, "text", 0, NULL);
  sprintf(txt, "profiles_%s", n->iface);
  gtk_box_pack_start(GTK_BOX(block), profiles, FALSE, FALSE, 0);
  GLADE_HOOKUP_OBJECT (parent, profiles, txt);

  GtkWidget *hbox = gtk_hbox_new(TRUE, 0);
  GtkWidget *toolbar = gtk_toolbar_new();
  gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_show_arrow(GTK_TOOLBAR(toolbar), FALSE);
  gtk_box_pack_start(GTK_BOX(block), hbox, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), toolbar, FALSE, FALSE, 0);
  GtkWidget *item = GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_ADD));
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), GTK_TOOL_ITEM(item),0);
  g_signal_connect ((gpointer) item, "clicked",
    G_CALLBACK(ajouter), g_strdup(n->iface));
  item = GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_EDIT));
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), GTK_TOOL_ITEM(item),1);
  g_signal_connect ((gpointer) item, "clicked",
    G_CALLBACK(modification), g_strdup(n->iface));
  item = GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_DELETE));
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), GTK_TOOL_ITEM(item),2);
  g_signal_connect ((gpointer) item, "clicked",
    G_CALLBACK(supprimer), g_strdup(n->iface));
  item = GTK_WIDGET(gtk_tool_button_new_from_stock(GTK_STOCK_APPLY));
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), GTK_TOOL_ITEM(item),3);
  g_signal_connect ((gpointer) item, "clicked",
    G_CALLBACK(appliquer), g_strdup(n->iface));

  g_signal_connect ((gpointer) profiles, "changed",
    G_CALLBACK(toggle_toolbar), GTK_OBJECT(toolbar));

  GtkWidget *info_table = gtk_table_new (4, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(info_table), 5);
  gtk_box_pack_start(GTK_BOX(block), info_table, FALSE, FALSE, 0);
  int index = 0;
  GtkWidget *label;
  if (n->type == WIRELESS) {
    label = gtk_label_new(_("Wireless information:"));
    sprintf(txt, "label_essid_%s", n->iface);
    GLADE_HOOKUP_OBJECT (parent, label, txt);
    gtk_table_attach(GTK_TABLE(info_table),label,0,1,index,index+1,GTK_EXPAND,0,0,0);
    label = gtk_progress_bar_new();
    sprintf(txt, "link_%s", n->iface);
    GLADE_HOOKUP_OBJECT (parent, label, txt);
    gtk_widget_set_size_request(label, 40, -1);
    gtk_table_attach(GTK_TABLE(info_table),label,2,3,index,1+index,GTK_SHRINK,0,0,0);
    label = gtk_label_new(NULL);
    sprintf(txt, "essid_%s", n->iface);
    GLADE_HOOKUP_OBJECT (parent, label, txt);
    gtk_table_attach(GTK_TABLE(info_table),label,1,2,index,index+1,GTK_EXPAND,0,0,0);
    ++index;
  }
  label = gtk_label_new(_("MAC address:"));
  sprintf(txt, "label_mac_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,0,1,index,1+index,GTK_EXPAND,0,0,0);
  label = gtk_label_new(NULL);
  sprintf(txt, "mac_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,1,3,index,1+index,GTK_EXPAND,0,0,0);
  ++index;
  label = gtk_label_new(_("IP address:"));
  sprintf(txt, "label_ipaddr_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,0,1,index,1+index,GTK_EXPAND,0,0,0);
  label = gtk_label_new(NULL);
  sprintf(txt, "ipaddr_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,1,3,index,1+index,GTK_EXPAND,0,0,0);
  ++index;
  label = gtk_label_new(_("Netmask:"));
  sprintf(txt, "label_net_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,0,1,index,1+index,GTK_EXPAND,0,0,0);
  label = gtk_label_new(NULL);
  sprintf(txt, "net_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,1,3,index,1+index,GTK_EXPAND,0,0,0);
  ++index;
  label = gtk_label_new(_("Default gateway:"));
  sprintf(txt, "label_gateway_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,0,1,index,1+index,GTK_EXPAND,0,0,0);
  label = gtk_label_new(NULL);
  sprintf(txt, "gateway_%s", n->iface);
  GLADE_HOOKUP_OBJECT (parent, label, txt);
  gtk_table_attach(GTK_TABLE(info_table),label,1,3,index,1+index,GTK_EXPAND,0,0,0);
  ++index;

  gtk_widget_show_all(block);

  free(txt);
  return block;
}

void edit_interface_block(GtkWidget *block, netswitch *n)
{
  char *txt = malloc(80), *comp = malloc(80);
  gboolean val;
  if (n->profiles->name) {
    sprintf(txt, _("Active profile: %s"), n->profiles->name);
    val = TRUE;
  } else {
    strcpy(txt, _("No active profile"));
    val = FALSE;
  }
  //sprintf(comp, "image_%s", n->iface);
  //gtk_widget_set_sensitive(lookup_widget(block, comp), val);
  sprintf(comp, "active_profile_%s", n->iface);
  gtk_label_set_text(GTK_LABEL(lookup_widget(block, comp)), txt);
  sprintf(comp, "profiles_%s", n->iface);
  GtkWidget *profiles = lookup_widget(block, comp);
  free(comp);
  if (!profiles) {
    return;
  }
  
  GtkListStore *model = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(profiles)));
  GtkTreeIter iter;
  gtk_list_store_clear(model);
  gtk_list_store_append(model, &iter);
  gtk_list_store_set(model, &iter, 0, _("Disable device"), -1);
  //gtk_list_store_append(model, &iter);
  //gtk_list_store_set(model, &iter, 0, _("Do not modify"), -1);
  gtk_combo_box_set_active_iter(GTK_COMBO_BOX (profiles), &iter);
  profiles_list *m1 = n->profiles;
  profiles_list *m2 = m1;
  do {
    if (m2->name) {
      gtk_list_store_append(model, &iter);
      gtk_list_store_set(model, &iter, 0, m2->name, -1);
      if (m1 == m2) gtk_combo_box_set_active_iter(GTK_COMBO_BOX (profiles), &iter);
    }
    m2 = m2->next;
  } while(m1 != m2);
  free(txt);
}

void
toggle_toolbar                         (GtkWidget       *button,
                                        gpointer	toolbar)
{
  int selected;
  if (gtk_combo_box_get_active(GTK_COMBO_BOX(button))) {
    gtk_widget_set_sensitive(GTK_WIDGET(gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar),1)),TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar),2)),TRUE);
  } else {
    gtk_widget_set_sensitive(GTK_WIDGET(gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar),1)),FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(gtk_toolbar_get_nth_item(GTK_TOOLBAR(toolbar),2)),FALSE);
  }
}

void
ajouter                                (GtkWidget       *button,
                                        char		*iface)
{
  netswitch *n = netswitch_lookup(iface);
  GtkWidget* fenetre = create_properties();
  //gtk_widget_set_name(fenetre, "ajouter");
  char *c = (char*)malloc(5);
  sprintf(c, "%s", iface);
  GtkWidget* dev = lookup_widget(fenetre, "device");
  GtkWidget* type = lookup_widget(fenetre, "type");
  gtk_entry_set_text(GTK_ENTRY(dev), c);
  if (n->type == WIRELESS) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(type), TRUE);
    interface = 1;
  } else interface= 0;
  gtk_widget_set_sensitive(dev, FALSE);
  gtk_widget_set_sensitive(type, FALSE);
  dhcp_toggle(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "bootproto")), NULL);
  resolv_toggle(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "resolv")), NULL);
  wireless_toggle(GTK_TOGGLE_BUTTON(type), NULL);
  on_netmask_change_value(GTK_SPIN_BUTTON(lookup_widget(fenetre, "netmask")), NULL);
  if (!NETSWITCH_HAS_SMTP) gtk_widget_set_sensitive(lookup_widget(fenetre, "email"), FALSE);
  if (!NETSWITCH_HAS_SERVICES) gtk_widget_set_sensitive(lookup_widget(fenetre, "services_pane"), FALSE);
  // Ugly hack because WPA is not yet supported
  if (1 || !NETSWITCH_HAS_WPA) gtk_combo_box_remove_text(GTK_COMBO_BOX(lookup_widget(fenetre, "encryption_type")), 3);
  gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(fenetre, "mode")), 0);
  gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(fenetre, "encryption_type")), 0);
  int res = gtk_dialog_run(GTK_DIALOG(fenetre));
  if (res == GTK_RESPONSE_OK) ajouter_creer(fenetre, n, iface);
  gtk_widget_destroy(fenetre);
  free(c);
}


void
supprimer                              (GtkWidget       *button,
                                        char		*iface)
{
  netswitch *n = netswitch_lookup(iface);
  char *nom=(char*)malloc(20);
  sprintf(nom, "profiles_%s", iface);
  GtkComboBox *combo = GTK_COMBO_BOX(lookup_widget(GTK_WIDGET(button), nom));
  int selected = gtk_combo_box_get_active(combo);
  /*if (selected < 1) {
    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(mainW), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, _("Select a profile"));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    return;
  }*/
  char *profile = gtk_combo_box_get_active_text(combo);
  GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(mainW), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _("Do you really want to delete %s from device %s"), profile, n->iface);
  gint result = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if (result == GTK_RESPONSE_YES) {
    netswitch_delete(n, profile);
  }
  free(nom);
}

void
appliquer                              (GtkWidget       *button,
                                        char		*iface)
{
  netswitch *n = netswitch_lookup(iface);
  gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(button), "table_netswitch"), FALSE);
  gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(button), "hbox2"), FALSE);
  gdk_window_set_cursor(mainW->window, gdk_cursor_new(GDK_WATCH));
  g_thread_create(enregistrement, n, FALSE, NULL);
}

gpointer enregistrement (gpointer user_data)
{
  netswitch *n = (netswitch*)user_data;
  char *nom = (char*)malloc(20);
  int i;
  sprintf(nom, "profiles_%s", n->iface);
  GtkComboBox *combo=GTK_COMBO_BOX(lookup_widget(mainW, nom));
  int selected = gtk_combo_box_get_active(combo);
  char *profile = gtk_combo_box_get_active_text(combo);
  if (!selected)
    netswitch_inactive(n);
  else if (selected > 1 && (n->profiles->name && strcmp(n->profiles->name, profile)) || !n->profiles->name)
    netswitch_active(n, profile);
  gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(mainW), "table_netswitch"), TRUE);
  gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(mainW), "hbox2"), TRUE);
  gdk_window_set_cursor(mainW->window, gdk_cursor_new(GDK_LEFT_PTR));
  g_idle_add(fake_update_profiles, NULL);
}

int lire_ligne(FILE* fic, char* buf, int max)
{
  int i;
  char c;
  for (i =0; i <max && (c=getc(fic))!=EOF && c!='\n'; ++i)
    buf[i]=c;
  buf[i]='\0';
  return i;
}

void update_infos() {
  g_idle_add(fake_update_profiles,NULL);
}

gboolean show_info(gpointer w);
void
maj_liste                              (GtkWidget       *widget,
                                        gpointer         user_data)
{
  mainW=widget;
  g_timeout_add(1000, show_info, widget);
  fake_update_profiles(NULL);
  check_event(update_infos);
  show_info(widget);
  /* Préparation de la liste de recherche */
  GtkTreeView *arbre = GTK_TREE_VIEW(lookup_widget(widget, "treescan"));
  GtkTreeViewColumn *col;
  col = gtk_tree_view_column_new_with_attributes(
	_("Network ID"), gtk_cell_renderer_text_new(),
   	"text", COLUMN_BSSID,NULL);
  gtk_tree_view_append_column(arbre, col);
  col = gtk_tree_view_column_new_with_attributes(
	_("Network name"), gtk_cell_renderer_text_new(),
   	"text", COLUMN_ESSID,NULL);
  gtk_tree_view_append_column(arbre, col);
  /*col = gtk_tree_view_column_new_with_attributes(
	_("Protocol"), gtk_cell_renderer_text_new(),
   	"text", COLUMN_PROTOCOL,NULL);
  gtk_tree_view_append_column(arbre, col);*/
  col = gtk_tree_view_column_new_with_attributes(
	"", gtk_cell_renderer_pixbuf_new(),
   	"pixbuf", COLUMN_MODE,NULL);
  gtk_tree_view_append_column(arbre, col);
  col = gtk_tree_view_column_new_with_attributes(
	"", gtk_cell_renderer_pixbuf_new(),
   	"stock-id", COLUMN_ENCRYPTION,NULL);
  gtk_tree_view_append_column(arbre, col);
  GtkCellRenderer *rend = GTK_CELL_RENDERER(gtk_cell_renderer_progress_new());
  g_object_set(rend, "text", "", NULL);
  col = gtk_tree_view_column_new_with_attributes(
	_("Quality"), rend,
	"value", COLUMN_LINK,NULL);
  //gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_FIXED);
  //gtk_tree_view_column_set_fixed_width(col, 50);
  gtk_tree_view_append_column(arbre, col);
  scanmodel = gtk_list_store_new(COLUMN_NUMBER,
	G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,
	G_TYPE_STRING,G_TYPE_OBJECT,G_TYPE_UINT);
  gtk_tree_view_set_model(arbre, GTK_TREE_MODEL(scanmodel));
}

gboolean fake_update_profiles(gpointer p)
{
  update_profiles();
  //show_info(mainW);
  return FALSE;
}

void update_profiles ()
{
  netswitch *m, *p, *n = netswitch_new();
  netswitch_profiles_list(n);
  netswitch_active_profiles(n);
  int first = 1;
  if (netswitches)
    first = 0;
  int j = 0;
  int is_there_wifi = 0;
  m = n;
  GtkWidget *table_netswitch = lookup_widget(mainW, "table_netswitch");
  do {
    ++j;
    char *name = malloc(40);
    sprintf(name, "netswitch_%s", m->iface);
    GtkWidget *obj = NULL;
    obj = lookup_widget(mainW, name);
    if (m->type == WIRELESS) ++is_there_wifi;
    if (!obj || GTK_WIDGET_VISIBLE(obj) == FALSE) {
      obj = create_interface_block(mainW, m);
      gtk_box_pack_start (GTK_BOX (table_netswitch), obj, TRUE, FALSE, 0);
      GLADE_HOOKUP_OBJECT(mainW, obj, name);
      gtk_widget_show(mainW);
      if (m->type == WIRELESS) rescan(NULL, lookup_widget(table_netswitch, "refresh_scan"));
    }
    edit_interface_block(obj, m);
    free(name);
    m = m->next;
  } while (m != n);
  
  if (netswitches) {
    p = netswitches;
    int ok;
    do {
      ok = 1;
      m = n;
      do {
        if (!strcmp(m->iface, p->iface)) {
          ok=0;
          break;
	}
	m = m->next;
      } while(m != n);
      if (ok) {
        char *name = malloc(40);
        sprintf(name, "netswitch_%s", p->iface);
        GtkWidget *obj = lookup_widget(mainW, name);
	if (obj) gtk_widget_destroy(obj);
	free(name);
      }
      p = p->next;
    } while (p != netswitches);
    netswitch_free(netswitches);
  }
  netswitches = n;

  if (!is_there_wifi) {
    gtk_widget_hide(lookup_widget(mainW, "recherche_box"));
    gtk_widget_hide(lookup_widget(mainW, "autoconnect"));
    if (auto_menu) gtk_widget_hide(auto_menu);
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(lookup_widget(mainW, "notebook")),FALSE);
  } else {
    gtk_widget_show(lookup_widget(mainW, "recherche_box"));
    gtk_widget_show(lookup_widget(mainW, "autoconnect"));
    if (auto_menu) gtk_widget_show(auto_menu);
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(lookup_widget(mainW, "notebook")),TRUE);
  }
}

gboolean is_selected(GtkWidget *w, char *nom)
{	
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(w, nom)));
}

G_CONST_RETURN char *get_text(GtkWidget *w, char *nom)
{
  GtkWidget *entry = lookup_widget(w, nom);
  const char *txt = gtk_entry_get_text(GTK_ENTRY(entry));
  if (txt && strlen(txt)) {
    char *t2 = malloc(1 + strlen(txt));
    strcpy(t2, txt);
    return t2;
  } else
    return NULL;
}

void
ajouter_creer                          (GtkWidget       *button, netswitch *n, char *texte)
{
  char *txt = malloc(40);
  sprintf(txt, "netswitch_%s", texte);
  GtkWidget *pop = lookup_widget(mainW, txt);
  if (!get_text(GTK_WIDGET(button), "nom") || !pop || GTK_WIDGET_VISIBLE(pop) == FALSE) {
    //g_idle_add(autoconnect_msg, NULL);
    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(mainW), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, _("Changes can't be saved"));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    free(txt);
    return;
  }
  free(txt);
  char **keys, *tmp;
  profile *profil = profile_new();
  profil->hwaddr = (char*)get_text(GTK_WIDGET(button), "hwaddr");
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(button), "bootproto")))) {
    profil->ipaddr = (char*)get_text(GTK_WIDGET(button), "ipaddr");
    profil->netmask = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), "netmask")));
    profil->broadcast = (char*)get_text(GTK_WIDGET(button), "broadcast");
    profil->gateway = (char*)get_text(GTK_WIDGET(button), "gateway");
    profil->domain = (char*)get_text(GTK_WIDGET(button), "domain");
    profil->metric = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), "metric")));
  } else {
    profil->ipaddr = NULL;
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(button), "peerdns"))))
      profil->peerdns = 0;
    else
      profil->peerdns = 1;
  }
  keys = malloc(sizeof(char*)*5);
  profil->nameservers = keys;
  tmp = (char*)get_text(GTK_WIDGET(button), "nameserver1");;
  if (tmp && strlen(tmp)) {
	  puts("1");
    keys[0] = malloc(1 + strlen(tmp));
    strcpy(keys[0], tmp);
    tmp = (char*)get_text(GTK_WIDGET(button), "nameserver2");;
    if (tmp && strlen(tmp)) {
      keys[1] = malloc(1 + strlen(tmp));
      strcpy(keys[1], tmp);
      tmp = (char*)get_text(GTK_WIDGET(button), "nameserver3");;
      if (tmp && strlen(tmp)) {
        keys[2] = malloc(1 + strlen(tmp));
        strcpy(keys[2], tmp);
        tmp = (char*)get_text(GTK_WIDGET(button), "nameserver4");;
        if (tmp && strlen(tmp)) {
          keys[3] = malloc(1 + strlen(tmp));
          strcpy(keys[3], tmp);
          keys[4] = NULL;
        } else keys[3] = NULL;
      } else keys[2] = NULL;
    } else keys[1] = NULL;
  } else keys[0] = NULL;

  if (n->type == WIRELESS) {
    tmp = (char*)get_text(GTK_WIDGET(button), "rts");
    if (tmp && strlen(tmp)) profil->rts = atoi(tmp); else profil->rts = 0;
    tmp = (char*)get_text(GTK_WIDGET(button), "frequency");
    if (tmp && strlen(tmp)) profil->freq = atoi(tmp); else profil->freq = 0;
    tmp = (char*)get_text(GTK_WIDGET(button), "frag");
    if (tmp && strlen(tmp)) profil->frag = atoi(tmp); else profil->frag = 0;
    tmp = (char*)get_text(GTK_WIDGET(button), "sensibility");
    if (tmp && strlen(tmp)) profil->sens = atoi(tmp); else profil->sens = 0;
    tmp = (char*)get_text(GTK_WIDGET(button), "channel");
    if (tmp && strlen(tmp)) profil->channel = atoi(tmp); else profil->channel = 0;
    tmp = (char*)get_text(GTK_WIDGET(button), "rate");
    if (tmp && strlen(tmp)) profil->rate = atoi(tmp); else profil->rate = 0;
    profil->mode = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget(GTK_WIDGET(button), "mode")));
    profil->essid = (char*)get_text(GTK_WIDGET(button), "essid");
    profil->nickname = (char*)get_text(GTK_WIDGET(button), "nickname");
    profil->nwid = (char*)get_text(GTK_WIDGET(button), "nwid");

    profil->key_mode = gtk_combo_box_get_active(GTK_COMBO_BOX(lookup_widget(GTK_WIDGET(button), "encryption_type")));
    if (profil->key_mode == OFF)
      profil->keys = NULL;
    else if (profil->key_mode == OPEN || profil->key_mode == RESTRICTED) {
      char **keys = malloc(sizeof(char*)*5);
      profil->keys = keys;
      char *tmp = (char*)get_text(GTK_WIDGET(button), "key1");;
      if (tmp && strlen(tmp)) {
        keys[0] = malloc(1 + strlen(tmp));
        strcpy(keys[0], tmp);
        profil->key_mode = OPEN;
        tmp = (char*)get_text(GTK_WIDGET(button), "key2");;
        if (tmp && strlen(tmp)) {
          keys[1] = malloc(1 + strlen(tmp));
          strcpy(keys[1], tmp);
          tmp = (char*)get_text(GTK_WIDGET(button), "key3");;
          if (tmp && strlen(tmp)) {
            keys[2] = malloc(1 + strlen(tmp));
            strcpy(keys[2], tmp);
            tmp = (char*)get_text(GTK_WIDGET(button), "key4");;
            if (tmp && strlen(tmp)) {
              keys[3] = malloc(1 + strlen(tmp));
              strcpy(keys[3], tmp);
              keys[4] = NULL;
            } else keys[3] = NULL;
          } else keys[2] = NULL;
        } else keys[1] = NULL;
        int val=1;
        if (is_selected(GTK_WIDGET(button), "cle1")) val=1;
        else if (is_selected(GTK_WIDGET(button), "cle2")) val=2;
        else if (is_selected(GTK_WIDGET(button), "cle3")) val=3;
        else if (is_selected(GTK_WIDGET(button), "cle4")) val=4;
        profil->sendkey = val;
      } else keys[0] = NULL;
    } // Ajouter le WPA
  }
  if (GTK_WIDGET_IS_SENSITIVE(lookup_widget(GTK_WIDGET(button), "email"))) {
    profil->smtp_host = (char*)get_text(GTK_WIDGET(button), "smtp_server");
    profil->smtp_port = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), "smtp_port")));
    profil->smtp_domain = (char*)get_text(GTK_WIDGET(button), "smtp_domain");
    if (GTK_WIDGET_IS_SENSITIVE(lookup_widget(GTK_WIDGET(button), "smtp_user"))) {
      profil->smtp_user = (char*)get_text(GTK_WIDGET(button), "smtp_user");
      profil->smtp_pass = (char*)get_text(GTK_WIDGET(button), "smtp_pass");
    }
  }
  
  tmp = (char*)get_text(GTK_WIDGET(button), "nom");
  char *pos = tmp;
  while ( (pos = strchr(pos,'.')) ) *pos = '_';
  char *fic = malloc(1 + strlen(tmp));
  strcpy(tmp, fic);
  if (!netswitch_install(n, profil, (char*)get_text(GTK_WIDGET(button), "nom"))) {
    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(mainW), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, _("Changes can't be saved"));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
  }
  free(fic);
  profile_free(profil);
}

char *strtolower(char *t)
{
  char *p=t;
  while (*p != '\0') {
    *p=tolower(*p);
    p++;
  }
  return t; 
}

void
modification                           (GtkWidget       *button,
                                        char		*iface)
{
  if (!iface) puts("ouille");
  netswitch *n = netswitches;
  do {
   n = n->next;
  } while(strcmp(iface, n->iface));
  char *nom=(char*)malloc(20);
  sprintf(nom, "profiles_%s", iface);
  int selected = 0;
  GtkComboBox *combo;
  if (lookup_widget(GTK_WIDGET(button), nom)) {
    combo = GTK_COMBO_BOX(lookup_widget(GTK_WIDGET(button), nom));
    selected = gtk_combo_box_get_active(combo);
  }
  /*if (selected < 1) {
    GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(mainW), GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, _("Select a profile"));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    return;
  }*/
  profile *profile = netswitch_get(n, (char*)gtk_combo_box_get_active_text(combo));

  //char *fichier = (char*)malloc(400);
  GtkWidget* fenetre = create_properties();
  GtkWidget *w;
  if (!NETSWITCH_HAS_SMTP) gtk_widget_set_sensitive(lookup_widget(fenetre, "email"), FALSE);
  if (!NETSWITCH_HAS_WPA) gtk_combo_box_remove_text(GTK_COMBO_BOX(lookup_widget(fenetre, "encryption_type")), 3);
  if (!NETSWITCH_HAS_SERVICES) gtk_widget_set_sensitive(lookup_widget(fenetre, "services_pane"), FALSE);
  gtk_entry_set_text(GTK_ENTRY(w = lookup_widget(fenetre, "nom")), (char*)gtk_combo_box_get_active_text(combo));
  gtk_widget_set_sensitive(w, FALSE);

  if (profile->hwaddr) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "hwaddr")), profile->hwaddr);
  if (profile->ipaddr) {
    gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "ipaddr")), profile->ipaddr);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "bootproto")), FALSE);
    if (profile->broadcast) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "broadcast")), profile->broadcast);
    if (profile->netmask) gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(fenetre, "netmask")), profile->netmask);
    if (profile->gateway) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "gateway")), profile->gateway);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(fenetre, "metric")), profile->metric);
  } else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "bootproto")), TRUE);
  if (!profile->peerdns)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "peerdns")), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "peerdns")), FALSE);
  char *numero;
  if (n->type == WIRELESS) {
    numero = malloc(7);
    sprintf(numero, "cle%d", profile->sendkey);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, numero)), TRUE);
    if (profile->essid) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "essid")), profile->essid);
    if (profile->nwid) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "nwid")), profile->nwid);
  
    if (profile->rts) {
      sprintf(numero, "%d", profile->rts); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "rts")), numero);
    }
    if (profile->freq) {
      sprintf(numero, "%d", profile->freq); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "frequency")), numero);
    }
    if (profile->frag) {
      sprintf(numero, "%d", profile->frag); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "frag")), numero);
    }
    if (profile->sens) {
      sprintf(numero, "%d", profile->sens); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "sensibility")), numero);
    }
    if (profile->rate) {
      sprintf(numero, "%d", profile->rate); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "rate")), numero);
    }
    if (profile->channel) {
      sprintf(numero, "%d", profile->channel); 
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "channel")), numero);
    }
    free(numero);
    gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(fenetre, "mode")), profile->mode);
    gtk_combo_box_set_active(GTK_COMBO_BOX(lookup_widget(fenetre, "encryption_type")), profile->key_mode);
    char **key = profile->keys;
    numero = malloc(12);
    int id = 1;
    while (key && *key) {
      sprintf(numero, "key%d", id++);
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, numero)), *key++);
    }
    key = profile->nameservers;
    id= 1;
    while (key && *key) {
      sprintf(numero, "nameserver%d", id++);
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, numero)), *key++);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "resolv")), TRUE);
    }
    free(numero);
  }
  if (profile->smtp_host) {
    gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "smtp_server")), profile->smtp_host);
    if (profile->smtp_domain) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "smtp_domain")), profile->smtp_domain);
    if (profile->smtp_user) {
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "smtp_user")), profile->smtp_user);
      if (profile->smtp_pass) gtk_entry_set_text(GTK_ENTRY(lookup_widget(fenetre, "smtp_pass")), profile->smtp_pass);
    }
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(fenetre, "smtp_port")), profile->smtp_port);
  }
  
  GtkWidget* dev = lookup_widget(fenetre, "device");
  GtkWidget* type = lookup_widget(fenetre, "type");
  gtk_entry_set_text(GTK_ENTRY(dev), n->iface);
  if (n->type == WIRELESS)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(type), TRUE);
  gtk_widget_set_sensitive(dev, FALSE);
  gtk_widget_set_sensitive(type, FALSE);

  dhcp_toggle(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "bootproto")), NULL);
  resolv_toggle(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "resolv")), NULL);
  toggle_user_smtp(GTK_TOGGLE_BUTTON(lookup_widget(fenetre, "requires_auth")), NULL);
  wireless_toggle(GTK_TOGGLE_BUTTON(type), NULL);
  profile_free(profile);
  int res = gtk_dialog_run(GTK_DIALOG(fenetre));
  if (res == GTK_RESPONSE_OK) ajouter_creer(fenetre, n, iface);
  gtk_widget_destroy(fenetre);
}

void
wireless_toggle                        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  GtkNotebook* notebook= GTK_NOTEBOOK(lookup_widget(GTK_WIDGET(togglebutton), "notebook1"));
  GtkWidget* page=gtk_notebook_get_nth_page(notebook, 3);
  if (gtk_toggle_button_get_active(togglebutton)) {
    gtk_widget_show(page);
  } else {
    gtk_widget_hide(page);
  }
}


void
dhcp_toggle                            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (gtk_toggle_button_get_active(togglebutton)) {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "ipaddr"), FALSE);
    //gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "hwaddr"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "netmask"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "netmask2"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "broadcast"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "gateway"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "metric"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "peerdns"), TRUE);
  } else {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "ipaddr"), TRUE);
    //gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "hwaddr"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "netmask"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "netmask2"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "broadcast"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "gateway"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "metric"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "peerdns"), FALSE);
  }
}


void
resolv_toggle                          (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (gtk_toggle_button_get_active(togglebutton)) {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver1"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver2"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver3"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver4"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "domain"), TRUE);
  } else {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver1"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver2"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver3"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "nameserver4"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "domain"), FALSE);
  }
}

char *texte=NULL;
int try;
gboolean rescan_reel                  (gpointer w)
{
  if (!texte) texte=malloc(6000);
  GtkTreeView *arbre = GTK_TREE_VIEW(lookup_widget(w, "treescan"));
  netlist* p = netswitch_scan();
  gtk_widget_set_sensitive(GTK_WIDGET(w), TRUE);
  GtkTreeIter iter;
  gtk_list_store_clear(scanmodel);
  if (p == NULL) {
    try += 4;
    gtk_list_store_append(scanmodel, &iter);
    gtk_list_store_set(scanmodel, &iter,
      COLUMN_BSSID, try < 100 ? _("Searching networks..."):
                                _("No network has been found"),
      COLUMN_ENCRYPTION, GTK_STOCK_DIALOG_QUESTION,
      COLUMN_LINK, try,
      -1);
    if (try < 97)
      return TRUE;
    else return FALSE;
  }
  netlist *q = p;
  do {
    if (p->essid) {
      gtk_list_store_append(scanmodel, &iter);
      gtk_list_store_set(scanmodel, &iter,
		      COLUMN_BSSID, p->bssid,
		      COLUMN_ESSID, p->essid,
		      COLUMN_PROTOCOL, strchr(p->protocol, ' ') ? strchr(p->protocol,' ') :p->protocol,
		      COLUMN_MODE, create_pixbuf(p->mode == MANAGED ?"icon-access.png":"icon-laptop.png"),
		      COLUMN_LINK, p->quality ? p->quality: 100 - p->link/2,
		      COLUMN_ENCRYPTION, p->encryption ? GTK_STOCK_DIALOG_AUTHENTICATION :"",
		      -1);
    }
    p = p->next;
  } while (p != q);
  return FALSE;
}

void
rescan                                 (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_set_sensitive(GTK_WIDGET(user_data), FALSE);
  try = 0;
  g_idle_add(rescan_reel, user_data);
}

void show_text_info(GtkWidget* base, char *widgetname, char *data, int flag)
{
  char *tmp = malloc(7 + strlen(widgetname));
  sprintf(tmp, "label_%s", widgetname);
  GtkWidget *val = lookup_widget(base, widgetname);
  GtkWidget *lbl = lookup_widget(base, tmp);
  if (data && flag) {
    gtk_label_set_text(GTK_LABEL(val), data);
    if (!GTK_WIDGET_VISIBLE(val)) {
      gtk_widget_show(val);
      if (tmp) gtk_widget_show(lbl);
    }
  } else if (val && GTK_WIDGET_VISIBLE(val)) {
    //gtk_label_set_text(GTK_LABEL(val), _("Disabled"));
    gtk_widget_hide(val);
    if (lbl) gtk_widget_hide(lbl);
  }
  free(tmp);
}

gboolean show_info(gpointer obj)
{
  GtkWidget *base = (GtkWidget*) obj;
  iface_info *info;
  netswitch *n = netswitches;
  char *nom = malloc(30), *txt = malloc(400);
  static GtkTooltips *tool = NULL;
  if (!tool) tool = gtk_tooltips_new();
  memset(txt, 0, 400);
  do {
    info = netswitch_info(n);
    if (info) {
      if (info->status && info->ipaddr) {
        sprintf(txt, _("%sInterface %s"), txt, n->iface);
        if (n->profiles->name)
          sprintf(txt, _("%s (profile %s)"), txt, n->profiles->name);
        strcat(txt, "\n");
      }
      sprintf(nom, "image_%s", n->iface);
      gtk_widget_set_sensitive(lookup_widget(base, nom),
		      info->status?TRUE:FALSE);
      sprintf(nom, "mac_%s", n->iface);
      show_text_info(base, nom, info->hwaddr, 1);
      sprintf(nom, "ipaddr_%s", n->iface);
      show_text_info(base, nom, info->ipaddr, info->status);
      if (info->ipaddr && info->status)
        sprintf(txt, _("%sIP address: %s\n"), txt, info->ipaddr);
      sprintf(nom, "gateway_%s", n->iface);
      show_text_info(base, nom, info->ipgateway, info->status);
      if (info->ipgateway && info->status)
        sprintf(txt, _("%sDefault gateway: %s\n"), txt, info->ipgateway);
      sprintf(nom, "net_%s", n->iface);
      show_text_info(base, nom, info->ipnet, info->status);
      //sprintf(nom, "table_%s", n->iface);
      //gtk_widget_show(lookup_widget(base, nom));
      if (n->type == WIRELESS) {
        sprintf(nom, "essid_%s", n->iface);
        show_text_info(base, nom, info->wifiessid, info->status);
        sprintf(nom, "link_%s", n->iface);
	if (info->status && info->wifiessid) {
          if (info->status) sprintf(txt, _("%sESSID: %s\n"), txt, info->wifiessid);
          gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(lookup_widget(base, nom)), (float)(info->wifilink)/100);
          gtk_widget_show(lookup_widget(base, nom));
	} else
          gtk_widget_hide(lookup_widget(base, nom));
      }
    }
    if (info) iface_info_free(info);
    n = n->next;
  } while (n != netswitches);
  if (strlen(txt) == 0)
    strcpy(txt, _("Netswitch: No network"));
  else
    txt[strlen(txt)-1]='\0';
  if (boite) gtk_tooltips_set_tip(tool, boite, txt, NULL);
  free(txt);
  return TRUE;
}

void
on_btn_about_clicked                   (GtkMenuItem       *button,
                                        gpointer         user_data)
{
  const gchar *authors[] = {
    "Olivier Rossiny <mrfreeze@tuxfamily.org>",
    NULL
  };

  const gchar *artists[] = {
    "Olivier Rossiny",
    "Jonathan Lammens (Logo)",
    NULL
  };
  
  GdkPixbuf *pixbuf=create_pixbuf("netswitch-77x77.png");
  gtk_show_about_dialog(GTK_WINDOW(mainW),
		  "name", "Netswitch",
		  "version", VERSION,
		  "copyright", "\u00A9 2004, 2006 Olivier Rossiny",
		  "website", "http://netswitch.tuxfamily.org/",
		  "comments", _("Network profiles manager"),
		  "authors", authors,
		  "artists", artists,
		  "logo", pixbuf,
		  NULL);
  g_object_unref(pixbuf);
}

void
on_actualiser_profils_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  g_idle_add(fake_update_profiles, NULL);
}

void
on_ipaddr_changed                      (GtkEditable     *editable,
                                        gpointer         user_data)
{
  GtkSpinButton *spin = GTK_SPIN_BUTTON(editable);
  GtkEntry *txt = GTK_ENTRY(user_data);
  unsigned int val = atoi(gtk_entry_get_text(txt)), min;
  if (val) {
    val = (val&0xC0)>>6;
    switch (val) {
      case 0: // D Class
        min = 1;
        break;
      case 1: // A Class
	min = 8;
	break;
      case 2: // B Class
	min = 16;
	break;
      default: // C Class
	min = 24;
    }
    gtk_spin_button_set_range(spin, min, 30);
  } else {
    gtk_entry_set_text(txt, "");
  }
}


void
on_hide_the_window_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (boite) gtk_widget_hide(mainW); else gtk_main_quit();
}

gboolean autoconnect_msg (gpointer data)
{
  GtkWidget *dialog = gtk_message_dialog_new(0, 0, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, _("No connection was established automatically") );
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  return FALSE;
}

gpointer autoconnect                   (gpointer progres)
{
  int res = netswitch_autoconnect(netswitches);
  // There is a bug here if no change appear
  //g_idle_add(res ? autoconnect_msg : fake_update_profiles, NULL);
  g_idle_add(fake_update_profiles, NULL);
  return NULL;
}

void
on_autoconnect_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (GTK_WIDGET_VISIBLE(lookup_widget(mainW, "recherche_box")) == FALSE)
    return;
  gtk_widget_set_sensitive(lookup_widget(mainW, "table_netswitch"), FALSE);
  gtk_widget_set_sensitive(lookup_widget(mainW, "hbox2"), FALSE);
  g_thread_create(autoconnect, netswitches, FALSE, NULL);
}


void
on_netmask_change_value                (GtkSpinButton   *spinbutton,
                                        gpointer         user_data)
{
  char *txt = malloc(16);
  txt[0] = '\0';
  int q=0, p, ones = gtk_spin_button_get_value_as_int(spinbutton);
  while (ones > 0) {
    ++q;
    if (strlen(txt)) strcat(txt, ".");
    if (ones > 7) {
      ones -= 8;
      p = 255;
    } else {
      if (ones == 7)
        p = 254;
      if (ones == 6)
        p = 252;
      if (ones == 5)
        p = 248;
      if (ones == 4)
        p = 240;
      if (ones == 3)
        p = 224;
      if (ones == 2)
        p = 192;
      if (ones == 1)
        p = 128;
      ones = 0;
    }
    sprintf(txt, "%s%d", txt, p);
  }
  while (q < 4) {
    strcat(txt, ".0");
    ++q;
  }
  gtk_entry_set_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(spinbutton), "netmask2")), txt);
  free(txt);
}

void
on_nom_changed                         (GtkEditable     *editable,
                                        gpointer         user_data)
{
  if (strcmp(gtk_entry_get_text(GTK_ENTRY(user_data)), _("New-Profile")))
    gtk_widget_set_sensitive(GTK_WIDGET(editable), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(editable), FALSE);
}


void
toggle_user_smtp                       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  if (gtk_toggle_button_get_active(togglebutton)) {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "smtp_user"), TRUE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "smtp_pass"), TRUE);
  } else {
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "smtp_user"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(GTK_WIDGET(togglebutton), "smtp_pass"), FALSE);
  }
}


void
on_encryption_type_changed             (GtkComboBox     *combobox,
                                        gpointer         user_data)
{
  int val = gtk_combo_box_get_active(GTK_COMBO_BOX(user_data));
  if (val == 2) val = 1;
  if (val == 3) val = 2;
  gtk_notebook_set_current_page(GTK_NOTEBOOK(combobox), val);
}

void
on_import_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}

void
on_export_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}

