/******************************************************************************\
 gnofin/cbb-import.c   $Revision: 1.5 $
 Copyright (C) 2000 Ryan Boren

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkcheckbutton.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "dialogs.h"
#include "data-if.h"
#include "cbb-import.h"

const char ERROR_NO_MEMORY[]   = N_("No memory available for processing CBB file.");
const char ERROR_INVALID_CBB[] = N_("The specified file is not a valid CBB file.");

#define MAXBUF 512

#define CBB_HEADER "# CBB"

static char *cbb_read_file           (FILE *stream, size_t size);
static char *cbb_read_line           (char *filebuf, int *eof, char *buf);
static int   cbb_find_one_of         (char ch, char *buf);
static void  cbb_define_record_types (Bankbook *book);

static inline size_t
get_file_size (FILE *file)
{
  struct stat st;
  fstat (fileno (file), &st);
  return st.st_size;
}

//-------------------------------------------------------------------------

gboolean
cbb_import (GtkWindow *win, const gchar *filename, Bankbook *book) {
  AccountInfo acc = {0};
  Account *account;

  char   linebuf[MAXBUF];
  char   scratchbuf[MAXBUF];
  char  *filebuf;
  char  *fileptr;
  FILE  *stream;
  size_t filesize;
  int    pos;
  int    eof;
  char  *ptr;
  int    i;
  int num_records = 0;

  const gchar *type_name = _("UNK");

  RecordInfo rec = {0};
  Record *record = NULL;
  int day, month, year;

  trace ("");

  stream = fopen (filename, "rt");
  if (stream == NULL) {
    dialog_error (win, _("Error importing file: %s\n[%s]"), filename, strerror(errno));
    return FALSE;
  }

  // get length of file
  filesize = get_file_size (stream);
  trace ("file size: %ld", filesize);

  // Ensure that we have a valid CBB file
  if (fread (linebuf, sizeof (char), 5, stream) != 5) {
    fclose (stream);
    dialog_error (win, _(ERROR_INVALID_CBB));
    return FALSE;
  }

  *(linebuf+5) = 0;  // NULL terminate the 5 character header
  if (strcmp (linebuf, CBB_HEADER) != 0) {
    fclose (stream);
    dialog_error (win, _(ERROR_INVALID_CBB));
    return FALSE;
  }

  rewind (stream);

  // read file into memory for higher performance in parsing
  if (!(filebuf = cbb_read_file (stream, filesize))) {
    dialog_error (win, _(ERROR_NO_MEMORY));
    return FALSE;
  }

  fileptr = filebuf;
  
  acc.name = g_basename (filename);
  acc.notes = _("Imported CBB file");
  account = if_bankbook_insert_account (book, &acc);

  cbb_define_record_types(book);

  // parse file and build new accounting set
  while (1) {
    // read in next line
    fileptr = cbb_read_line (fileptr, &eof, linebuf);

    if (eof) {
      g_free (filebuf);
      break;
    }

    trace ("line: %s", linebuf);

    // check for a zero length line
    if (strlen (linebuf) == 0)
      continue;

    // If the first character of a line is '#' then skip that line.
    if ( linebuf[0] == '#' ) {
      trace("Skipping comment line.");
      continue;
    }

    num_records++;

    ptr = linebuf;

    // Run through our eight fields.
    for (i = 0; i <= 7 ; i++) {
      // Get tab delimited field.
      pos = cbb_find_one_of ('\t', ptr);
      
      if (pos > 0)
	ptr[pos] = '\0';

      switch(i) {
      case 0:
	// First field is the date in YYYYMMDD format.

	// Get the year.
        strncpy (scratchbuf, ptr, 4);
	scratchbuf[4] = '\0';  // strncpy does not NULL terminate
        sscanf (scratchbuf, "%d", &year);
	ptr = ptr + 4;

	// And the month.
        strncpy (scratchbuf, ptr, 2);
	scratchbuf[2] = '\0';  // strncpy does not NULL terminate
        sscanf (scratchbuf, "%d", &month);
	ptr = ptr + 2;

	// And the day.
        strncpy (scratchbuf, ptr, 2);
	scratchbuf[2] = '\0';  // strncpy does not NULL terminate
        sscanf (scratchbuf, "%d", &day);
	ptr = ptr + 2;

	ptr = ptr + 1;

	trace("Date: %d %d %d", year, month, day);

	g_date_clear (&rec.date, 1);
	g_date_set_dmy (&rec.date, day, month, year);
	break;

      case 1:
	// Second field is the transaction type/check#.
	if ( pos == 0 ) {
	  trace("Trans Type: None");
	  ptr = ptr + 1;
	  break;
	}

	if (*ptr >= '1' && *ptr <= '9') {  // check/ref number
          sscanf (ptr, "%d", &rec.number); 
          trace ("Trans Type: CHK%d", rec.number);
	  type_name = _("CHK");
	} else {
	  rec.number = 0;
	  // scan for record type
	  if (strcmp (ptr, "DEP") == 0)
	    type_name = _("DEP");
	  else
	  if (strcmp (ptr, "Visa") == 0)
	    type_name = _("CC");
	  else
	  if (strcmp (ptr, "ATM") == 0)
	    type_name = _("ATM");
	  else
	  if (strcmp (ptr, "EFT") == 0)
	    type_name = _("EFT");
	  else
	  {
	    strncpy (scratchbuf, ptr, sizeof scratchbuf);
	    type_name = scratchbuf;
	  }

	  trace("Trans Type: %s", type_name);
	}

        ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;

      case 2:
	// Third field is the payee.
	rec.payee = g_strdup (ptr);

	trace("Payee: %s", rec.payee);

        ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;

      case 3:
	// Fourth field is the debit amount.
	if ( pos == 0 ) {
	  trace("Debit Amount: None");
	  ptr = ptr + 1;
	  break;
	}

	// Reverse sign on amount.
        money_parse_f (ptr, &rec.amount, '.', ',');
	rec.amount = -rec.amount;
        trace ("Debit Amount: %d", rec.amount);

        ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;

      case 4:
	// Fifth field is the credit amount.
	if ( pos == 0 ) {
	  trace("Credit Amount: None");
	  ptr = ptr + 1;
	  break;
	}

	// If we already have a debit amount then ignore credit amount.
	if ( rec.amount != 0 ) {
	  trace("Credit Amount: skipping %s", ptr);
	  ptr = ptr + ( pos + 1);
	  break;
	}

        money_parse_f (ptr, &rec.amount, '.', ','); 

	trace ("Credit Amount: %d", rec.amount);

        ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;

      case 5:
	// Sixth field is the category.
	rec.category = g_strdup (ptr);

	trace("Category: %s", rec.category);

	// Check for transfers.  Transfers are denoted by an account
	//  name within []. Ex: [Checking]
	if ( *ptr == '[' ) {
	  trace("Found a transfer...");

	  // Copy everything but leading '['.
	  strncpy (scratchbuf, (ptr + 1), sizeof scratchbuf);
	  scratchbuf[pos - 2] = '\0'; // Trim trailing ']'.
	  trace("Transfer account: %s", scratchbuf);
	  rec.linked_acc_name = g_strdup(scratchbuf);
	  rec.link_broken = 1;
	  type_name = _("TXFR");
	}

	ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;
	
      case 6:
	// Seventh field is a comment.
	if ( pos == 0 ) {
	  trace("Memo: None");
	  ptr = ptr + 1;
	  break;
	}

	rec.memo = g_strdup (ptr);

	trace("Memo: %s", rec.memo);
	
	ptr = ptr + ( pos + 1);   // Step past delimiter.
	break;
	
      case 7:
	// Eight field is cleared state.
	if (*ptr == 'x' || *ptr == 'X') {
	  rec.cleared = 1;
	  trace("Cleared: Yes");
	} else {
	  trace("Cleared: No");
	}

	break;
      }
    }
    
    rec.type = if_bankbook_get_record_type_by_name (book, type_name);

    if (rec.type == NULL)
      {
	// the type_name wasn't recognized... so create it.
	RecordTypeInfo typ = {0};
	typ.name = (gchar *) type_name;
	typ.description = (gchar *) type_name;
	typ.linked = 0;
	typ.numbered = 0;
	typ.sign = RECORD_TYPE_SIGN_ANY;
	rec.type = if_bankbook_insert_record_type (book, &typ);
      }

    // exchange rate must be set
    rec.exchange_rate = 1.0;
    
    record = if_account_insert_record (account, &rec);
    trace ("record stored");

    // clear rec structure
    g_free (rec.payee);
    g_free (rec.memo);
    g_free (rec.category);
    g_free (rec.linked_acc_name);
    memset (&rec, 0, sizeof rec);
    
    type_name = _("UNK");
  }

  trace("%d records processed.", num_records);

  trace ("exit");
  return TRUE;
}

//-------------------------------------------------------------------------

static void
cbb_define_record_types (Bankbook *book)
{
  RecordTypeInfo typ = {0};

  trace ("");

  typ.name = _("ATM");
  typ.description = _("Automated Teller Machine");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("CC");
  typ.description = _("Check Card");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("CHK");
  typ.description = _("Check");
  typ.linked = 0;
  typ.numbered = 1;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("DEP");
  typ.description = ("Deposit");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("EFT");
  typ.description = ("EFT");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("TXFR");
  typ.description = ("Transfer");
  typ.linked = 1;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("UNK");
  typ.description = ("Unknown");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);
}

//-------------------------------------------------------------------------

static char *
cbb_read_file (FILE *stream, size_t filesize)
{
  char *filebuf;

  trace ("");

  filebuf = g_new0 (char, filesize);
  if (filebuf == NULL) {
    trace ("Failed to allocate memory for CBB file.");
    return NULL;
  }

  if (fread (filebuf, sizeof (char), filesize, stream) != filesize) {
    trace ("Error reading CBB file.");
    g_free (filebuf);
    return NULL;
  }

  return filebuf;
}

//-------------------------------------------------------------------------

static char *
cbb_read_line (char *filebuf, int *eof, char *linebuf)
{
  // FIXME: this function could very easily overrun the linebuf buffer

  char ch = 0;

  trace ("");

  *eof = 0;

  do {
    ch = *filebuf++;
    if ((ch != '\n') && (ch != '\r') && (ch != '\0'))
      *linebuf++ = ch;
    if (ch == '\0') {
      trace ("EOF Reached");
      *eof = 1;
      break;
    }
  } while ((ch != '\n') && (ch != '\r'));

  *linebuf = '\0';
  return filebuf;
}

//-------------------------------------------------------------------------

static int
cbb_find_one_of (char ch, char *buf)
{
  char *ptr;

  trace ("");

  ptr = strchr (buf, ch);
  trace ("ptr: %s", ptr);

  if (ptr == NULL) // didnt find ch terminator
    return 0;

  return (int) (ptr - buf);
}
