/******************************************************************************\
 gnofin/cached-string.c   $Revision: 1.3 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "common.h"
#include "cached-string.h"

CachedString *
cached_string_new (const gchar *string)
{
  CachedString *cs;

  trace ("%s", string);

  cs = g_new0 (CachedString, 1);
  cs->string = g_strdup (string ? string : "");
  cached_string_ref (cs);

  return cs;
}

CachedString *
cached_string_new_attached (const gchar *string, StringCache *cache)
{
  CachedString *cs;

  trace ("%s", string);
  g_return_val_if_fail (cache, NULL);

  cs = string_cache_find (cache, string);
  if (cs == NULL)
  {
    cs = cached_string_new (string);
    cs = cached_string_attach (cs, cache);
    cached_string_unref (cs); /* Only the cache references it */
  }
  else
    cached_string_ref (cs);
  return cs;
}

void
cached_string_ref (CachedString *cs)
{
  trace ("%p", cs);
  g_return_if_fail (cs);

  cs->ref_count++;
}

void
cached_string_unref (CachedString *cs)
{
  trace ("%p", cs);

  if (cs && (--cs->ref_count == 0))
  {
    g_assert (cs->attach_count == 0);
    g_assert (cs->parent == NULL);

    g_free (cs->string);
    g_free (cs);
  }
}

CachedString *
cached_string_copy (CachedString *cs)
{
  trace ("%p", cs);
  g_return_val_if_fail (cs, NULL);

  return cached_string_new (cs->string);
}

CachedString *
cached_string_copy_and_unref (CachedString *cs)
{
  trace ("%p", cs);
  g_return_val_if_fail (cs, NULL);

  if (cs->ref_count > 1)
  {
    CachedString *copy = cached_string_copy (cs);
    cached_string_unref (cs);
    return copy;
  }
  return cs;
}

CachedString *
cached_string_attach (CachedString *cs, StringCache *parent)
{
  trace ("%p, cache=%p", cs, parent);
  g_return_val_if_fail (cs, NULL);
  g_return_val_if_fail (parent, NULL);

  /* Quick return if already attached */
  if (cs->parent == parent)
  {
    cs->attach_count++;
    return cs;
  }

  g_return_val_if_fail (cs->parent == NULL, NULL);

  if (!parent->cache)
    parent->cache = g_list_prepend (NULL, cs);
  else
  {
    GList *it;
    gint cmp = 0;

    for (it=parent->cache; it; it=it->next)
    {
      CachedString *temp;
      
      temp = LIST_DEREF (CachedString, it);
      cmp = strcmp (temp->string, cs->string);

      /* String may already exists in cache */
      if (cmp == 0)
      {
        if (cs != temp)
	{
	  cached_string_unref (cs);
	  cached_string_ref (temp);
	}
	temp->attach_count++;
	return temp;
      }
      else if (cmp > 0)
      {
	parent->cache = g_list_first (g_list_prepend (it, cs)); 
	break;
      }
      else if (it->next == NULL)
      {
        g_list_append (it, cs);
	break;
      }
    }
  }
  cs->parent = parent;
  cs->attach_count++;
  parent->length++;

  /* The parent now references this string */
  cached_string_ref (cs);
  return cs;
}

void
cached_string_detach (CachedString *cs)
{
  trace ("%p", cs);
  g_return_if_fail (cs);

  if (cs->parent && (--cs->attach_count == 0))
  {
    cs->parent->cache = g_list_remove (cs->parent->cache, cs);
    cs->parent->length--;
    cs->parent = NULL;

    /* The parent no longer references this string */
    cached_string_unref (cs);
  }
}

void
cached_string_set_string (CachedString *cs, const gchar *string)
{
  trace ("%p \"%s\"", cs, string);
  g_return_if_fail (cs);
  g_return_if_fail (cs->parent == NULL);  /* String can't be changed once cached */

  g_free (cs->string);
  cs->string = g_strdup (string ? string : "");
}

gint
cached_string_index (CachedString * cs)
{
  g_return_val_if_fail (cs, -1);

  if (cs->parent)
    return g_list_index (cs->parent->cache, cs);
  else
    return -1;
}

CachedString *
string_cache_find (StringCache *cache, const gchar *string)
{
  GList *it;

  trace ("%p \"%s\"", cache, string);
  g_return_val_if_fail (cache, NULL);
  g_return_val_if_fail (string, NULL);

  for (it = cache->cache; it; it=it->next)
  {
    CachedString *temp;
    int cmp;
    
    temp = LIST_DEREF (CachedString, it);
    cmp = strcmp (string, temp->string);

    if (cmp == 0)
      return temp;
    else if (cmp < 0)  /* Assuming cache is sorted */
      break;
  }
  return NULL;
}

GList *
string_cache_get_strings (const StringCache *cache)
{
  GList *list = NULL;
  GList *it;

  trace ("%p", cache);
  g_return_val_if_fail (cache, NULL);

  for (it=cache->cache; it; it=it->next)
  {
    CachedString *cs = LIST_DEREF (CachedString, it);

    list = g_list_prepend (list, cs->string);
  }
  return g_list_reverse (list);
}

void
string_cache_dump (const StringCache *cache, const gchar *label)
{
  GList *it;
  gint i;

  g_print ("  %s - %p {len=%u}\n",
           label,
           cache,
	   cache->length);

  for (it=cache->cache, i=0; it; it=it->next, i++)
  {
    CachedString * cs = LIST_DEREF (CachedString, it);

    g_print ("    %03d - %p[rc=%u,ac=%u] {str=\"%s\"}\n",
             i,
	     cs,
	     cs->ref_count,
	     cs->attach_count,
	     cs->string);
  }
}

// vim: ts=8 sw=2
