/******************************************************************************\
 gnofin/data-if.h   $Revision: 1.12 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#ifndef DATA_INTERFACE_H
#define DATA_INTERFACE_H

#include "data-types.h"

GNOFIN_API_BEGIN


/*
 * The functions below constitute the primary interface to the data-structure
 * module.  There are 4 principle opaque data types used:
 *
 *	- Bankbook        contains accounts and defines record types
 *      - Account         contains records
 *      - Record          contains transaction data
 *      - RecordType      dictates how record data is interpreted
 *
 */

Bankbook    *if_bankbook_new                      (void);
void	     if_bankbook_destroy                  (Bankbook *);
const gchar *if_bankbook_get_error                (Bankbook *);
void         if_bankbook_dump                     (Bankbook *);

void         if_bankbook_enable_history           (Bankbook *, gboolean);  /* history is by default disabled */
gboolean     if_bankbook_can_undo                 (const Bankbook *);
gboolean     if_bankbook_can_redo                 (const Bankbook *);
void         if_bankbook_undo                     (Bankbook *);
void         if_bankbook_redo                     (Bankbook *);
void	     if_bankbook_clear_history            (Bankbook *);
gboolean     if_bankbook_is_dirty                 (const Bankbook *);
void         if_bankbook_reset_dirty              (Bankbook *);

void         if_bankbook_enter_batch_mode         (Bankbook *);
void         if_bankbook_leave_batch_mode         (Bankbook *);
 
Account     *if_bankbook_insert_account           (Bankbook *, const AccountInfo *);
RecordType  *if_bankbook_insert_record_type       (Bankbook *, const RecordTypeInfo *);
void         if_bankbook_insert_category          (Bankbook *, const gchar *category_name);

const GList *if_bankbook_get_accounts             (const Bankbook *);
const GList *if_bankbook_get_record_types         (const Bankbook *);
GList       *if_bankbook_get_payee_strings        (const Bankbook *);  /* the returned list must be free'd */
GList       *if_bankbook_get_memo_strings         (const Bankbook *);  /* the returned list must be free'd */
GList       *if_bankbook_get_category_strings     (const Bankbook *);  /* the returned list must be free'd */
GList       *if_bankbook_get_linked_acc_strings   (const Bankbook *, const Account *skip);  /* the returned list must be free'd */

Account     *if_bankbook_get_account_by_name      (const Bankbook *, const gchar *name);
Account     *if_bankbook_get_account_by_index     (const Bankbook *, guint index);
RecordType  *if_bankbook_get_record_type_by_name  (const Bankbook *, const gchar *name);
RecordType  *if_bankbook_get_record_type_by_index (const Bankbook *, guint index);

void         if_account_delete                    (Account *);
gboolean     if_account_set_info                  (Account *, guint mask, const AccountInfo *);
void	     if_account_get_info                  (const Account *, guint mask, AccountInfo *);
Record      *if_account_insert_record             (Account *, const RecordInfo *);
const GList *if_account_get_records               (const Account *);
gboolean     if_account_sort_records              (Account *, guint sort_field, gboolean sort_rev);
Record      *if_account_get_record_by_index       (const Account *, guint index);
gint         if_account_get_index	          (const Account *);
Bankbook    *if_account_get_parent                (const Account *);
const gchar *if_account_get_name		  (const Account *);
guint        if_account_get_last_number           (const Account *, const RecordType *);
gboolean     if_account_is_foreign                (const Account *);

void         if_record_delete                     (Record *);
gboolean     if_record_set_info                   (Record *, guint mask, const RecordInfo *);
void	     if_record_get_info	                  (const Record *, guint mask, RecordInfo *);
RecordType  *if_record_get_type                   (const Record *);
gint         if_record_get_index                  (const Record *);
Account     *if_record_get_parent                 (const Record *);

void         if_record_type_delete                (RecordType *);
gboolean     if_record_type_set_info              (RecordType *, guint mask, const RecordTypeInfo *);
void	     if_record_type_get_info              (const RecordType *, guint mask, RecordTypeInfo *);
gint         if_record_type_get_index             (const RecordType *);
Bankbook    *if_record_type_get_parent            (const RecordType *);
const gchar *if_record_type_get_name              (const RecordType *);
gboolean     if_record_type_is_numbered           (const RecordType *);
gboolean     if_record_type_is_linked             (const RecordType *);

guint        if_account_info_copy                 (AccountInfo *dest, const AccountInfo *src, guint mask);
void         if_account_info_clear                (AccountInfo *, guint mask);
gboolean     if_account_info_diff                 (const AccountInfo *, const AccountInfo *, guint mask);

guint        if_record_info_copy                  (RecordInfo *dest, const RecordInfo *src, guint mask);
void         if_record_info_clear                 (RecordInfo *, guint mask);
gboolean     if_record_info_diff                  (const RecordInfo *, const RecordInfo *, guint mask);

guint        if_record_type_info_copy             (RecordTypeInfo *dest, const RecordTypeInfo *src, guint mask);
void         if_record_type_info_clear            (RecordTypeInfo *, guint mask);
gboolean     if_record_type_info_diff             (const RecordTypeInfo *, const RecordTypeInfo *, guint mask);

GNOFIN_API_END

#endif

// vim: ts=8 sw=2
