/******************************************************************************\
 gnofin/fin-io.c   $Revision: 1.5 $
 Copyright (C) 1999-2000 Darin Fisher
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

/*
 * FIN! i/o: Implementation of old Gnofin file format.
 *
 * Author:
 *   Darin Fisher (dfisher@jagger.me.berkeley.edu)
 */

#include "common.h"
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include "file-filter.h"
#include "dialogs.h"
#include "data-if.h"
#include "fin-io.h"


/* various error messages */
#define ERROR_SYSTEM  -1
#define ERROR_CORRUPT  1
#define ERROR_FORMAT   2
#define ERROR_VERSION  3

static const gchar *
fin_io_strerror (int err)
{
  switch (err)
  {
  case ERROR_SYSTEM:
    return strerror (errno);
  case ERROR_CORRUPT:
    return _("Invalid or corrupted data file");
  case ERROR_FORMAT:
    return _("Invalid file format");
  case ERROR_VERSION:
    return _("Unsupported file version");
  }
  return _("Unknown");
}

/* FIN ascii data file format:
 * 
 * header_line (optional, starts with '#')
 * tag
 * version
 * width height
 * colsp1 colsp2 ... colsp6
 * num_infos
 * info[1]
 * ...
 * info[num_infos]
 * num_accounts
 * ...
 * account[i].name
 * account[i].info
 * account[i].num_records
 * account[i].record[i]
 * ...
 * account[i].record[num_records]
 * ...
 */

/* version notes:
 *
 * 0.3 -- amount field stored as a float
 * 0.4 -- amount field stored as a signed integer (money_t)
 */

#define FIN_DATAFILE_TAG	"FIN!"
#define FIN_MAX_LINE		1024

#define set_fail(reason) \
  G_STMT_START { \
    trace("Error: %s", fin_io_strerror(reason)); \
    result = reason; \
  } G_STMT_END

#define read_line(file, buf) \
  G_STMT_START { \
    if (fgets((buf), FIN_MAX_LINE, (file)) == NULL) \
      { set_fail(ERROR_CORRUPT); goto parse_error; } \
    (buf)[strlen(buf)-1]='\0'; \
  } G_STMT_END

#define read_int(file, buf, i) \
  G_STMT_START { \
    read_line((file), buf); \
    trace("<read_int> read \"%s\"", buf); \
    if (sscanf(buf, "%d", (i)) != 1) \
      { set_fail(ERROR_CORRUPT); goto parse_error; } \
  } G_STMT_END


/******************************************************************************
 * Emulation support
 */

static gchar *
remove_line_breaks (gchar *buf)
{
  gchar *p;

  trace ("");
  g_return_val_if_fail (buf, NULL);

  while ((p = strchr (buf, '\n')) != NULL)
    *p = ' ';
  return buf;
}

enum
{
  FIN_RECORD_TYPE_ATM,  /* withdrawal via ATM */
  FIN_RECORD_TYPE_CC,   /* withdrawal via cash card */
  FIN_RECORD_TYPE_CHK,  /* withdrawal via check */
  FIN_RECORD_TYPE_DEP,  /* deposit via check/cash */
  FIN_RECORD_TYPE_EFT,  /* deposit via electronic funds transfer */
  FIN_RECORD_TYPE_XFR,  /* transfer between accounts */
  FIN_RECORD_TYPE_NUM   /* number of unique record types */
};

enum  /* Version 0.6.1 added the FEE type which unfortunately 
       * was inserted in the middle of the list of types instead
       * of at the end, thereby making it difficult to distinguish
       * XFR's from FEE's.  Why wasn't the file version incremented? */
{
  FIN_061_RECORD_TYPE_ATM,  /* withdrawal via ATM */
  FIN_061_RECORD_TYPE_CC,   /* withdrawal via cash card */
  FIN_061_RECORD_TYPE_CHK,  /* withdrawal via check */
  FIN_061_RECORD_TYPE_DEP,  /* deposit via check/cash */
  FIN_061_RECORD_TYPE_EFT,  /* deposit via electronic funds transfer */
  FIN_061_RECORD_TYPE_FEE,  /* misc charges like bank fees */
  FIN_061_RECORD_TYPE_XFR,  /* transfer between accounts */
  FIN_061_RECORD_TYPE_NUM   /* number of unique record types */
};

static RecordType *
match_record_type_fwd (const Bankbook *book, guint type, gboolean v061)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  if (v061)
  {
    switch (type)
    {
    case FIN_061_RECORD_TYPE_ATM:
      return if_bankbook_get_record_type_by_name (book, "ATM");
    case FIN_061_RECORD_TYPE_CC:
      return if_bankbook_get_record_type_by_name (book, "CC");
    case FIN_061_RECORD_TYPE_CHK:
      return if_bankbook_get_record_type_by_name (book, "CHK");
    case FIN_061_RECORD_TYPE_DEP:
      return if_bankbook_get_record_type_by_name (book, "DEP");
    case FIN_061_RECORD_TYPE_EFT:
      return if_bankbook_get_record_type_by_name (book, "EFT");
    case FIN_061_RECORD_TYPE_FEE:
      return if_bankbook_get_record_type_by_name (book, "FEE");
    case FIN_061_RECORD_TYPE_XFR:
      return if_bankbook_get_record_type_by_name (book, "XFR");
    };
  }
  else
  {
    switch (type)
    {
    case FIN_RECORD_TYPE_ATM:
      return if_bankbook_get_record_type_by_name (book, "ATM");
    case FIN_RECORD_TYPE_CC:
      return if_bankbook_get_record_type_by_name (book, "CC");
    case FIN_RECORD_TYPE_CHK:
      return if_bankbook_get_record_type_by_name (book, "CHK");
    case FIN_RECORD_TYPE_DEP:
      return if_bankbook_get_record_type_by_name (book, "DEP");
    case FIN_RECORD_TYPE_EFT:
      return if_bankbook_get_record_type_by_name (book, "EFT");
    case FIN_RECORD_TYPE_XFR:
      return if_bankbook_get_record_type_by_name (book, "XFR");
    };
  }
  return NULL;
}

static guint
match_record_type_rev (const Bankbook *book, const RecordType *type)
{
  RecordTypeInfo info;

  trace ("");
  g_return_val_if_fail (book, 0);
  g_return_val_if_fail (type, 0);

  if_record_type_get_info (type, 0, &info);

  if (info.numbered)
    return FIN_RECORD_TYPE_CHK;
  if (info.linked)
    return FIN_RECORD_TYPE_XFR;
  if (info.sign == RECORD_TYPE_SIGN_POS)
    return FIN_RECORD_TYPE_DEP;
  if (info.sign == RECORD_TYPE_SIGN_ANY)
    return FIN_RECORD_TYPE_EFT;

  /* We unfortunately cannot distinguish between the
   * other types, so we just map everything else to ATM */
  return FIN_RECORD_TYPE_ATM;
}

static void
define_record_types (Bankbook *book)
{
  RecordTypeInfo info = {0};

  trace ("");
  g_return_if_fail (book);

  /* None of the v060 and earlier versions forced 
   * a particular sign on the record types. 
   * However, in v061 the FEE and CHK type were
   * modified to force the amount to be negative. 
   *
   * Since there is no guarantee of how people have
   * used the types in previous versions, we must
   * not impose any sign restrictions when we import
   * such files.
   */

  info.name = "ATM";
  info.description = _("Automated Teller Machine Transaction");
  info.numbered = 0;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "CC";
  info.description = _("Check Card Transaction");
  info.numbered = 0;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "CHK";
  info.description = _("Check Transaction");
  info.numbered = 1;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "DEP";
  info.description = _("Deposit Transaction");
  info.numbered = 0;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "EFT";
  info.description = _("Electronic Funds Transaction");
  info.numbered = 0;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "FEE";
  info.description = _("Bank Account Fee");
  info.numbered = 0;
  info.linked = 0;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);

  info.name = "XFR";
  info.description = _("Transfer Between Accounts");
  info.numbered = 0;
  info.linked = 1;
  info.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &info);
}


/******************************************************************************
 * IO subroutines
 */

static gint
read_record (FILE     *file,
	     GSList   *info_cache,
	     Bankbook *book,
	     Account  *acct,
	     int       major,
	     int       minor)
{
  RecordInfo info = {0};
  Record *record;
  gboolean v061 = FALSE;
  gint day, month, year;
  gint type, type_d1, type_d2;
  gint info_index;
  gint cleared;
  money_t amount;

  trace ("");
  g_return_val_if_fail (file, FALSE);
  g_return_val_if_fail ((major == 0), FALSE);
  g_return_val_if_fail ((minor == 3) || (minor == 4), FALSE);

  trace ("parsing record ...");

  /* parse record */
  if (minor == 3)
  {
    float amount_f;
    int n =
    /* 		  DD MM YY TN T1 T2 II ST AM */ 
    fscanf (file, "%d %d %d %d %d %d %d %d %f",
	    &day,
	    &month,
	    &year,
	    &type,
	    &type_d1,
	    &type_d2,
	    &info_index,
	    &cleared,
	    &amount_f);
    if (n != 9)
    {
      trace ("read only %i of 9 columns !!", n);
      goto parse_error;
    }
    amount = (money_t) rint((double) (amount_f * 100.0f));
  }
  else
  {
    long amount_l;
    int n =
    /* 		  DD MM YY TN T1 T2 II ST AM */ 
    fscanf (file, "%d %d %d %d %d %d %d %d %ld",
	    &day,
	    &month,
	    &year,
	    &type,
	    &type_d1,
	    &type_d2,
	    &info_index,
	    &cleared,
	    &amount_l);
    if (n != 9)
    {
      trace ("read only %i of 9 columns !!", n);
      goto parse_error;
    }
    amount = (money_t) amount_l;

    /* Test for Gnofin 0.6.1 FEE record type */
    v061 = (((type == FIN_061_RECORD_TYPE_FEE) && (!type_d1) && (!type_d2)) ||
             (type == FIN_061_RECORD_TYPE_XFR));
  }

  if ((v061 && (type == FIN_061_RECORD_TYPE_XFR)) ||
     (!v061 && (type == FIN_RECORD_TYPE_XFR)))
  {
    /* It is important that we only insert the record if the
     * linked account already exists... otherwise it would be
     * an error. */
    guint acc_index = if_account_get_index (acct);

    if (type_d1 >= acc_index)
      return TRUE;

    info.linked_acc = if_bankbook_get_account_by_index (book, type_d1);
    if (info.linked_acc == NULL)
    {
      trace ("Nonexistent account linked by transfer: index=%d", type_d1);
      return FALSE;
    }
  }

  /* lookup corresponding info in info_cache */
  info.payee = g_slist_nth_data (info_cache, info_index);
  if (info.payee == NULL)
  {
    trace ("reference to non-existent info string !!");
    goto parse_error;
  }

  /* assign record fields */
  g_date_clear (&info.date, 1);
  trace ("month: %d  day: %d  year: %d", month, day, year);
  g_date_set_dmy(&info.date, day, month, year);

  if (!g_date_valid (&info.date))
    goto parse_error;

  info.type = match_record_type_fwd (book, type, v061);
  info.cleared = cleared;
  info.amount = amount;

  if (type == FIN_RECORD_TYPE_CHK)
    info.number = type_d1;

  record = if_account_insert_record (acct, &info);
  return (record != NULL);

parse_error:
  return FALSE;
}

static gint
write_record (FILE           *file,
	      Record         *record,
	      const Bankbook *book,
	      GList          *info_strings,
	      gboolean        v05X_compat)
{
  RecordInfo info;
  gint info_index = 0;
  guint type_d1, type_d2;
  guint type;

  trace ("");
  g_return_val_if_fail (file, FALSE);
  g_return_val_if_fail (record, FALSE);

  if_record_get_info (record, 0, &info);

  /* Locate payee string in info cache */
  info_index = g_list_index (info_strings, info.payee);

  /* Match type as best as possible, and get type data */
  type = match_record_type_rev (book, info.type);
  switch (type)
  {
  case FIN_RECORD_TYPE_XFR:
    type_d1 = if_account_get_index (info.linked_acc);
    type_d2 = if_record_get_index (info.linked_rec);
    break;
  case FIN_RECORD_TYPE_CHK:
    type_d1 = info.number;
    type_d2 = 0;
    break;
  default:
    type_d1 = 0;
    type_d2 = 0;
    break;
  }

  /* Print record */
  if (v05X_compat)
  {
    double float_amt;

    float_amt = ((double) info.amount) / 100;

    /* 		   DD MM YY TN T1 T2 II ST AM */ 
    fprintf (file, "%d %d %d %d %d %d %d %d %f\n",
	     g_date_day (&info.date),
	     g_date_month (&info.date),
	     g_date_year (&info.date),
	     type,
	     type_d1,
	     type_d2,
	     info_index,
	     info.cleared ? 1 : 0,
	     float_amt);
  }
  else
  {
    long long_amt = (long) info.amount;
    /* 		   DD MM YY TN T1 T2 II ST AM */ 
    fprintf (file, "%d %d %d %d %d %d %d %d %ld\n",
	     g_date_day (&info.date),
	     g_date_month (&info.date),
	     g_date_year (&info.date),
	     type,
	     type_d1,
	     type_d2,
	     info_index,
	     info.cleared ? 1 : 0,
	     long_amt);
  }

  return TRUE;
}

static int
read_header (FILE            * file,
	     gchar	     * linebuf,
	     int	     * major,
	     int	     * minor)
{
  int result = 0;

  trace ("");

  /* read first line */
  read_line (file, linebuf);

  /* allow a shell style comment in the first line of the file */
  if (linebuf[0] == '#')
    read_line (file, linebuf);

  if (0 != strcmp (linebuf, FIN_DATAFILE_TAG))
  {
    set_fail (ERROR_FORMAT);
    goto parse_error;
  }

  read_line (file, linebuf);
  if (sscanf (linebuf, "%d.%d", major, minor) != 2)
  {
    set_fail (ERROR_FORMAT);
    goto parse_error;
  }
  if ((*major != 0) || (*minor < 3) || (*minor > 4))
  {
    set_fail (ERROR_VERSION);
    goto parse_error;
  }

parse_error:
  return result;
}
		      
static gint
read_info_cache (FILE    *file,
		 gchar   *linebuf,
		 GSList **info_cache)
{
  gint result = 0;
  gint num_strings, i;

  trace ("");

  read_int (file, linebuf, &num_strings);
  trace ("num_strings = %d", num_strings);

  /* Read in info_cache strings, prepend to list and
   * then reverse the list... more efficient this way. */
  for (i=0; i<num_strings; ++i)
  {
    read_line (file, linebuf);
    trace ("info_string = \"%s\"", linebuf);

    *info_cache = g_slist_prepend (*info_cache, g_strdup (linebuf));
  }
  *info_cache = g_slist_reverse (*info_cache);
  return result;

parse_error:
  /* Free any strings already read */
  while (*info_cache)
  {
    g_free ((*info_cache)->data);
    *info_cache = g_slist_remove_link (*info_cache, *info_cache);
  }
  return result;
}

static int
read_records (FILE     *file,
	      gchar    *linebuf,
	      GSList   *info_cache,
	      Bankbook *book,
	      Account  *acct,
	      gint	major,
	      gint	minor)
{
  gint result = 0;
  gint num_records, i;

  trace ("");

  read_int (file, linebuf, &num_records);
  trace ("num_records = %d", num_records);

  for (i=0; i<num_records; ++i)
  {
    if (!read_record (file, info_cache, book, acct, major, minor))
    {
      set_fail (ERROR_CORRUPT);
      goto parse_error;
    }
  }

  /* this is only necessary if we have non-empty accounts...
   * fscanf does not read the end-of-line character */
  if (num_records > 0)
    fgets (linebuf, FIN_MAX_LINE, file);

parse_error:
  return result;
}

static int
read_accounts (FILE     *file,
	       gchar	*linebuf,
	       Bankbook *book,
	       GSList   *info_cache,
	       gint	 major,
	       gint	 minor)
{
  gint result = 0;
  gchar linebuf2[FIN_MAX_LINE];
  int num_accounts, i;

  trace ("");
  
  read_int (file, linebuf, &num_accounts);
  trace ("num_accounts = %d", num_accounts);

  for (i=0; i<num_accounts; ++i)
  {
    AccountInfo info = {0};
    Account *acct;

    trace ("loading account [%d]", i);

    read_line (file, linebuf);  /* read account name */
    read_line (file, linebuf2); /* read account notes */

    /* Allocate account */
    info.name = linebuf;
    info.notes = linebuf2;
    acct = if_bankbook_insert_account (book, &info);

    if ((result = read_records (file, linebuf, info_cache, book, acct, major, minor)) != 0)
      goto parse_error;
  }
parse_error:
  return result;
}


/******************************************************************************
 * Probing
 */

gboolean
fin_io_probe (const gchar * filename)
{
  FILE * file;
  gchar linebuf[FIN_MAX_LINE];
  gint result, major, minor;

  trace ("%s", filename);
  g_return_val_if_fail (filename, FALSE);

  file = fopen (filename, "r");
  if (!file)
    return FALSE;

  /* Verify header */
  result = read_header (file, linebuf, &major, &minor);

  fclose (file);
  return (result == 0);
}


/******************************************************************************
 * Loading
 */

gboolean
fin_io_load (GtkWindow * parent, const gchar * filename, Bankbook * book)
{
  FILE *file;
  gchar linebuf[FIN_MAX_LINE];
  gint major, minor;
  gint result = 0;
  GSList *info_cache = NULL;

  trace ("%s", filename);
  g_return_val_if_fail (filename, FALSE);

  file = fopen (filename, "r");
  if (!file)
  {
    set_fail(ERROR_SYSTEM);
    goto parse_error;
  }

  if ((result = read_header (file, linebuf, &major, &minor)) != 0)
    goto parse_error;

  /* Since the old file format did not contain record type
   * information, we must "synthesize" appropriate record types. */
  define_record_types (book);

  /* We next extract the info_cache, to be used when generating
   * records.  A singly-linked list will do just fine. */
  if ((result = read_info_cache (file, linebuf, &info_cache)) != 0)
    goto parse_error;

  if ((result = read_accounts (file, linebuf, book, info_cache, major, minor)) != 0)
    goto parse_error;
  
  while (info_cache)
  { 
    g_free (info_cache->data);
    info_cache = g_slist_remove_link (info_cache, info_cache);
  }

  trace ("load succeeded !!");

parse_error:
  if (file)
    fclose (file);
  if (result)
    dialog_error (parent, _("Error loading file: %s\n[%s]"),
    		  filename, fin_io_strerror (result));
  return (result == 0);
}


/******************************************************************************
 * Saving
 */

static gboolean
fin_io_save (GtkWindow *parent, const gchar *filename,
	     const Bankbook *book, gboolean v05X_compat)
{
  gint result = 0;
  const GList *ac;
  GList *it, *info_strings = NULL;
  FILE *file;

  trace ("%s", filename);
  g_return_val_if_fail (filename, FALSE);
  g_return_val_if_fail (book, FALSE);

  /* Create file */
  file = fopen (filename, "wt");
  if (!file)
  {
    set_fail (ERROR_SYSTEM);
    goto end;
  }

  /* Write header */
  fprintf (file, "%s\n%s\n", FIN_DATAFILE_TAG, (v05X_compat ? "0.3" : "0.4"));

  /* Write info cache */
  info_strings = if_bankbook_get_payee_strings (book);
  fprintf (file, "%d\n", g_list_length (info_strings));
  for (it=info_strings; it; it=it->next)
    fprintf (file, "%s\n", LIST_DEREF (gchar, it));
  
  /* Write accounts */
  ac = if_bankbook_get_accounts (book);
  fprintf (file, "%d\n", g_list_length ((GList *) ac));
  for (; ac; ac=ac->next)
  {
    AccountInfo acc_info;
    Account *account = LIST_DEREF (Account, ac);
    gint i;

    if_account_get_info (account, 0, &acc_info);

    /* FIN! format requires a single line notes field */
    acc_info.notes = remove_line_breaks (acc_info.notes);

    /* Write account info */
    it = (GList *) if_account_get_records (account);
    fprintf (file, "%s\n%s\n%d\n",
	     acc_info.name,
	     acc_info.notes,
	     g_list_length (it));

    /* Write records */
    for (i=0; it; it=it->next, ++i)
    {
      Record *record = LIST_DEREF (Record, it);

      trace ("writing record [%i]", i);
      write_record (file, record, book, info_strings, v05X_compat);
    }
  }
  fclose (file);

  if (info_strings)
    g_list_free (info_strings);
  
end:
  if (result)
    dialog_error (parent, _("Error saving file: %s\n[%s]"),
		  filename, fin_io_strerror (result));

  return (result == 0); 
}


/******************************************************************************
 * Wrappers
 */

static gboolean
fin_io_save_05X (GtkWindow *p, const gchar *f, const Bankbook *b)
{
  return fin_io_save (p, f, b, TRUE);
}

static gboolean
fin_io_save_06X (GtkWindow *p, const gchar *f, const Bankbook *b)
{
  return fin_io_save (p, f, b, FALSE);
}


/******************************************************************************
 * Register fin_io_save as an export filter
 */

void
fin_io_init ()
{
  static FileFilter filt_06X = {0};
  static FileFilter filt_05X = {0};

  trace ("");

  /* We register 2 export filters...
   * one for each previous file version */
    
  filt_05X.label = _("Gnofin 0.5.X compatible");
  filt_05X.export = fin_io_save_05X;
  file_filter_register (&filt_05X);

  filt_06X.label = _("Gnofin 0.6.X compatible");
  filt_06X.export = fin_io_save_06X;
  file_filter_register (&filt_06X);
}

// vim: ts=8 sw=2
