/******************************************************************************\
 gnofin/ui-record-editor.c   $Revision: 1.26 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <glib.h>
#include <gtk/gtktable.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkarrow.h>
#include <gtk/gtklist.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktooltips.h>
#include <gdk/gdkkeysyms.h>
#include "ui-record-editor.h"
#include "gtk-auto-combo.h"
#include "record-stringizer.h"
#include "record-parser.h"
#include "data-if.h"
#include "date.h"
#include "dialogs.h"
#include "data-structure/record.h"

static GtkTableClass *parent_class;

struct _UI_RecordEditor
{
  GtkTable     table;

  Bankbook    *book;
  Account     *account;
  Record      *record;
  guint        mask;

  GList       *account_names;
  guint        num_account_names;

  money_t      last_amount;  // last money amount that we could parse
  guint        last_number;
  gchar       *last_account_name;

  GtkTooltips *tips;

  GDate        current_date;

  GtkEntry    *date;
  guint        entry_date_changed_id;
  GtkWidget   *calendar_window;
  GtkButton   *calendar_button;
  GtkCalendar *calendar;
  GtkWidget   *calendar_dialog;
  guint        calendar_month_changed_id;
  guint        calendar_day_changed_id;
  GtkCombo    *type;
  GtkEntry    *number;
  GtkCombo    *linked_acc;
  GtkCombo    *category;
  GtkCombo    *payee;
  GtkCombo    *memo;
  GtkEntry    *exrate;
  GtkEntry    *amount;
  GtkButton   *insert_button;
  GtkButton   *update_button;

  guint        enable_number         : 1;
  guint        enable_linked_acc     : 1;
  guint        enable_exrate         : 1;
  guint        block_calendar_update : 1;
  guint        have_last_amount      : 1;
};

#define UPPER_ROW    0, 1
#define LOWER_ROW    1, 2

#define TABLE_END    60

#define DATE_BEG     0
#define DATE_END     9
#define DATE_ROW     UPPER_ROW

#define TYPE_BEG     DATE_END
#define TYPE_END     14
#define TYPE_ROW     UPPER_ROW

#define PAYEE_BEG    TYPE_END
#define PAYEE_MID    48
#define PAYEE_END    54
#define PAYEE_ROW    UPPER_ROW

#define MEMO_BEG     PAYEE_BEG
#define MEMO_END     31
#define MEMO_ROW     LOWER_ROW

#define CATEGORY_BEG MEMO_END
#define CATEGORY_END PAYEE_MID
#define CATEGORY_ROW LOWER_ROW

#define NUMBER_BEG   TYPE_BEG
#define NUMBER_END   TYPE_END
#define NUMBER_ROW   LOWER_ROW

#define LINK_BEG     DATE_BEG
#define LINK_END     DATE_END
#define LINK_ROW     LOWER_ROW

#define EXRATE_BEG   PAYEE_MID
#define EXRATE_END   PAYEE_END
#define EXRATE_ROW   UPPER_ROW

#define AMOUNT_BEG   PAYEE_END
#define AMOUNT_END   TABLE_END
#define AMOUNT_ROW   UPPER_ROW

#define INSBUT_BEG   PAYEE_MID
#define INSBUT_END   PAYEE_END
#define INSBUT_ROW   LOWER_ROW

#define UPDBUT_BEG   PAYEE_END
#define UPDBUT_END   TABLE_END
#define UPDBUT_ROW   LOWER_ROW

enum {
  INSERT_RECORD,
  UPDATE_RECORD,
  LAST_SIGNAL
};
static gint signals [LAST_SIGNAL] = {0};

static void create_calendar_dialog (UI_RecordEditor *editor);

/******************************************************************************
 * Helper functions
 */

static inline void
block_entry_date_changed (UI_RecordEditor *editor)
{
  trace ("");
  gtk_signal_handler_block (GTK_OBJECT (editor->date),
  			    editor->entry_date_changed_id);
}

static inline void
unblock_entry_date_changed (UI_RecordEditor *editor)
{
  trace ("");
  gtk_signal_handler_unblock (GTK_OBJECT (editor->date),
  			      editor->entry_date_changed_id);
}

static inline void
block_calendar_date_changed (UI_RecordEditor *editor)
{
  trace ("");
  gtk_signal_handler_block (GTK_OBJECT (editor->calendar),
  			    editor->calendar_month_changed_id);
  gtk_signal_handler_block (GTK_OBJECT (editor->calendar),
  			    editor->calendar_day_changed_id);
}

static inline void
unblock_calendar_date_changed (UI_RecordEditor *editor)
{
  trace ("");
  gtk_signal_handler_unblock (GTK_OBJECT (editor->calendar),
  			      editor->calendar_month_changed_id);
  gtk_signal_handler_unblock (GTK_OBJECT (editor->calendar),
  			      editor->calendar_day_changed_id);
}

static inline const RecordType *
get_current_type (UI_RecordEditor *editor)
{
  const gchar *name;

  trace ("");
  g_return_val_if_fail (editor, NULL);

  name = gtk_entry_get_text (GTK_ENTRY (editor->type->entry));
  return if_bankbook_get_record_type_by_name (editor->book, name);
}

static gboolean
build_record_info (UI_RecordEditor *editor, guint mask, RecordInfo *rec, gboolean silent)
{
  RecordTypeInfo typ = {0};
  const RecordType *type;
  gboolean result;
  GtkWindow *win = NULL;

  trace ("");
  g_return_val_if_fail (editor, FALSE);
  g_return_val_if_fail (editor->book, FALSE);
  g_return_val_if_fail (rec, FALSE);

  if (mask == 0)
    mask = RECORD_ALL_WRITABLE_FIELDS;

  if (!silent)
    win = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (editor)));

  type = get_current_type (editor);
  if_record_type_get_info (type, 0, &typ);

  if (mask & RECORD_FIELD_TYPE)
    rec->type = (RecordType *) type;

  if (mask & RECORD_FIELD_DATE)
  {
    result = parse_record_field (editor->book, RECORD_FIELD_DATE,
				 gtk_entry_get_text (editor->date), rec);
    if (!result)
    {
      if (!silent)
	dialog_error (win, _("Error parsing date field."));
      return FALSE;
    }
  }

  if (mask & RECORD_FIELD_EXCHANGE_RATE)
  {
    result = parse_record_field (editor->book, RECORD_FIELD_EXCHANGE_RATE,
				 gtk_entry_get_text (editor->exrate), rec);
    if (!result)
    {
      if (!silent)
	dialog_error (win, _("Error parsing exchange rate field."));
      return FALSE;
    }
  }

  if (mask & RECORD_FIELD_AMOUNT)
  {
    result = parse_record_field (editor->book, RECORD_FIELD_AMOUNT,
				 gtk_entry_get_text (editor->amount), rec);
    if (!result)
    {
      if (!silent)
	dialog_error (win, _("Error parsing amount field."));
      return FALSE;
    }
  }

  if (mask & RECORD_FIELD_CATEGORY)
    rec->category = gtk_entry_get_text (GTK_ENTRY (editor->category->entry));

  if (mask & RECORD_FIELD_PAYEE) 
    rec->payee = gtk_entry_get_text (GTK_ENTRY (editor->payee->entry));

  if (mask & RECORD_FIELD_MEMO)
    rec->memo = gtk_entry_get_text (GTK_ENTRY (editor->memo->entry));

  if (typ.numbered && (mask & RECORD_FIELD_NUMBER))
  {
    result = parse_record_field (editor->book, RECORD_FIELD_NUMBER,
				 gtk_entry_get_text (editor->number), rec);
    if (!result)
    {
      if (!silent)
	dialog_error (win, _("Error parsing number field."));
      return FALSE;
    }
  }
  
  if (typ.linked && (mask & RECORD_FIELD_LINKED_ACC_NAME))
  {
    gchar *name;

    name = gtk_entry_get_text (GTK_ENTRY (editor->linked_acc->entry));
    rec->linked_acc_name = name;
  }

  if (typ.linked && (mask & RECORD_FIELD_LINKED_ACC))
  {
    gchar *name;

    name = gtk_entry_get_text (GTK_ENTRY (editor->linked_acc->entry));
    rec->linked_acc = if_bankbook_get_account_by_name (editor->book, name);
  }

  return TRUE;
}

static inline Account *
get_linked_account (UI_RecordEditor *editor)
{
  RecordInfo rec;

  trace ("");

  build_record_info (editor, RECORD_FIELD_LINKED_ACC, &rec, TRUE);
  return rec.linked_acc;
}

static void
refresh_popdown_strings (GtkCombo *combo, GList *strings)
{
  gchar *text;

  trace ("");
  g_return_if_fail (combo);

  text = g_strdup (gtk_entry_get_text (GTK_ENTRY (combo->entry)));

  if (strings)
    gtk_combo_set_popdown_strings (combo, strings);
  else
    gtk_list_clear_items (GTK_LIST (combo->list), 0, -1);

  gtk_entry_set_text (GTK_ENTRY (combo->entry), text);
  g_free (text);
}

static void
show_exchange_rate (UI_RecordEditor *editor)
{
  Account *account1;
  Account *account2;
  gboolean enable_exrate;

  trace ("");

  account1 = editor->account;
  account2 = get_linked_account (editor);

  enable_exrate = (editor->enable_linked_acc &&
		  (if_account_is_foreign (account1) ||
		   (account2 ? if_account_is_foreign (account2) : FALSE) ));
    
  trace ("account1=%p, account2=%p", account1, account2);
  trace ("enable_exrate = %d", (enable_exrate == 1));

  if (enable_exrate != editor->enable_exrate)
  {
    GtkTable *table = GTK_TABLE (editor);
    GtkWidget *payee = GTK_WIDGET (editor->payee);

    gtk_widget_ref (payee);
    gtk_container_remove (GTK_CONTAINER (table), payee);

    if (enable_exrate)
    {
      gtk_table_attach_defaults (table, payee, PAYEE_BEG, PAYEE_MID, 0, 1);
      gtk_widget_set_sensitive (GTK_WIDGET (editor->exrate), TRUE);
    }
    else
    {
      gtk_table_attach_defaults (table, payee, PAYEE_BEG, PAYEE_END, 0, 1);
      gtk_widget_set_sensitive (GTK_WIDGET (editor->exrate), FALSE);
    }

    editor->enable_exrate = enable_exrate;
  }
}

static void
sync_to_current_type (UI_RecordEditor *editor, gboolean type_changed)
{
  /* Synchronize editor fields with currently selected type
   */
  const RecordType *type;
  gboolean numbered, linked;

  trace ("");
  g_return_if_fail (editor);

  if (!editor->book || !editor->account)
    return;

  type = get_current_type (editor);
  linked = if_record_type_is_linked (type);
  numbered = if_record_type_is_numbered (type);

  if (!linked)
    editor->mask &= ~RECORD_FIELD_LINKED_ACC_NAME;
  if (!numbered)
    editor->mask &= ~RECORD_FIELD_NUMBER;

  /* Update state of number entry
   */
  editor->enable_number = numbered;
  gtk_widget_set_sensitive (GTK_WIDGET (editor->number), numbered);
  if (numbered == 0)
    gtk_entry_set_text (editor->number, "");
  else
  {
    gchar *text;

    if (type_changed)  /* Type was changed to a numbered type */
    {
      gboolean incr = TRUE;

      /* We will need to increment the last_record_number of the
       * current type unless the record selected is of the current
       * type. */
      if (editor->record && (if_record_get_type (editor->record) == type))
        incr = FALSE;
      
      if (incr)
      {
        editor->last_number = if_account_get_last_number (editor->account, type);
        editor->last_number++;
      }
    }

    text = g_strdup_printf ("%d", editor->last_number);
    gtk_entry_set_text (editor->number, text);
    g_free (text);
  }

  /* Update state of linked account selector
   */
  editor->enable_linked_acc = linked;
  if (linked)
  {
    gchar *text = editor->last_account_name;
    if (text == NULL)
    {
      GtkList *list = GTK_LIST (editor->linked_acc->list);
      if (list->children)
	gtk_label_get (GTK_LABEL (GTK_BIN (list->children->data)->child), &text);
      else
        text = "";
    }
    gtk_entry_set_text (GTK_ENTRY (editor->linked_acc->entry), text);
    gtk_widget_set_sensitive (GTK_WIDGET (editor->linked_acc), TRUE);
  }
  else
  {
    gtk_entry_set_text (GTK_ENTRY (editor->linked_acc->entry), "");
    gtk_widget_set_sensitive (GTK_WIDGET (editor->linked_acc), FALSE);
  }

  /* Check to see if we need to expose/hide the exchange rate entry box
   */
  show_exchange_rate (editor);
}

static void
refresh_enabled_widgets (UI_RecordEditor *editor)
{
  gboolean enable;

  trace ("");
  g_return_if_fail (editor);

  enable = (editor->account != NULL);

  gtk_widget_set_sensitive (GTK_WIDGET (editor->date), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->calendar_button), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->type), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->number), enable && editor->enable_number);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->linked_acc), enable && editor->enable_linked_acc);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->category), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->payee), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->memo), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->amount), enable);
  gtk_widget_set_sensitive (GTK_WIDGET (editor->insert_button), enable);

  enable = (enable && (editor->record != NULL) && (editor->mask != 0));
  gtk_widget_set_sensitive (GTK_WIDGET (editor->update_button), enable);
}

static void
append_to_update_mask (UI_RecordEditor *editor, guint field)
{
  gboolean enable = FALSE;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (field);

  if (editor->record)
  {
    RecordInfo a, b;

    if_record_get_info (editor->record, field, &a);

    if (build_record_info (editor, field, &b, TRUE))
      enable = if_record_info_diff (&a, &b, field);
    else
      enable = TRUE;

    if (enable)
      editor->mask |= field;
  }
  gtk_widget_set_sensitive (GTK_WIDGET (editor->update_button), enable);
}

static gboolean
warn_if_record_already_exists (UI_RecordEditor *editor, const Account *account,
			      const Record *skip_record, const RecordInfo *rec)
{
  const GList *node;
  const Record *record;
  static gboolean hide_dialog = FALSE;

  trace ("");
  g_return_val_if_fail (editor, FALSE);
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (rec, FALSE);

  if (!hide_dialog) /* This user don't want see anymore the dialog */
   {
     for (node=if_account_get_records (account); node; node=node->next)
      {
	RecordInfo info = {0};
	record = LIST_DEREF (Record, node);
	if_record_get_info (record, RECORD_ALL_WRITABLE_FIELDS, &info);

	if ((record != skip_record) && !record_info_diff(rec,&info,RECORD_ALL_WRITABLE_FIELDS))
	 {
	   GtkWindow *parent = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (editor)));

	   gint ans = dialog_question_yes_no_with_checkbutton (parent,
	       _("The record you have entered is already in use.\n"
		 "Do you wish to continue?"),
	       _("Don't show this message again."), &hide_dialog);

	   if (ans != DIALOG_YES)
	     return TRUE;  // accept warning
	 }
      }
   }
  return FALSE;
}

static gboolean
warn_if_record_number_exists (UI_RecordEditor *editor, const Account *account,
			      const Record *skip_record, const RecordInfo *rec)
{
  const GList *node;
  Record *record;

  trace ("");
  g_return_val_if_fail (editor, FALSE);
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (rec, FALSE);
  g_return_val_if_fail (if_record_type_is_numbered (rec->type), FALSE);

  for (node=if_account_get_records (account); node; node=node->next)
  {
    RecordInfo info = {0};
    record = LIST_DEREF (Record, node);

    if_record_get_info (record, RECORD_FIELD_TYPE|RECORD_FIELD_NUMBER, &info);

    if ((record != skip_record) &&
        (info.type == rec->type) && (info.number == rec->number))
    {
      GtkWindow *parent = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (editor)));

      gint ans = dialog_question_yes_no (parent,
		    _("The record number you have entered is already in use.\n"
		      "Do you wish to continue?"));
       
      if (ans != DIALOG_YES)
        return TRUE;  // accept warning
    }
  }
  return FALSE;
}

static gboolean
warn_if_record_cleared (UI_RecordEditor *editor, const RecordInfo *rec)
{
  static gboolean hide_dialog = FALSE;

  trace ("");
  g_return_val_if_fail (editor, FALSE);
  g_return_val_if_fail (rec, FALSE);

  if (rec->cleared && !hide_dialog)
  {
    GtkWindow *parent = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (editor)));

    gint ans = dialog_question_yes_no_with_checkbutton (parent,
    		_("The record you are modifying has already been cleared.\n"
		  "Are you sure you wish continue?"),
		_("Don't show this message again."), &hide_dialog);

    if (ans != DIALOG_YES)
      return TRUE;  // accept warning
  }
  return FALSE;
}


/******************************************************************************
 * Signal handlers
 */

static void
on_focus_in_event (GtkEntry *entry, GdkEventFocus *event, GtkWidget *button)
{
  trace ("");

  if (GTK_WIDGET_IS_SENSITIVE (entry))
    gtk_entry_select_region (GTK_ENTRY (entry), 0, -1);

  if (button)
    gtk_widget_show (button);
}

static void
on_focus_out_event (GtkEntry *entry, GdkEventFocus *event, GtkWidget *button)
{
  trace ("");

  if (GTK_WIDGET_IS_SENSITIVE (entry))
    gtk_entry_select_region (GTK_ENTRY (entry), 0, 0);

  if (button)
    gtk_widget_hide (button);
}

static void
on_entry_date_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  g_return_if_fail (editor);

  if (editor->calendar && GTK_WIDGET_VISIBLE (editor->calendar)
      && (!editor->block_calendar_update))
  {
    GDate date;

    g_date_clear (&date, 1);
    g_date_set_parse (&date, gtk_entry_get_text (editor->date));

    if (g_date_valid (&date))
    {
      /* Synchronize date with calendar widget
       */
      guint day, month, year;

      day = g_date_day (&date);
      month = g_date_month (&date) - 1;
      year = g_date_year (&date);

      block_calendar_date_changed (editor);

      gtk_calendar_freeze (editor->calendar);
      gtk_calendar_select_month (editor->calendar, month, year);
      gtk_calendar_select_day (editor->calendar, day);
      gtk_calendar_thaw (editor->calendar);

      unblock_calendar_date_changed (editor);
    }
  }

  append_to_update_mask (editor, RECORD_FIELD_DATE);
}

static void
on_entry_date_key_pressed (GtkWidget *w, GdkEventKey *event, UI_RecordEditor *editor)
{
  GDate date;
  gchar *text;
  trace ("");
  g_return_if_fail (editor);

  switch (event->keyval)
   {
    case GDK_KP_Add:
      event->keyval='+';
      break;
    case GDK_KP_Multiply:
      event->keyval=']';
      break;
    case GDK_KP_Subtract:
      event->keyval='-';
      break;
    case GDK_KP_Divide:
      event->keyval='[';
      break;
   }

  switch (event->keyval)
   {
    case '+':
    case '-':
    case '[':
    case ']':
      g_date_clear (&date, 1);
      g_date_set_parse (&date, gtk_entry_get_text (editor->date));
      if (g_date_valid (&date))
       {
	 if (event->keyval=='+')
	   g_date_add_days(&date,1);
	 else if (event->keyval=='-')
	   g_date_subtract_days(&date,1);
	 else if (event->keyval==']')
	   g_date_add_months(&date,1);
	 else if (event->keyval=='[')
	   g_date_subtract_months(&date,1);

	 text = date_stringize (NULL, 0, &date);
	 editor->block_calendar_update = 1;
	 gtk_entry_set_text (editor->date, text);
	 editor->block_calendar_update = 0;
	 g_free (text);
       }
      gtk_signal_emit_stop_by_name(GTK_OBJECT(w),"key-press-event");
      break;
   }
}

static void
on_calendar_date_changed (GtkCalendar *calendar, UI_RecordEditor *editor)
{
  GDate date;
  gchar *text;
  guint year, month, day;

  trace ("");
  g_return_if_fail (calendar);
  g_return_if_fail (editor);

  gtk_calendar_get_date (calendar, &year, &month, &day);
  month++;

  /* Generate date string */
  g_date_clear (&date, 1);
  g_date_set_dmy (&date, day, month, year);

  text = date_stringize (NULL, 0, &date);

  editor->block_calendar_update = 1;
  gtk_entry_set_text (editor->date, text);
  editor->block_calendar_update = 0;

  g_free (text);
}

static void
on_show_calendar (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  g_return_if_fail (editor);

  if (editor->calendar == NULL)
    create_calendar_dialog (editor);
  
  if (GTK_WIDGET_VISIBLE (editor->calendar_dialog))
    gtk_widget_hide (editor->calendar_dialog);
  else
  {
    if (strlen (gtk_entry_get_text (editor->date)) > 0)
    {
      /* Force the calendar to be updated before it is displayed */
      on_entry_date_changed (NULL, editor);
    }
    else
    {
      /* Set the date entry to the value of the calendar, and
       * make the calendar show today's date. */
      guint day, month, year;
      GDate date;

      date_now (&date);

      day = g_date_day (&date);
      month = g_date_month (&date) - 1;
      year = g_date_year (&date);

      block_calendar_date_changed (editor);

      gtk_calendar_freeze (editor->calendar);
      gtk_calendar_select_month (editor->calendar, month, year);
      gtk_calendar_select_day (editor->calendar, day);
      gtk_calendar_thaw (editor->calendar);

      unblock_calendar_date_changed (editor);

      on_calendar_date_changed (editor->calendar, editor);
    }
    gtk_widget_show (editor->calendar_dialog);
  }
}

static void
on_hide_calendar (GtkWidget *w, GdkEvent *event, UI_RecordEditor *editor)
{
  trace ("");
  g_return_if_fail (editor);

  gtk_widget_hide (editor->calendar_dialog);
}

static void
on_record_type_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  sync_to_current_type (editor, TRUE);
  append_to_update_mask (editor, RECORD_FIELD_TYPE);
}

static void
on_category_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_CATEGORY);
}

static void
on_payee_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_PAYEE);
}

static void
on_memo_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_MEMO);
}

static void
on_exchange_rate_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_EXCHANGE_RATE);
}

static void
on_amount_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_AMOUNT);

  if (money_parse (gtk_entry_get_text (editor->amount), &editor->last_amount))
    editor->have_last_amount = 1;
  else
    editor->have_last_amount = 0;
}

static void
on_number_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_NUMBER);
}

static void
on_linked_account_changed (GtkWidget *w, UI_RecordEditor *editor)
{
  trace ("");
  append_to_update_mask (editor, RECORD_FIELD_LINKED_ACC_NAME);
  show_exchange_rate (editor);
}

static void
on_insert_record (GtkWidget *w, UI_RecordEditor *editor)
{
  RecordInfo rec = {0};

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->account);

  if (build_record_info (editor, 0, &rec, FALSE))
  {
    /* The cleared flag is always set to false when a record is inserted. */
    rec.cleared = 0;

    /* Check for a record number that is already in use */
    if (if_record_type_is_numbered (rec.type))
    {
      if (warn_if_record_number_exists (editor, editor->account, NULL, &rec))
        return;  /* User decided not to insert the record */
    }
    /* Check for a record that is already in the account */
    if (warn_if_record_already_exists(editor,editor->account,NULL, &rec))
      return; /* User decided not to insert the record */

    editor->mask = 0;

    if_account_insert_record (editor->account, &rec);
    gtk_signal_emit (GTK_OBJECT (editor), signals [INSERT_RECORD], &rec);
  }
}

static void
on_update_record (GtkWidget *w, UI_RecordEditor *editor)
{
  RecordInfo rec = {0};

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->record);
  g_return_if_fail (editor->mask);

  if (build_record_info (editor, editor->mask, &rec, FALSE))
  {
    g_assert (editor->record);

    /* Make sure we do not change the cleared field if it has already been set */
    if_record_get_info (editor->record, RECORD_FIELD_CLEARED, &rec);

    if (warn_if_record_cleared (editor, &rec))
      return; /* User accepted warning not to update an already cleared record */

    if (rec.type == NULL)
      if_record_get_info (editor->record, RECORD_FIELD_TYPE, &rec);

    /* Check for a record number that is already in use */
    if (if_record_type_is_numbered (rec.type))
    {
      if (warn_if_record_number_exists (editor, editor->account, editor->record, &rec))
        return;  /* User accepted warning not to duplicate the record number */
    }

    if_record_set_info (editor->record, editor->mask, &rec);
    gtk_signal_emit (GTK_OBJECT (editor), signals [UPDATE_RECORD], &rec);
  }
}

static void
on_date_format_changed (UI_RecordEditor *editor)
{
  GDate date;
  gchar *text;

  trace ("");

  text = gtk_entry_get_text (editor->date);
  if (date_parse (text, &date))
  {
    text = date_stringize (NULL, 0, &date);
    gtk_entry_set_text (editor->date, text);
    g_free (text);
  }
}

static void
on_money_format_changed (UI_RecordEditor *editor)
{
  trace ("");
  
  if (editor->have_last_amount)
  {
    gchar *text = money_stringize (NULL, 0, editor->last_amount);
    gtk_entry_set_text (editor->amount, text);
    g_free (text);
  }
}


/******************************************************************************
 * Calendar dialog (not modal)
 */

static void
create_calendar_dialog (UI_RecordEditor *editor)
{
  GtkWindow *dialog, *parent;
  GtkWidget *calendar;

  trace ("");

  parent = GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (editor)));
  
  /* Create dialog
   */
  dialog = GTK_WINDOW (gtk_window_new (GTK_WINDOW_DIALOG));
  if (parent)
    gtk_window_set_transient_for (dialog, parent);
  gtk_window_set_title (dialog, _("Calendar"));
  gtk_window_set_position (dialog, GTK_WIN_POS_CENTER);
  gtk_window_set_policy (dialog, 0, 0, 0);

  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
  		      GTK_SIGNAL_FUNC (on_hide_calendar), editor);
  
  editor->calendar_dialog = GTK_WIDGET (dialog);

  /* Create calendar
   */
  calendar = gtk_calendar_new ();
  gtk_calendar_display_options (GTK_CALENDAR (calendar),
  				GTK_CALENDAR_SHOW_HEADING |
				GTK_CALENDAR_SHOW_DAY_NAMES);
  gtk_container_add (GTK_CONTAINER (dialog), calendar);
  gtk_widget_show (calendar);

  editor->calendar_month_changed_id =
    gtk_signal_connect (GTK_OBJECT (calendar), "month_changed",
    			GTK_SIGNAL_FUNC (on_calendar_date_changed), editor);
  editor->calendar_day_changed_id =
    gtk_signal_connect (GTK_OBJECT (calendar), "day_selected",
    			GTK_SIGNAL_FUNC (on_calendar_date_changed), editor);

  editor->calendar = GTK_CALENDAR (calendar);
}


/******************************************************************************
 * Interface functions
 */

static void
ui_record_editor_class_init (GtkObjectClass *object_class)
{
  trace ("");

  parent_class = gtk_type_class (gtk_table_get_type ());

  signals [INSERT_RECORD] = gtk_signal_new (
  	"insert_record",
	GTK_RUN_FIRST,
	object_class->type,
	GTK_SIGNAL_OFFSET (UI_RecordEditorClass, insert_record),
	gtk_marshal_NONE__POINTER,
	GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

  signals [UPDATE_RECORD] = gtk_signal_new (
  	"update_record",
	GTK_RUN_FIRST,
	object_class->type,
	GTK_SIGNAL_OFFSET (UI_RecordEditorClass, update_record),
	gtk_marshal_NONE__POINTER,
	GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

  gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
ui_record_editor_init (UI_RecordEditor *editor)
{
  GtkTable  *table;
  GtkWidget *combo;
  GtkWidget *entry;
  GtkWidget *button;
  GtkWidget *hbox;
  GtkWidget *arrow;

  trace ("");

  editor->book = NULL;
  editor->account = NULL;
  editor->record = NULL;
  editor->mask = 0;
  editor->account_names = NULL;
  editor->num_account_names = 0;
  editor->last_number = 0;
  editor->last_account_name = NULL;
  editor->block_calendar_update = 0;

  notification_list_add (&date_format_change_listeners,
			 NOTIFICATION_FUNC (on_date_format_changed), editor);
  notification_list_add (&money_format_change_listeners,
  			 NOTIFICATION_FUNC (on_money_format_changed), editor);

  editor->tips = gtk_tooltips_new ();
  
  /* Table */
  table = GTK_TABLE (editor);
  table->homogeneous = TRUE;
  gtk_table_resize (table, 2, TABLE_END);

  /* Date: hbox containing entry next to button
   */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_table_attach_defaults (table, hbox, DATE_BEG, DATE_END, DATE_ROW);

  entry = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
  editor->date = GTK_ENTRY (entry);
  gtk_tooltips_set_tip (editor->tips, entry, _("Date"), NULL);

  {
    GDate date;
    gchar *text;

    date_now (&date);
    text = date_stringize (NULL, 0, &date);
    gtk_entry_set_text (editor->date, text);
    g_free (text);
  }

  editor->entry_date_changed_id = 
    gtk_signal_connect (GTK_OBJECT (entry), "changed",
			GTK_SIGNAL_FUNC (on_entry_date_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "key-press-event",
      GTK_SIGNAL_FUNC (on_entry_date_key_pressed), editor);

  /* Calendar popup button */
  button = gtk_button_new ();
  GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  editor->calendar_button = GTK_BUTTON (button);
  gtk_tooltips_set_tip (editor->tips, button, _("Popup Calendar"), NULL);

  arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);

  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_show_calendar), editor);

  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), button);

  /* Type */
  combo = gtk_auto_combo_new();
  entry = GTK_COMBO (combo)->entry;
  gtk_table_attach_defaults (table, combo, TYPE_BEG, TYPE_END, TYPE_ROW);
  gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
  gtk_signal_connect (GTK_OBJECT (entry), "changed",
		      GTK_SIGNAL_FUNC (on_record_type_changed), editor);
  editor->type = GTK_COMBO (combo);
  gtk_tooltips_set_tip (editor->tips, GTK_COMBO (combo)->entry, _("Type"), NULL);
  
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
                      GTK_SIGNAL_FUNC (on_focus_in_event), editor->type->button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
                      GTK_SIGNAL_FUNC (on_focus_out_event), editor->type->button);

  /* Category entry box */
  combo = gtk_auto_combo_new ();
  gtk_table_attach_defaults (table, combo, CATEGORY_BEG, CATEGORY_END, CATEGORY_ROW);
  editor->category = GTK_COMBO (combo);
  entry = editor->category->entry;
  gtk_tooltips_set_tip (editor->tips, entry, _("Category"), NULL);

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
                      GTK_SIGNAL_FUNC (on_category_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
                      GTK_SIGNAL_FUNC (on_focus_in_event), editor->category->button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
                      GTK_SIGNAL_FUNC (on_focus_out_event), editor->category->button);

  /* Payee entry box */
  combo = gtk_auto_combo_new ();
  gtk_table_attach_defaults (table, combo, PAYEE_BEG, PAYEE_END, PAYEE_ROW);
  editor->payee = GTK_COMBO (combo);
  entry = editor->payee->entry;
  gtk_tooltips_set_tip (editor->tips, entry, _("Payee/payor description"), NULL);

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_payee_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), editor->payee->button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), editor->payee->button);

  /* Memo entry box */
  combo = gtk_auto_combo_new ();
  gtk_table_attach_defaults (table, combo, MEMO_BEG, MEMO_END, MEMO_ROW);
  editor->memo = GTK_COMBO (combo);
  entry = editor->memo->entry;
  gtk_tooltips_set_tip (editor->tips, entry, _("Memo"), NULL);

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_memo_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), editor->memo->button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), editor->memo->button);

  /* Exchange rate */
  entry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (entry), "1");
  gtk_widget_set_sensitive (entry, FALSE);
  gtk_table_attach_defaults (table, entry, EXRATE_BEG, EXRATE_END, EXRATE_ROW);
  editor->exrate = GTK_ENTRY (entry);
  gtk_tooltips_set_tip (editor->tips, entry, _("Exchange Rate (multiplier)"), NULL);

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_exchange_rate_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), NULL);

  /* Amount */
  entry = gtk_entry_new ();
  gtk_table_attach_defaults (table, entry, AMOUNT_BEG, AMOUNT_END, AMOUNT_ROW);
  editor->amount = GTK_ENTRY (entry);
  gtk_tooltips_set_tip (editor->tips, entry, _("Amount"), NULL);

  {
    gchar *text = money_stringize (NULL, 0, 0);
    gtk_entry_set_text (editor->amount, text);
    g_free (text);
  }

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_amount_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), NULL);

  /* Transfer Account List */
  combo = gtk_auto_combo_new ();
  entry = GTK_COMBO (combo)->entry;
  gtk_table_attach_defaults (table, combo, LINK_BEG, LINK_END, LINK_ROW);
  editor->linked_acc = GTK_COMBO (combo);
  gtk_widget_set_sensitive (combo, FALSE);
  gtk_tooltips_set_tip (editor->tips, GTK_COMBO (combo)->entry, _("Transfer from"), NULL);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combo)->entry), "changed",
  		      GTK_SIGNAL_FUNC (on_linked_account_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), editor->linked_acc->button);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), editor->linked_acc->button);

  /* Number */
  entry = gtk_entry_new ();
  gtk_table_attach_defaults (table, entry, NUMBER_BEG, NUMBER_END, NUMBER_ROW);
  editor->number = GTK_ENTRY (entry);
  gtk_tooltips_set_tip (editor->tips, entry, _("Number"), NULL);

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_number_changed), editor);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_in_event",
  		      GTK_SIGNAL_FUNC (on_focus_in_event), NULL);
  gtk_signal_connect (GTK_OBJECT (entry), "focus_out_event",
  		      GTK_SIGNAL_FUNC (on_focus_out_event), NULL);

  /* Insert button */
  button = gtk_button_new_with_label (_("Insert"));
  gtk_table_attach_defaults (table, button, INSBUT_BEG, INSBUT_END, INSBUT_ROW);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_insert_record), editor);
  editor->insert_button = GTK_BUTTON (button);
  gtk_tooltips_set_tip (editor->tips, button, _("Insert new record"), NULL);

  /* Update button */
  button = gtk_button_new_with_label (_("Update"));
  gtk_table_attach_defaults (table, button, UPDBUT_BEG, UPDBUT_END, UPDBUT_ROW);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (on_update_record), editor);
  editor->update_button = GTK_BUTTON (button);
  gtk_tooltips_set_tip (editor->tips, button, _("Update selected record"), NULL);

  gtk_widget_show_all (GTK_WIDGET (editor));

  /* We need to hide the popdown buttons here */
  gtk_widget_hide (editor->type->button);
  gtk_widget_hide (editor->payee->button);
  gtk_widget_hide (editor->memo->button);
  gtk_widget_hide (editor->category->button);
  gtk_widget_hide (editor->linked_acc->button);
  gtk_widget_hide (GTK_WIDGET (editor->calendar_button));
}

GtkType
ui_record_editor_get_type (void)
{
  static GtkType type = 0;

  if (!type)
  {
    GtkTypeInfo info = {
      "UI_RecordEditor",
      sizeof (UI_RecordEditor),
      sizeof (UI_RecordEditorClass),
      (GtkClassInitFunc) ui_record_editor_class_init,
      (GtkObjectInitFunc) ui_record_editor_init,
      NULL,
      NULL,
      (GtkClassInitFunc) NULL
    };
    type = gtk_type_unique (gtk_table_get_type (), &info);
  }
  return type;
}

GtkWidget *
ui_record_editor_new ()
{
  trace ("");
  return GTK_WIDGET (gtk_type_new (ui_record_editor_get_type ()));
}

void
ui_record_editor_set_bankbook (UI_RecordEditor *editor, Bankbook *book)
{
  trace ("");
  g_return_if_fail (editor);

  editor->book = book;

  /* Reset
   */
  editor->account = NULL;
  editor->record = NULL;
  editor->last_number = 0;
  editor->last_account_name = NULL;

  /* Skip any refreshing, assuming it will happen later.
   * FIXME: maybe this should be changed?? */

  refresh_enabled_widgets (editor);
}

void
ui_record_editor_set_record (UI_RecordEditor *editor, Account *account, Record *record)
{
  RecordInfo rec = {0};
  gboolean account_changed;

  trace ("");
  g_return_if_fail (editor);

  account_changed = (editor->account != account);

  /* Prevent record editor from showing a residual account name in the
   * linked account entry box */
  editor->last_account_name = NULL;

  editor->account = account;
  if (account)
  {
    /* Since the account has been changed, we must refresh the list of
     * other accounts for the linked account entry box */
    ui_record_editor_refresh_linked_acc_list (editor);
  }

  editor->record = record;
  if (record)
  {
    if_record_get_info (record, 0, &rec);

    /* Set entry fields, minus number and linked account */
    {
      gchar *text;
      gint i;

      const struct
      {
	GtkEntry *entry;
	guint     field;
      }
      fields [] =
      {
	{ GTK_ENTRY (editor->date),            RECORD_FIELD_DATE          },
	{ GTK_ENTRY (editor->type->entry),     RECORD_FIELD_TYPE          },
	{ GTK_ENTRY (editor->category->entry), RECORD_FIELD_CATEGORY      },
	{ GTK_ENTRY (editor->payee->entry),    RECORD_FIELD_PAYEE         },
	{ GTK_ENTRY (editor->memo->entry),     RECORD_FIELD_MEMO          },
	{ GTK_ENTRY (editor->exrate),          RECORD_FIELD_EXCHANGE_RATE },
	{ GTK_ENTRY (editor->amount),          RECORD_FIELD_AMOUNT        }
      };

      for (i=0; i < sizeof_array (fields); i++)
      {
	text = stringize_record_field (NULL, 0, fields[i].field, &rec);
	gtk_entry_set_text (fields[i].entry, text);
	g_free (text);
      }
    }

    /* Set number and linked account (if appropriate)... */
    if (if_record_type_is_numbered (rec.type))
      editor->last_number = rec.number;

    if (if_record_type_is_linked (rec.type))
      editor->last_account_name = rec.linked_acc_name;
  }

  /* May need to reset last_number.  We only do this if the account
   * is switched via this subroutine. */
  if (account_changed)
  {
    if (!if_record_type_is_numbered (get_current_type (editor)))
      editor->last_number = 0;
  }

  sync_to_current_type (editor, FALSE);

  /* Clear change mask */
  editor->mask = 0;

  refresh_enabled_widgets (editor);
}

void
ui_record_editor_refresh_type_list (UI_RecordEditor *editor)
{
  /* Simply rebuild the type list, do not change other fields
   */
  const GList *types;
  GList *strings = NULL;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->book);

  types = if_bankbook_get_record_types (editor->book);
  for (; types; types = types->next)
  {
    RecordType *type = LIST_DEREF (RecordType, types);
    strings = g_list_prepend (strings, (gpointer) if_record_type_get_name (type));
  }
  if (strings)
  {
    strings = g_list_reverse (strings);
    gtk_combo_set_popdown_strings (editor->type, strings);
    g_list_free (strings);
  }
  else
    gtk_list_clear_items (GTK_LIST (editor->type->list), 0, -1);
}

void
ui_record_editor_refresh_category_list (UI_RecordEditor *editor)
{
  /* Simply rebuild the category list, do not change other fields
   */
  GList *strings;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->book);

  strings = if_bankbook_get_category_strings (editor->book);
  refresh_popdown_strings (editor->category, strings);
  g_list_free (strings);
}

void
ui_record_editor_refresh_payee_list (UI_RecordEditor *editor)
{
  /* Simply rebuild the payee list, do not change other fields
   */
  GList *strings;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->book);

  strings = if_bankbook_get_payee_strings (editor->book);
  refresh_popdown_strings (editor->payee, strings);
  g_list_free (strings);
}

void
ui_record_editor_refresh_memo_list (UI_RecordEditor *editor)
{
  /* Simply rebuild the memo list, do not change other fields
   */
  GList *strings;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->book);

  strings = if_bankbook_get_memo_strings (editor->book);
  refresh_popdown_strings (editor->memo, strings);
  g_list_free (strings);
}

void
ui_record_editor_refresh_linked_acc_list (UI_RecordEditor *editor)
{
  /* Simply rebuild the linked account list, do not change other fields
   */
  GList *strings;

  trace ("");
  g_return_if_fail (editor);
  g_return_if_fail (editor->book);

  strings = if_bankbook_get_linked_acc_strings (editor->book, editor->account);
  refresh_popdown_strings (editor->linked_acc, strings);
  g_list_free (strings);
}
