/* Copyright (c) 2006 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

using System;
using System.IO;
using System.Collections;
using System.Text;
using System.Net;
using Google.GData.Client;

namespace Google.GData.GoogleBase
{

    ///////////////////////////////////////////////////////////////////////
    /// <summary>Generates Uri for Google Base feeds.
    ///
    /// Most applications will use
    /// <see cref="Default">the default factory</see>, which generates
    /// Uris that are sent to <c>www.google.com</c> and will not
    /// need to create new factories.
    /// </summary>
    ///////////////////////////////////////////////////////////////////////
    public class GBaseUriFactory
    {
        private static readonly Uri DefaultBaseUri =
            new Uri("http://www.google.com/base/");
        private static GBaseUriFactory DefaultFactory =
            new GBaseUriFactory(DefaultBaseUri);


        private readonly Uri baseUri;


        ///////////////////////////////////////////////////////////////////////
        /// <summary>Gets the default factory, which connects to
        /// <c>www.google.com</c></summary>
        ///////////////////////////////////////////////////////////////////////
        public static GBaseUriFactory Default
        {
            get
            {
                return DefaultFactory;
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Creates a new factory that connects to another
        /// host, or using another protocol (such as https).
        ///
        /// The URIs generated by this factory will all be relative to
        /// the given URI. The base URI *MUST* contain <c>/base/</c>
        /// to connect to <c>/base/feeds/snippets</c>, for example.
        /// </summary>
        /// <param name="baseUri">base uri to replace the default:
        /// <c>http://www.google.com/base/</c></param>
        ///////////////////////////////////////////////////////////////////////
        public GBaseUriFactory(Uri baseUri)
        {
            this.baseUri = baseUri;
        }


        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the snippets feed.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri SnippetsFeedUri
        {
            get
            {
                return new Uri(baseUri, "feeds/snippets");
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the snippets feed for getting several entries
        /// in one request.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri SnippetsFeedBatchUri
        {
            get
            {
                return new Uri(baseUri, "feeds/snippets/batch");
            }
        }


        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the items feed for inserting and querying.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri ItemsFeedUri
        {
            get
            {
                return new Uri(baseUri, "feeds/items");
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the items feed for posting batch feeds.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri ItemsFeedBatchUri
        {
            get
            {
                return new Uri(baseUri, "feeds/items/batch");
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the locales feed.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri LocalesFeedUri
        {
            get
            {
                return new Uri(baseUri, "feeds/locales");
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the attributes feed.</summary>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri AttributesFeedUri
        {
            get
            {
                return new Uri(baseUri, "feeds/attributes");
            }
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Uri of the item types feed for a particular locale.
        /// </summary>
        /// <param name="locale">locale of the items to consider, for example:
        /// <c>en_US</c>, <c>en_GB</c> or <c>de_DE</c></param>
        /// <returns>An Uri, usually passed to GoogleBaseQuery() for setting
        /// the parameters.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri GetItemTypesFeedUri(string locale)
        {
            return new Uri(baseUri, "feeds/itemtypes/" + locale);
        }

        //////////////////////////////////////////////////////////////////////
        /// <summary>Returns the URI of the item feed of another customers
        /// (for aggregators)</summary>
        /// <param name="customerId">customer identifier</param>
        //////////////////////////////////////////////////////////////////////
        public Uri GetCustomerItemFeedUri(string customerId)
        {
            return new Uri(baseUri, "feeds/" + customerId + "/items");
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Returns the URI of a specific item in the items feed of
        /// a different customer (for aggregators).</summary>
        /// <param name="customerId">identifier of the customer who owns
        /// the item</param>
        /// <param name="id">identifier for the uri, a numerical value
        /// </param>
        /// <returns>an uri to the entry itself, usually passed to
        /// the Update or Delete service method.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri GetCustomerItemEntryUri(string customerId, string id)
        {
            return new Uri(baseUri, "feeds/" + customerId + "/items/" + id);
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Returns the URI of a specific item in the items feed.
        /// </summary>
        /// <param name="id">identifier for the uri, a numerical value
        /// </param>
        /// <returns>an uri to the entry itself, usually passed to
        /// the Update or Delete service method.</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri GetItemEntryUri(string id)
        {
            return new Uri(baseUri, "feeds/items/" + id);
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Returns the URI of a specific item in the snippets
        /// feed.</summary>
        /// <param name="id">identifier for the uri, a numerical value
        /// </param>
        /// <returns>an uri to the entry itself</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri GetSnippetEntryUri(string id)
        {
            return new Uri(baseUri, "feeds/snippets/" + id);
        }

        ///////////////////////////////////////////////////////////////////////
        /// <summary>Returns the URI of a specific item type in
        /// the item types feed</summary>
        /// <param name="locale">locale of the item type, for example:
        /// <c>en_US</c>, <c>en_GB</c> or <c>de_DE</c></param>
        /// <param name="name">name of the item type</param>
        /// <returns>an uri to the entry itself</returns>
        ///////////////////////////////////////////////////////////////////////
        public Uri GetItemTypeEntryUri(string locale, string name)
        {
            return new Uri(baseUri, "feeds/itemtypes/" + locale + "/" + name);
        }
    }

}
