/* main.cc
 *
 * Copyright (C) 2003 gnome-vfsmm Development Team
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <libgnomevfsmm.h>
#include <iostream>

int print_error (const Gnome::Vfs::exception& ex)
{
  std::cout << "Error occurred: "  << ex.what() << std::endl;
            
  return 1;
}

bool on_transfer_progress(const Gnome::Vfs::Transfer::ProgressInfo& info)
{
  bool returnState = true;
  
  if(info)
  {
    switch (info.get_status())
    {
      case Gnome::Vfs::XFER_PROGRESS_STATUS_VFSERROR:
        std::cout << "VFS Error: " << gnome_vfs_result_to_string(static_cast<GnomeVFSResult>(info.get_vfs_status())) << std::endl;
        exit(1);
        
      case Gnome::Vfs::XFER_PROGRESS_STATUS_OVERWRITE:
        std::cout << "Overwriting " << info.get_target_name() << " with " << info.get_source_name() << std::endl;
        exit(1);
        
      case Gnome::Vfs::XFER_PROGRESS_STATUS_OK:
      {
        std::cout << "Status: OK" << std::endl;
        
        switch (info.get_phase())
        {
          case Gnome::Vfs::XFER_PHASE_INITIAL:
            std::cout << "  Initial phase." << std::endl;
            break;
          case Gnome::Vfs::XFER_CHECKING_DESTINATION:
            std::cout << "  Checking destination." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_COLLECTING:
            std::cout << "  Collecting file list." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_READYTOGO:
            std::cout << "  Ready to go!" << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_OPENSOURCE:
            std::cout << "  Opening source." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_OPENTARGET:
            std::cout << "  Opening target." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_COPYING:
            std::cout << "  Copying '" << info.get_source_name() << "' to '" << info.get_target_name() <<
              "' (file " << info.get_file_index() << "/" << info.get_total_files() << ", byte " <<
              info.get_bytes_copied() << "/" << info.get_file_size() << " in file, " <<
              info.get_total_bytes_copied() << "/" << info.get_total_bytes() << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_MOVING:
            std::cout << "  Moving '" << info.get_source_name() << "' to '" << info.get_target_name() <<
              "' (file " << info.get_file_index() << "/" << info.get_total_files() << ", byte " <<
              info.get_bytes_copied() << "/" << info.get_file_size() << " in file, " <<
              info.get_total_bytes_copied() << "/" << info.get_total_bytes() << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_READSOURCE:
            std::cout << "  Reading source." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_CLOSESOURCE:
            std::cout << "  Closing source." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_CLOSETARGET:
            std::cout << "  Closing target." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_DELETESOURCE:
            std::cout << "  Deleting source." << std::endl;
            break;
          case Gnome::Vfs::XFER_PHASE_FILECOMPLETED:
            std::cout << "  Done with '" << info.get_source_name() << "' -> '" << info.get_target_name() <<
              "', going next." << std::endl;
              //TODO: What does "going next" mean here? If you mean that one file has been created and another will soon be deleted, then say so. Murray.
            break;
          case Gnome::Vfs::XFER_PHASE_COMPLETED:
            std::cout << "  Completed." << std::endl;
            break;
          default:
            std::cout << "  Unexpected phase " << info.get_phase() << std::endl;
        }
      }
      case Gnome::Vfs::XFER_PROGRESS_STATUS_DUPLICATE:
        break;
    }
  }

  return returnState;
}

int main (int argc, char** argv)
{
  //This copies one location to another.
  
  //Get the command-line arguments:
  std::string input_uri_string, output_uri_string;
  if(argc > 2 )
  {
    input_uri_string = argv[1];
    output_uri_string = argv[2];
  }
  else
  {
    std::cout << "Call this with input and output URLs, such as" << std::endl
              << "./example http://www.gnome.org file:///home/murrayc/test.txt" << std::endl;
    return 0;
  }

  //Initialize gnome-vfsmm:
  Gnome::Vfs::init();
 
  try
  {
    //Gnome::Vfs::Transfer::transfer(input_uri_string, output_uri_string); //The simpler version, without progress feedback.
    Gnome::Vfs::Transfer::transfer(input_uri_string, output_uri_string,
       Gnome::Vfs::XFER_DEFAULT,
       Gnome::Vfs::XFER_ERROR_MODE_QUERY,
       Gnome::Vfs::XFER_OVERWRITE_MODE_QUERY,
       SigC::slot(&on_transfer_progress) );
  }
  catch(const Gnome::Vfs::exception& ex)
  {
    // if the operation was not successful, print the error and abort
    return print_error(ex);
  }
  
  return 0;
}


