/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <string.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "kiss.h"

static int x1, y1, dragged;
static int ox, oy;

unsigned int mouse_x, mouse_y;

void
on_exit_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  clean_up();
  lha_close();
  gtk_main_quit();
}


gboolean
on_application_delete_event            (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  clean_up();
  lha_close();
  gtk_main_quit();
  return FALSE;
}


gboolean
on_area_expose_event                   (GtkWidget       *widget,
                                        GdkEventExpose  *event,
                                        gpointer         user_data)
{
  gint32 y, height;
  guchar *buffer;

#if 0
/* Performance measurements */

  struct timeval start_time, stop_time;
  long msecs;

  gettimeofday(&start_time, NULL);
#endif

  if (config.rgb_buf == NULL) return FALSE;

  if (global_lock == KISS_PENDING) {
    /* Clear background */
    buffer= global_start;

    while (buffer < global_start + config.row_stride ) {
      *(buffer++)= palettes[config.pal_set[view]][0];
      *(buffer++)= palettes[config.pal_set[view]][1];
      *(buffer++)= palettes[config.pal_set[view]][2];
    }
    while (buffer < global_end) {
      memcpy(buffer, global_start, config.row_stride);
      buffer+= config.row_stride;
    }

    g_slist_foreach (cells, (GFunc) cell_render, NULL);
    global_end= config.rgb_buf;
    global_start= config.rgb_buf + (config.height * config.row_stride);
  }

  y= event->area.y;
  height= (y + event->area.height <= config.height) ? event->area.height
                                     : (config.height - y);

  gdk_draw_rgb_image (widget->window, widget->style->fg_gc[widget->state],
                      0, y, config.width, height, GDK_RGB_DITHER_NORMAL,
                      config.rgb_buf + (y * config.row_stride), config.row_stride);

#if 0
  gettimeofday(&stop_time, NULL);
  msecs = (stop_time.tv_sec - start_time.tv_sec) * 1000
        + (stop_time.tv_usec - start_time.tv_usec) / 1000;

  printf("%ld\n", msecs);
#endif

  global_lock= KISS_DONE; /* Rendering done */
  return TRUE;
}


gboolean
on_area_configure_event                (GtkWidget       *widget,
                                        GdkEventConfigure *event,
                                        gpointer         user_data)
{
  return FALSE;
}


void
on_set_activate                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gchar *name;

  if (GTK_CHECK_MENU_ITEM(menuitem)->active) {
    name = gtk_widget_get_name((GtkWidget *)menuitem);
    switch_view (name[4] - '0');
  }
}

void
on_set_clicked                         (GtkButton       *button,
                                        gpointer         user_data)
{
  gchar *name;

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button))) {
    name = gtk_widget_get_name((GtkWidget *)button);
    switch_view (name[8] - '0');
  }
}

void
on_group_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gchar *name;
  if (GTK_CHECK_MENU_ITEM(menuitem)->active) {
    name = gtk_widget_get_name((GtkWidget *)menuitem);
    switch_color (name[6] - '0');
  }
}

void
on_preferences_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_widget_show (pref_dialog);
}


void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_widget_show (about);
}

gboolean
on_area_button_press_event             (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  char *status_entry;
  if (event->type != GDK_BUTTON_PRESS) return FALSE;

  /* Update global mouse info */
  mouse_x= event->x; mouse_y= event->y;

  x1= (int) event->x; y1= (int) event->y;
  config.selected= config.target= intersect(x1, y1);
  dragged= 0;

  if (config.target == NULL) {
    change_cursor(widget->window, GDK_X_CURSOR);
    return FALSE;
  } else {
    change_cursor(widget->window, GDK_HAND2);
  }

  ox = config.target->object->x[view];
  oy = config.target->object->y[view];

  status_entry= g_strdup_printf("%s #%u %s %d (%d, %d)", config.description,
             config.target->object->id, config.target->name,
             config.target->object->fix, config.target->object->x[view],
             config.target->object->y[view]);
  gnome_appbar_pop(GNOME_APPBAR(appbar));
  gnome_appbar_push(GNOME_APPBAR(appbar), status_entry);

  events(config.target->object->press);
  events(config.target->events->press);

  /* If this object is MFO then we're done */
  if (config.target->object->fix >= KISS_MFO)
    return FALSE;

  if (config.target->object->fix) {
    events(config.target->object->fixcatch);
    events(config.target->events->fixcatch);
  }

  if (config.target->object->fix == 0 || !prefs.catch_drop) {
    events(config.target->object->catch);
    events(config.target->events->catch);
  }

  if (config.target->object->fix > 0) {
    config.target->object->fix--;
    if (!config.target->object->fix) {
      events(config.target->object->unfix);
      events(config.target->events->unfix);
    }
  }

  return FALSE;
}

gboolean
on_area_motion_notify_event            (GtkWidget       *widget,
                                        GdkEventMotion  *event,
                                        gpointer         user_data)
{
  int x2, y2;

  /* Update global mouse info */
  gdk_window_get_pointer(event->window, &mouse_x, &mouse_y, NULL);

  if (config.target == NULL)
    return FALSE;

  x2= mouse_x; y2= mouse_y;

  if (config.target->object->fix >= KISS_MFO) {
    /* do nothing */
  } else if (config.target->object->fix == 0) {
    render_object(config.target->object);
    object_set_location(config.target->object, ox + x2 - x1, oy + y2 - y1);
    render_object(config.target->object);
  } else if (dragged == 0) {
    dragged= 1;
    events(config.target->object->fixdrop);
    events(config.target->events->fixdrop);
    if (!prefs.catch_drop) {
      events(config.target->object->drop);
      events(config.target->events->drop);
    }
  }

  return FALSE;
}

gboolean
on_area_button_release_event           (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  int x2, y2;

  /* Update global mouse info */
  mouse_x= event->x; mouse_y= event->y;

  change_cursor(widget->window, GDK_LEFT_PTR);

  if (config.target == NULL)
    return FALSE;

  if (config.target->object->fix == 0) {
    x2= (int) event->x; y2= (int) event->y;
    /* put object back to start location for collision check purposes */
    render_object(config.target->object);
    object_set_location(config.target->object, ox, oy);
    object_move(config.target->object, ox + x2 - x1, oy + y2 - y1);
  }

  /* just in case object fix has changed due to the movement + collisions */

  if (config.target->object->fix >= KISS_MFO) {
    /* do nothing */
  } else if (config.target->object->fix == 0) {
    events(config.target->object->drop);
    events(config.target->events->drop);
  } else if (dragged == 0) {
    events(config.target->object->fixdrop);
    events(config.target->events->fixdrop);
    if (!prefs.catch_drop) {
      events(config.target->object->drop);
      events(config.target->events->drop);
    }
  }

  events(config.target->object->release);
  events(config.target->events->release);

  config.target= NULL;
  return FALSE;
}

void
on_open_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_file_selection_complete(GTK_FILE_SELECTION(set_open), "*.lzh");
  gtk_widget_show (set_open);
}


void
on_set_open_ok                         (GtkButton       *button,
                                        gpointer         user_data)
{
  char *filename;

  gtk_widget_hide (set_open);
  filename= gtk_file_selection_get_filename(GTK_FILE_SELECTION(set_open));
  open_either(filename);
}


void
on_set_open_cancel                     (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide (set_open);
}

void
on_open_config_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_file_selection_complete(GTK_FILE_SELECTION(config_open), "*.cnf");
  gtk_widget_show (config_open);
}


void
on_config_open_ok                      (GtkButton       *button,
                                        gpointer         user_data)
{
  gchar *filename;
  gtk_widget_hide (config_open);
  filename= gtk_file_selection_get_filename(GTK_FILE_SELECTION(config_open));
  parse_file(filename);
}


void
on_config_open_cancel                  (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide (config_open);
}

void
on_close_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  events(config.end);
  clean_up();
}


void
on_open_add_on_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_file_selection_complete(GTK_FILE_SELECTION(add_open), "*.lzh");
  gtk_widget_show (add_open);
}


void
on_close_errors_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide(error_list);
}


void
on_add_open_ok                         (GtkButton       *button,
                                        gpointer         user_data)
{
  gchar *filename;
  gtk_widget_hide (add_open);
  filename= gtk_file_selection_get_filename(GTK_FILE_SELECTION(add_open));
  lha_extend(filename);
}


void
on_add_open_cancel                     (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_hide (add_open);
}

gint
on_preferences_close                   (GnomeDialog     *gnomedialog,
                                        gpointer         user_data)
{
  gtk_widget_hide(pref_dialog);
  return TRUE;
}

void
on_preferences_apply                   (GnomePropertyBox *gnomepropertybox,
                                        gint             page,
                                        gpointer         user_data)
{
  gnome_config_set_bool("Sound/midi", prefs.midi);
  gnome_config_set_bool("Sound/esound", prefs.esound);
  gnome_config_set_bool("Diagnostics/error_list", prefs.error_list);
  gnome_config_set_bool("Diagnostics/trace", prefs.trace);
  gnome_config_set_bool("Diagnostics/warnings", prefs.warnings);
  gnome_config_set_bool("Features/ignore_colon", prefs.ignore_colon);
  gnome_config_set_bool("Features/default_palette", prefs.default_palette);
  gnome_config_set_bool("Features/fkiss_case", prefs.fkiss_case);
  gnome_config_set_bool("Features/catch_drop", prefs.catch_drop);
  gnome_config_set_int("Features/speed_factor", prefs.speed_factor);
  gnome_config_set_int("Features/min_timer", prefs.min_timer);
  gnome_config_set_string("Sound/midi_player", prefs.midi_player);
  gnome_config_sync();
}

void
on_pref_midi_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.midi = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}

void
on_pref_esound_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.esound = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}

void
on_pref_error_list_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.error_list = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_trace_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.trace = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_warnings_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.warnings = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_ignore_colon_toggled           (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.ignore_colon = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_default_palette_toggled        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.default_palette = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_fkiss_case_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.fkiss_case = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_catch_drop_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  prefs.catch_drop = gtk_toggle_button_get_active(togglebutton);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_speed_factor_changed           (GtkEditable     *editable,
                                        gpointer         user_data)
{
  prefs.speed_factor = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(editable));
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}


void
on_pref_min_timer_changed              (GtkEditable     *editable,
                                        gpointer         user_data)
{
  prefs.min_timer = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(editable));
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}

void
on_pref_midi_player_changed            (GtkEditable     *editable,
                                        gpointer         user_data)
{
  g_free(prefs.midi_player);
  prefs.midi_player = gtk_editable_get_chars(editable, 0, -1);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(pref_dialog));
}

void
on_application_drag_data_received      (GtkWidget       *widget,
                                        GdkDragContext  *drag_context,
                                        gint             x,
                                        gint             y,
                                        GtkSelectionData *data,
                                        guint            info,
                                        guint            time,
                                        gpointer         user_data)
{
  guchar *filename;
  int k;

  filename= data->data;
  for (k= 0; k < data->length; ++k) {
    if (filename[k] == '\r') {
      filename[k]= '\0';
      break;
    }
  }

  if ( !strncmp(filename, "file:", 5)) {
    open_either(filename + 5);
  }
}

void
on_unfix_clicked                       (GtkButton       *button,
                                        gpointer         user_data)
{
  if (config.selected && config.selected->object->fix > 1) {
    config.selected->object->fix= 1; /* will trigger unfix() on next click */
  }
}

void
on_clear_errors_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_clist_clear(GTK_CLIST(error_clist));
}

void
on_error_list_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  gtk_widget_show(error_list);
}

void
on_unfix_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (config.selected && config.selected->object->fix > 1) {
    config.selected->object->fix= 1; /* will trigger unfix() on next click */
  }
}

