/* FlegitaGimp - Gimp Sane Plugin
 *
 * main.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gnomescanui.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/*  Local function prototypes  */
#define	PROCEDURE_SCAN	"flegita_scan"
#define PROCEDURE_SCAN_AS_LAYER	"flegita_scan_as_layer"

/* Local variables */

typedef struct {
  GtkWidget *dialog;
  GtkWidget *layer_name_field;
  gboolean is_new;
  gint32 image;
} FlegitaGimpInfo;

static void	query			(void);

static void	run			(const gchar      *name,
					 gint              nparams,
					 const GimpParam  *param,
					 gint             *nreturn_vals,
					 GimpParam       **return_vals);

void		fg_image_received	(GnomeScanContext *context,
					 GnomeScanResult *info,
					 FlegitaGimpInfo *fgi);


/*  Local variables  */

GimpPlugInInfo PLUG_IN_INFO =
  {
    NULL,  /* init_proc  */
    NULL,  /* quit_proc  */
    query, /* query_proc */
    run,   /* run_proc   */
  };

MAIN ()

static void
query (void)
{
  gchar *help_path;
  gchar *help_uri;

  static GimpParamDef scan_args[] = {
    { GIMP_PDB_INT32,    "run_mode",   "Interactive, non-interactive"    },
  };

  static GimpParamDef scan_return[] = {
    { GIMP_PDB_STATUS,   "status",     "Return status"                 },
    { GIMP_PDB_IMAGE,    "image",      "New image"                     },
  };

  static GimpParamDef scan_as_layer_args[] = {
    { GIMP_PDB_INT32,    "run_mode",   "Interactive, non-interactive"    },
    { GIMP_PDB_IMAGE,    "image",      "Input image"                     },
    { GIMP_PDB_DRAWABLE, "drawable",   "Input drawable"                  },
  };

  gimp_plugin_domain_register (GETTEXT_PACKAGE, LOCALEDIR);

  help_path = g_build_filename (DATADIR, "help", NULL);
  help_uri = g_filename_to_uri (help_path, NULL, NULL);
  g_free (help_path);

  gimp_install_procedure (PROCEDURE_SCAN,
			  _("Scan into a new image."),
			  _("Help"),
			  "Étienne Bersac <bersace03@laposte.net>",
			  "Étienne Bersac <bersace03@laposte.net>",
			  "2006",
			  _("Scan ..."),
			  NULL,
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (scan_args), G_N_ELEMENTS (scan_return),
			  scan_args, scan_return);

  gimp_plugin_menu_register (PROCEDURE_SCAN, "<Toolbox>/File/Acquire");
  gimp_plugin_icon_register (PROCEDURE_SCAN, GIMP_ICON_TYPE_IMAGE_FILE, ICONDIR "/flegita-gimp_scan.svg");

  gimp_install_procedure (PROCEDURE_SCAN_AS_LAYER,
			  _("Scan picture as new layer..."),
			  _("Help"),
			  "Étienne Bersac <bersace03@laposte.net>",
			  "Étienne Bersac <bersace03@laposte.net>",
			  "2006",
			  _("Scan as Layer..."),
			  "RGB*",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (scan_as_layer_args), G_N_ELEMENTS (scan_return),
			  scan_as_layer_args, scan_return);

  gimp_plugin_menu_register (PROCEDURE_SCAN_AS_LAYER, "<Image>/File/Open");
  gimp_plugin_icon_register (PROCEDURE_SCAN_AS_LAYER, GIMP_ICON_TYPE_IMAGE_FILE, ICONDIR "/flegita-gimp_scan-as-layer.svg");
}

static void
run (const gchar      *name,
     gint              n_params,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam   values[2];
  GimpDrawable      *drawable;
  gint32             image_ID;
  GimpRunMode        run_mode;
  GimpPDBStatusType  status = GIMP_PDB_SUCCESS;

  GnomeScanContext *context;
  GtkListStore *store;
  GtkCellRenderer *renderer;
  GtkTreeIter iter;
  gboolean is_new;
  gint n_layers;
  gdouble res, xres, yres;

  FlegitaGimpInfo *fgi = g_new0 (FlegitaGimpInfo, 1);

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  gimp_ui_init (PACKAGE, FALSE);
  gnome_scan_ui_init ();

  *nreturn_vals = 1;
  *return_vals  = values;

  run_mode = param[0].data.d_int32;

  /* if we are scanning, create a new image */
  if (gs_strcmp (name, PROCEDURE_SCAN) == 0) {
    image_ID = gimp_image_new (1, 1, GIMP_RGB);
    is_new = TRUE;
  }
  /* else use the existing one. */
  else {
    image_ID = param[1].data.d_image;
    is_new = FALSE;
    gimp_drawable_detach (gimp_drawable_get (param[2].data.d_drawable));
  }

  fgi->image = image_ID;
  fgi->is_new = is_new;

  /* INIT */

  context = gnome_scan_context_new ();

  if (is_new) {
    gimp_image_get_resolution (image_ID, &xres, &yres);
    res = MAX (xres, yres);
  }
  else {
    res = 100.;
  }

  gnome_scan_context_set_resolution (context, res);
  g_signal_connect (context,
		    "image-received",
		    G_CALLBACK (fg_image_received),
  		    fgi);

  /* apply app default value */
  fgi->dialog = gnome_scan_dialog_new (context, NULL);
  gtk_window_set_default_size (GTK_WINDOW (fgi->dialog), 300, 400);

  /* layer name */
  fgi->layer_name_field = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (fgi->layer_name_field),
		      /* translator, default new layer name */
		      _("Scanned layer"));
  gnome_scan_dialog_add_front_widget (GNOME_SCAN_DIALOG (fgi->dialog),
				      _("Layer name"),
				      fgi->layer_name_field);

  gnome_scan_dialog_run (GNOME_SCAN_DIALOG (fgi->dialog));

  /*   gtk_widget_destroy (dialog); */
  gnome_scan_context_destroy (context);

  /* if image is new, open a display */
  gimp_image_get_layers (image_ID,
			 &n_layers);
  if (is_new) {
    /* if we have a layer, then display */
    if (n_layers == 1) {
      gimp_display_new (image_ID);
    }
    /* else, the scan has been cancel */
    else {
      gimp_image_delete (image_ID);
    }
  }

  g_free (fgi);
  gimp_displays_flush ();

  values[1].type = GIMP_PDB_IMAGE;
  values[1].data.d_image = image_ID;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;
}

/* This function is loosely copy pasted from gimp 2.4 CVS. Thanks to
   sven. I'll remove it as soon as gimp goes mainstream. */
gint32
gs_layer_new_from_pixbuf (gint32                image_ID,
			  const gchar          *name,
			  GdkPixbuf            *pixbuf,
			  gdouble               opacity,
			  GimpLayerModeEffects  mode,
			  gdouble               progress_start,
			  gdouble               progress_end)
{
  GimpDrawable *drawable;
  GimpPixelRgn	rgn;
  const guchar *pixels;
  gpointer      pr;
  gint32        layer;
  gint          width;
  gint          height;
  gint          rowstride;
  gint          bpp;
  gdouble       range = progress_end - progress_start;
  guint         count = 0;
  guint         done  = 0;

  g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), -1);

  if (gimp_image_base_type (image_ID) != GIMP_RGB)
    {
      g_warning ("gimp_layer_new_from_pixbuf() needs an RGB image");
      return -1;
    }

  if (gdk_pixbuf_get_colorspace (pixbuf) != GDK_COLORSPACE_RGB)
    {
      g_warning ("gimp_layer_new_from_pixbuf() assumes that GdkPixbuf is RGB");
      return -1;
    }

  width  = gdk_pixbuf_get_width (pixbuf);
  height = gdk_pixbuf_get_height (pixbuf);
  bpp    = gdk_pixbuf_get_n_channels (pixbuf);

  layer = gimp_layer_new (image_ID, name, width, height,
                          bpp == 3 ? GIMP_RGB_IMAGE : GIMP_RGBA_IMAGE,
                          opacity, mode);

  if (layer == -1)
    return -1;

  drawable = gimp_drawable_get (layer);

  gimp_pixel_rgn_init (&rgn, drawable, 0, 0, width, height, TRUE, FALSE);

  g_assert (bpp == rgn.bpp);

  rowstride = gdk_pixbuf_get_rowstride (pixbuf);
  pixels    = gdk_pixbuf_get_pixels (pixbuf);

  for (pr = gimp_pixel_rgns_register (1, &rgn);
       pr != NULL;
       pr = gimp_pixel_rgns_process (pr))
    {
      const guchar *src  = pixels + rgn.y * rowstride + rgn.x * bpp;
      guchar       *dest = rgn.data;
      gint          y;

      for (y = 0; y < rgn.h; y++)
        {
          memcpy (dest, src, rgn.w * rgn.bpp);

          src  += rowstride;
          dest += rgn.rowstride;
        }

      if (range > 0.0)
        {
          done += rgn.h * rgn.w;

          if (count++ % 16 == 0)
            gimp_progress_update (progress_start +
                                  (gdouble) done / (width * height) * range);
        }
    }

  if (range > 0.0)
    gimp_progress_update (progress_end);

  gimp_drawable_detach (drawable);

  return layer;
}

/* Create a layer from the pixbuf */
void
fg_image_received (GnomeScanContext *context,
		   GnomeScanResult *info,
		   FlegitaGimpInfo *fgi)
{
  gint32 image, layer;
  GimpPixelRgn *region;
  GimpDrawable *drawable;
  gint width, height, rowstride, bpp, y;
  guchar *pixels, *dest;
  const guchar *src;
  const gchar *progress;
  gboolean status;
  gpointer iterator;

  image = fgi->image;

  status = gimp_progress_init (_("Importing scan"));

  width = gdk_pixbuf_get_width (info->image);
  height = gdk_pixbuf_get_height (info->image) ;
 
  if (fgi->is_new) {
    gimp_image_undo_freeze (image);

    /* Extend to needed size. */
    gimp_image_resize (image,
		       MAX (width, gimp_image_width (image)),
		       MAX (height, gimp_image_height (image)),
		       0, 0);

    gimp_image_set_resolution (image, info->resolution, info-> resolution);
    gimp_image_undo_thaw (image);
  }

  /* Create layer */
  layer = gs_layer_new_from_pixbuf (image,
				    gtk_entry_get_text (GTK_ENTRY (fgi->layer_name_field)),
				    info->image,
				    100.0,
				    GIMP_NORMAL_MODE,
				    0., 1.);

  gimp_image_add_layer (image,
			layer,
			-1);
  /* center layer */
  gimp_image_undo_freeze (image);
  gimp_layer_translate (layer,
			(gimp_image_width (image) - width) / 2,
			(gimp_image_height (image) - height) / 2);

  gimp_image_undo_thaw (image);

  gnome_scan_context_stop_acquisition (context);
  gtk_main_quit ();
}
