/* GnomeScan - Gnome Scanning Infrastructure
 *
 * gnomescantypes.h
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/**
 * SECTION:gnomescantypes
 * @title: Types
 * @short_description: Data types and enums.
 *
 * Blah.
 */

#ifndef GNOME_SCAN_TYPES_H
#define	GNOME_SCAN_TYPES_H

#include <glib-object.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "gnomescantypebuiltins.h"

#define GNOME_TYPE_SCAN_RANGE			(gnome_scan_range_get_type ())
#define GNOME_TYPE_SCAN_GEOMETRY		(gnome_scan_geometry_get_type ())
#define GNOME_TYPE_SCAN_CONTEXT_INFO		(gnome_scan_context_info_get_type ())
#define GNOME_TYPE_SCAN_FORECAST		(gnome_scan_forecast_get_type ())
#define GNOME_TYPE_SCAN_RESULT			(gnome_scan_result_get_type ())

G_BEGIN_DECLS

typedef struct	_GnomeScanGeometry		GnomeScanGeometry;
typedef struct	_GnomeScanArea			GnomeScanArea;
typedef struct	_GnomeScanRange			GnomeScanRange;
typedef struct	_GnomeScanContextInfo		GnomeScanContextInfo;
typedef struct	_GnomeScanForecast		GnomeScanForecast;
typedef struct	_GnomeScanResult		GnomeScanResult;
typedef union	_GnomeScannerOptionConstraint	GnomeScannerOptionConstraint;

/**
 * GnomeScanUnit:
 * @GNOME_SCAN_UNIT_NONE:		No unit
 * @GNOME_SCAN_UNIT_PIXEL:		Pixel
 * @GNOME_SCAN_UNIT_BIT:		Bit
 * @GNOME_SCAN_UNIT_MM:			Millimeter
 * @GNOME_SCAN_UNIT_DPI:		Resolution in dot/inch
 * @GNOME_SCAN_UNIT_PERCENT:		Percentage
 * @GNOME_SCAN_UNIT_MICROSECOND:	Microsecond
 *
 * All unit used to dialog with devices.
 */
typedef enum {
  GNOME_SCAN_UNIT_NONE,
  GNOME_SCAN_UNIT_PIXEL,
  GNOME_SCAN_UNIT_BIT,
  GNOME_SCAN_UNIT_MM,
  GNOME_SCAN_UNIT_DPI,
  GNOME_SCAN_UNIT_PERCENT,
  GNOME_SCAN_UNIT_MICROSECOND  
} GnomeScanUnit;

/**
 * GnomeScannerType:
 * @GNOME_SCANNER_UNKNOWN:		Unknown scanner type
 * @GNOME_SCANNER_FLATBED:		Flatbed scanner
 * @GNOME_SCANNER_MULTI_FUNCTION:	All-in-one printer
 * @GNOME_SCANNER_SHEETFED:		Sheetfed scanner
 * @GNOME_SCANNER_HANDHELD:		Handheld scanner
 * @GNOME_SCANNER_CARD_READER:		Business card reader scanner
 *
 * The various recognized kind of device.
 */
typedef enum {
  GNOME_SCANNER_UNKNOWN,
  GNOME_SCANNER_FLATBED,
  GNOME_SCANNER_MULTI_FUNCTION,
  GNOME_SCANNER_SHEETFED,
  GNOME_SCANNER_HANDHELD,
  GNOME_SCANNER_CARD_READER
} GnomeScannerType;

/**
 * GnomeScannerSource:
 * @GNOME_SCANNER_SOURCE_NONE:				No source.
 * @GNOME_SCANNER_SOURCE_AUTOMATIC:			The backend is able to automatically choose
 * 							the document source.
 * @GNOME_SCANNER_SOURCE_FLATBED:			A flatbed source is available.
 * @GNOME_SCANNER_SOURCE_AUTOMATIC_DOCUMENT_FEEDER:	An Automatic
 * 							Document Feeder is available.
 *
 * The supported document source of a scanner device.
 **/
typedef enum {
  GNOME_SCANNER_SOURCE_NONE,
  GNOME_SCANNER_SOURCE_AUTOMATIC,
  GNOME_SCANNER_SOURCE_FLATBED,
  GNOME_SCANNER_SOURCE_AUTOMATIC_DOCUMENT_FEEDER	/*< desc=Automatic Document Feeder >*/
} GnomeScannerSource;

/**
 * GnomeScanColorspace:
 * @GNOME_SCAN_COLORSPACE_GRAY:		Grayscale colorspace
 * @GNOME_SCAN_COLORSPACE_RGB:		Red-Green-Blue colorspace
 * @GNOME_SCAN_COLORSPACE_LINEART:	Lineart colorspace
 * @GNOME_SCAN_COLORSPACE_HALFTONE:	Halftone colorspace
 *
 * Colorspace of the final image.
 */
typedef enum {
  GNOME_SCAN_COLORSPACE_GRAY,
  GNOME_SCAN_COLORSPACE_RGB,
  GNOME_SCAN_COLORSPACE_LINEART,
  GNOME_SCAN_COLORSPACE_HALFTONE
} GnomeScanColorspace;

/**
 * GnomeScanGeometry:
 * @width:	available width in mm.
 * @height:	available height in mm.
 *
 * The dimension of an rectangle, but length are store in gdouble.
 **/
struct _GnomeScanGeometry {
  gdouble	width;
  gdouble	height;
};

/**
 * GnomeScanArea:
 * @custom	Wether the area is user defined or fixed (e.g. paper size).
 * @x:		top left corner X-coordinate
 * @y:		top left corner ordinate
 * @width:	area's width
 * @height:	area's height
 *
 * A geometry relative to coordonates.
 **/
struct _GnomeScanArea {
  gboolean	custom;
  gdouble	x;
  gdouble	y;
  gdouble	width;
  gdouble	height;
};

/**
 * GnomeScanRange:
 * @lower: minimal value
 * @upper: maximal value
 * @step: interval between values
 *
 * A range of gdouble using a step. A 0.0 @step indicate that no
 * interval are specified.
 **/
struct _GnomeScanRange {
  gdouble	lower;
  gdouble	upper;
  gdouble	step;
};

/**
 * GnomeScannerOptionConstraint:
 * @enumeration: the enum constraint
 * @range: the range constraint
 *
 * The constraint of an option can be either an enumertion or a
 * range. Use GnomeScannerOptionFlags to know which constraint is
 * used.
 **/
union _GnomeScannerOptionConstraint {
  GSList		*enumeration;
  GnomeScanRange	*range;
};

/**
 * GnomeScanContextInfo:
 *
 * A boxed type that represent the scan option's value to pass to de
 * backend.
 **/
struct _GnomeScanContextInfo {
  /*< private >*/
  gpointer 			scanner;
  gdouble			resolution;
  gboolean			preview;
  GnomeScanArea			*area;
  GnomeScannerSource		source;
  GnomeScanColorspace		colorspace;
  gint				depth;
  /* to be added: gamma, contrast, etc. */
};

/**
 * GnomeScanForecast:
 * @geometry:	image geometry in pixel
 * @size:	raw image size in bytes.
 * @rowstride:	image rowstride
 *
 * A structure that hold forecast information about acquisition.
 **/
struct _GnomeScanForecast {
  GnomeScanGeometry	geometry;
  gsize			size;
  gint			depth;
  gsize			rowstride;
};

/**
 * GnomeScanResult:
 * @preview:	wether the result is a preview or a final image.
 * @resolution:	the actual resolution of the final image
 * @source:	the source used to scan the image
 * @image:	the acquired #GdkPixbuf
 *
 * A structure representing the result of an acquisition. Since the
 * resolution may be rounded, the @resolution field provide the final
 * image resolution.
 **/
struct _GnomeScanResult {
  gboolean		preview;
  gdouble		resolution;
  GnomeScannerSource	source;
  GdkPixbuf		*image;
};


GType				gnome_scan_range_get_type		(void)	G_GNUC_CONST;
GType				gnome_scan_geometry_get_type		(void)	G_GNUC_CONST;

GType				gnome_scan_context_info_get_type	(void)	G_GNUC_CONST;
GnomeScanContextInfo*		gnome_scan_context_info_copy		(GnomeScanContextInfo *info);
void				gnome_scan_context_info_free		(GnomeScanContextInfo *info);

GType				gnome_scan_result_get_type		(void)	G_GNUC_CONST;
GnomeScanResult*		gnome_scan_result_copy			(GnomeScanResult *result);
void				gnome_scan_result_free			(GnomeScanResult *result);

gchar*				gnome_scanner_source_get_title		(GnomeScannerSource source);

G_END_DECLS

#endif
