//  Gnomoradio - rainbow/http-client.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "http-client.h"
#include "config.h"
#include "rainbow/util.h"

#include <algorithm>
#include <glibmm.h>
#include <iostream>
#include <sstream>
#include <cstdio>

using namespace Glib;
using namespace SigC;
using namespace std;

#define USER_AGENT "Gnomoradio " VERSION
#define CRLF "\r\n"

Rainbow::HttpClient::HttpClient (const ustring &hostname,
				 unsigned short port,
				 bool _keep_alive)
	: net_sock(0),
	  requested_host(hostname),
	  keep_alive(_keep_alive),
	  thread(0)
{
	struct hostent *host = gethostbyname(hostname.c_str());
	if (!host) {
		logmsg << "HttpClient: Cannot create host entry for " << hostname << endl;
		return;
	}

	server.sin_addr = *(struct in_addr*) host->h_addr;
	server.sin_family = AF_INET;
	server.sin_port = htons(port);

	my_signal_done.connect(slot(*this, &HttpClient::send_signal_done));
	my_signal_percent.connect(slot(*this, &HttpClient::send_signal_percent));

	if(!thread_supported())
		thread_init();
}

Rainbow::HttpClient::~HttpClient ()
{
	cancel();
}

void Rainbow::HttpClient::head (const ustring &rem)
{
	// make sure we are not in the middle of a get
	if (thread) {
		signal_request_done(false); // fixme: this also sends it out to the god thread.......
		return;
	}

	head_only = true;
	remote_file = rem;
	buffer = "";
	header.clear();

	thread = Thread::create(slot(*this, &HttpClient::request_thread), false);
}

void Rainbow::HttpClient::get (const ustring &rem,
			       const ustring &loc,
			       unsigned int off,
			       unsigned int len)
{
	post(rem, ustring(), loc, off, len);
}

void Rainbow::HttpClient::post (const ustring &rem,
				const ustring &pd,
				const ustring &loc,
				unsigned int off,
				unsigned int len)
{
	// make sure we are not in the middle of a get
	if (thread) {
		signal_request_done(false);
		return;
	}

	head_only = false;
	offset = off;
	length = len;
	remote_file = rem;
	local_file = loc;
	post_data = pd;
	buffer = "";
	header.clear();

	thread = Thread::create(slot(*this, &HttpClient::request_thread), false);
}

static inline unsigned int hex_to_int (const string &s)
{
	unsigned int retval = 0;
	string::const_iterator i = s.begin();
	while (i != s.end() && ((*i >= '0' && *i <= '9') || (*i >= 'a' && *i <='f'))) {
		retval *= 16;
		if (*i <= '9')
			retval += int(*i - '0');
		else
			retval += int(*i - 'a') + 10;
		++i;
	}
	return retval;
}

void Rainbow::HttpClient::cancel ()
{
	if (!connected())
		return;

	if (thread)
		1; // FIXME: thread->join();
	if (net_sock)
		close(net_sock);
}

void Rainbow::HttpClient::request_thread ()
{
	logmsg << "HttpClient: New thread to handle " << requested_host << endl;

	bool to_file = (local_file.length() != 0);
	bool first_request = (net_sock <= 0);

	first_line = string();

	// first establish a connection
	if (!connected()) {
		net_sock = socket(PF_INET, SOCK_STREAM, 0);
		if (net_sock < 1) {
			logmsg << "HttpClient: Cannot create socket" << endl;
			success = false;
			my_signal_done();
			return;
		}
		if (connect(net_sock, (struct sockaddr*) &server, sizeof(server))) {
			close(net_sock);
			net_sock = 0;
			logmsg << "HttpClient: Cannot connect to " << requested_host << endl;
			success = false;
			my_signal_done();
			return;
		}
	}

	// now construct the request
	string request = head_only ? "HEAD" : (post_data.size() ? "POST" : "GET");
	request += " " + remote_file + " HTTP/1.1" CRLF;
	logmsg << "HttpClient: " << request;
	request += "Host: " + requested_host + CRLF;
	request.append("User-Agent: " USER_AGENT CRLF);
	if (post_data.size()) {
		ostringstream pd;
		pd << post_data.size();
		request += "Content-Length: " + pd.str() + CRLF;
	}
	if (!keep_alive)
		request += "Connection: close" CRLF;
	request += CRLF;
	request += post_data;

	// now send the request
	if (!send_data_on_socket(net_sock, request, 8)) {
		close(net_sock);
		net_sock = 0;
		logmsg << "HttpClient: Broken socket to " << requested_host << endl;
		success = false;
		my_signal_done();
		return;
	}

	const size_t buf_size = 16384;
	char buf[buf_size];

	const int incoming_limit = 30000000;
	
	for (;;) {
		ssize_t r = read(net_sock, buf, buf_size);

		if (r <= 0 || buffer.size() > incoming_limit) {
			close(net_sock);
			net_sock = 0;
			logmsg << "HttpClient: Read failed" << endl;
			success = false;
			my_signal_done();
			return;
		}

		// parse request headers
		buffer.append(buf, r);
	header_loop:
		string::size_type p = buffer.find(CRLF);
		if (p == string::npos)
			continue; // no more headers left in this read
		string line = buffer.substr(0, p);
		buffer = buffer.substr(p + 2);
		if (line.size() == 0)
			break; // last header
		if (first_line.size() == 0) {
			first_line = line;
			goto header_loop;
		}
		string::size_type colon = line.find(": ");
		if (colon == string::npos)
			break; // malformed http header ... FIXME: maybe it would be better to error out than to try to continue
		// make lowercase
		transform(line.begin(), (line.begin() + colon),
			  line.begin(), ::tolower);
		header.insert(make_pair(line.substr(0, colon), line.substr(colon + 2)));
		goto header_loop;
	}

	// declare variables before goto's
	std::map<string,string>::iterator h;
	ssize_t remaining_bytes;
	size_t downloaded_bytes = buffer.size();
	bool chunked;
	FILE *file = 0;

	if (head_only)
		goto success_done;

	if (to_file) {
		try {
			file = fopen(filename_from_utf8(local_file).c_str(), "w");
		} catch (...) {
		}
		if (!file) {
			logmsg << "HttpClient: Could not write to local file \"" << local_file << "\"" << endl;
			close(net_sock);
			net_sock = 0;
			success = false;
			my_signal_done();
			return;
		}
	}

	h = header.find("content-length");
	if (h != header.end()) {
		size_t file_size = atoi(h->second.c_str());
		if (file_size > incoming_limit) {
			logmsg << "HttpClient: Content length is above limit" << endl;
			close(net_sock);
			net_sock = 0;
			success = false;
			my_signal_done();
			return;
		}
		remaining_bytes = file_size - downloaded_bytes;
		chunked = false;
		if (to_file) {
			fwrite(buffer.data(), 1, buffer.size(), file);
			buffer.erase();
		}
	} else {
		remaining_bytes = 0 - downloaded_bytes;
		//h = header.find("transfer-encoding");
		chunked = true;
	}

	// download file
	unsigned int hex;
	string::size_type start, end;

	for (;;) {
		if (!chunked) {
			if (remaining_bytes == 0)
				goto success_done;
			if (remaining_bytes < 0)
				goto error_done;
			int pcnt = (downloaded_bytes * 100) / (downloaded_bytes + remaining_bytes);
			percent_mutex.lock();
			if (pcnt != percent) {
				percent = pcnt;
				percent_mutex.unlock();
				my_signal_percent();
			} else {
				percent_mutex.unlock();
			}
		}
		if (chunked) {
			while (remaining_bytes < 0) {
				start = buffer.size() + remaining_bytes;
				if (remaining_bytes <= 2 && buffer.substr(start, 2) == CRLF) { // signifies end of chunk
					buffer.erase(start, 2);
					remaining_bytes += 2;
				}
				end = buffer.find(CRLF, start);
				if (end != string::npos) {
					hex = hex_to_int(buffer.substr(start, end - start));
					buffer.erase(start, end - start + 2);
					if (hex == 0)
						goto success_done;
					remaining_bytes = hex + start - buffer.size();
				} else
					break;
			}
		}
		if (downloaded_bytes > incoming_limit)
			goto error_done;
		ssize_t r = read(net_sock, buf, (chunked || remaining_bytes > buf_size) ? buf_size : remaining_bytes);
		if (r < 0)
			goto error_done;
		remaining_bytes -= r;
		downloaded_bytes += r;
		if (r == 0) {
			if (!chunked && remaining_bytes <= 0)
				goto success_done;
			else
				goto error_done;
		}
		if (to_file)
			fwrite(buf, 1, r, file); // FIXME: check for errors
		else
			buffer.append(buf, r);
	}

 success_done:
	if (to_file)
		fclose(file); // FIXME: check for errors
	if (!keep_alive) {
		close(net_sock);
		net_sock = 0;
	}
	success = true;
	my_signal_done();
	return;

 error_done:
	if (to_file)
		fclose(file); // FIXME: check for errors, possibly delete
	close(net_sock);
	net_sock = 0;
	success = false;
	my_signal_done();
}

bool Rainbow::HttpClient::parse_url(const ustring &url,
				    ustring &host,
				    unsigned short &port,
				    ustring &file)
{
	ustring::size_type p;
	
	// hostname
	p = url.find("//");
	if (p == ustring::npos)
		return false;
	host = url.substr(p + 2);

	// filename
	p =  host.find('/');
	if (p == ustring::npos)
		file = "/";
	else {
		file = host.substr(p);
		host.resize(p);
	}

	// port
	p = host.find(':');
	if (p == ustring::npos)	
		port = 80;
	else {
		port = atoi(host.substr(p + 1).c_str());
		host = host.substr(p);
		if (port == 0)
			return false;
	}

	return true;
}

string Rainbow::HttpClient::url_encode (const char *original)
{
	const char *chunk_begin = original;
	const char *chunk_end = original;
	string retval;

	while (*chunk_end) {
		if (!((*chunk_end >= 'A' && *chunk_end <= 'Z')
		    || (*chunk_end >= 'a' && *chunk_end <= 'z')
		    || (*chunk_end >= '0' && *chunk_end <= '9'))) {
			if (chunk_begin != chunk_end)
				retval.append(chunk_begin, chunk_end - chunk_begin);
			if (*chunk_end == ' ')
				retval.append("+");
			else {
				char hex[] = "%  ";
				hex[1] = *chunk_end / 16;
				hex[2] = *chunk_end % 16;
				retval.append(hex);
			}
			chunk_begin = chunk_end + 1;
		}
		chunk_end++;
	}

	if (chunk_begin != chunk_end)
		retval.append(chunk_begin, chunk_end - chunk_begin);

	return retval;
}

void Rainbow::HttpClient::send_signal_done ()
{
	logmsg << "HttpClient: " << (success ? "Successful" : "Error in") << " download of \"" << remote_file << "\" from " << requested_host << endl;

	thread = 0;

	percent_mutex.lock();
	percent = success ? 100 : 0;
	unsigned int p = percent;
	percent_mutex.unlock();

	signal_download_percent(p);
	signal_request_done(success);
}

void Rainbow::HttpClient::send_signal_percent ()
{
	percent_mutex.lock();
	unsigned int p = percent;
	percent_mutex.unlock();

	signal_download_percent(p);
}
