//  Gnomoradio - roboradio/song.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "song.h"

// for SongRef, which could be located elsewhere
#include "song-local.h"
#include "song-rainbow.h"

using namespace std;
using namespace Glib;
using namespace Roboradio;

namespace Roboradio
{
	static map<ustring,SongRef> known_songs;
	static set<SongRef> need_availability_check;

	SigC::Signal1<void,SongRef> Song::signal_global_new_song;
	SigC::Signal1<void,SongRef> Song::signal_global_song_info_changed;
	SigC::Signal1<void,SongRef> Song::signal_global_song_rating_changed;
	SigC::Signal1<void,SongRef> Song::signal_global_song_length_changed;
	SigC::Signal1<void,SongRef> Song::signal_global_song_status_changed;
	SigC::Signal2<void,SongRef,unsigned int> Song::signal_global_song_import_progress;
	SigC::Signal1<void,SongRef> Song::signal_global_song_done;
}

Roboradio::SongRef::SongRef (const ustring &url, bool check_ready, bool known_available)
{
	// find existing song with same url
	map<ustring,SongRef>::const_iterator pos;
	pos = known_songs.find(url);
	if (pos != known_songs.end()) {
		s = pos->second.s;
		s->ref();
		return;
	}

	// otherwise, create new song object
	if (url.size() > 0 && url[0] == '/') {
		s = new SongLocal(url);
	} else if (url.substr(0, 7) == "http://") {
		s = new SongRainbow(url);
	} else {
		// invalid url
		s = 0;
		return;
	}
	known_songs.insert(make_pair(url, *this));
	Song::signal_global_new_song(*this);
	if (known_available)
		s->set_status_available(true);
	if (check_ready) {
		s->obtain_available_info();
		//eventually: need_availability_check.insert(*this);
	}
}

Roboradio::Song::Song (const ustring &loc)
	: length(0),
	  rating(0),
	  last_play(0),
	  times_play(0),
	  url(loc),
	  refcnt(1)
{
}

Roboradio::Song::~Song ()
{
}

void Roboradio::Song::set_info (const ustring &key, const ustring &value)
{
	// check to make sure key actually changed
	map<ustring,ustring>::const_iterator pos;
	pos = info.find(key);
	if (pos != info.end() && pos->second == value)
		return;

	// update key
	info.erase(key);
	if (value != "")
		info.insert(make_pair(key, value));
	signal_info_changed();
	signal_global_song_info_changed(SongRef(this));
}

ustring Roboradio::Song::get_info (const ustring &key) const
{
	map<ustring,ustring>::const_iterator pos;

	pos = info.find(key);
	if (pos != info.end())
		return pos->second;
	else
		return "";
}

void Roboradio::Song::get_info (vector<ustring> &keys, vector<ustring> &values) const
{
	map<ustring,ustring>::const_iterator pos;

	for (pos = info.begin(); pos != info.end(); ++pos) {
		keys.push_back(pos->first);
		values.push_back(pos->second);
	}
}

void Roboradio::Song::set_rating (int new_rating)
{
	if (new_rating == rating)
		return; // no change
	if (new_rating <= max_rating && new_rating >= min_rating) {
		rating = new_rating;
		signal_rating_changed(new_rating);
		signal_global_song_rating_changed(SongRef(this));
	}
}

void Roboradio::Song::ref ()
{
	++refcnt;
}

void Roboradio::Song::unref ()
{
	if (--refcnt == 0)
		delete this;
}

void Roboradio::Song::playing_ref ()
{
	if (status.playing++ == 0)
		signal_status_changed(status);
}

void Roboradio::Song::playing_unref ()
{
	if (--status.playing == 0)
		signal_status_changed(status);
}

void Roboradio::Song::upcoming_ref ()
{
	if (status.upcoming++ == 0)
		signal_status_changed(status);
}

void Roboradio::Song::upcoming_unref ()
{
	if (--status.upcoming == 0)
		signal_status_changed(status);
}

#if 0
int Roboradio::Song::get_import_progress ()
{
	return -1;
}
#endif

vector<SongRef> Roboradio::Song::get_known_songs ()
{
	vector<SongRef> retval;
	map<ustring,SongRef>::const_iterator pos;

	for (pos = known_songs.begin(); pos != known_songs.end(); ++pos)
		retval.push_back(pos->second);
	
	return retval;
}

void Roboradio::Song::set_status_available (bool a)
{
	if (a != status.available) {
		status.available = a;
		signal_status_changed(status);
		signal_global_song_status_changed(SongRef(this));
	}
}

void Roboradio::Song::set_status_ready (bool r)
{
	if (r != status.ready) {
		status.ready = r;
		signal_status_changed(status);
		signal_global_song_status_changed(SongRef(this));
	}
}

void Roboradio::Song::set_length (Time len)
{
	if (len != length) {
		length = len;
		signal_length_changed(len);
		signal_global_song_length_changed(SongRef(this));
	}
}

void Roboradio::Song::set_url (const ustring &new_url)
{
	if (new_url != url) {
		url = new_url;
		// FIXME: add to map, or merge if url already exists
		signal_url_changed(url);
	}
}

void Roboradio::Song::set_import_progress (unsigned int p)
{
	signal_import_progress(p);
	signal_global_song_import_progress(SongRef(this), p);
}

void Roboradio::Song::done ()
{
	last_play = time(NULL);
	times_play++;
	signal_done();
	signal_global_song_done(SongRef(this));
}
