/* vi:set ts=8 sts=0 sw=8:
 * $Id: print.c,v 1.24 2000/03/21 08:25:45 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include "main.h"
#include "print.h"
#include "file.h"
#include "doc.h"
#include "dialog.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "misc.h"

#include "gnpintl.h"

/*** local function prototypes ***/
static char *get_filename(win_t *w);
static char *generate_temp_file(doc_t *d);
static void print_cancel(GtkWidget *wgt, gpointer cbdata);
static void print_destroy(GtkWidget *widget, gpointer cbdata);
static void print_execute(GtkWidget *w, gpointer cbdata);


/*** global function definitions ***/
/*
 * PUBLIC: print_cb
 *
 * creates print dialog box.  this should be the only routine global to
 * the world.
 */
void
print_cb(GtkWidget *widget, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;
	char *title;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	if (misc_show_and_raise(d->printwin))
		return;

#ifdef GTK_HAVE_FEATURES_1_1_0
	title = g_strdup_printf(_("Print '%s' ?"), doc_basefname(d));
#else
	title = g_new(char, strlen(doc_basefname(d)) + 11);
	sprintf(title, _("Print '%s' ?"), doc_basefname(d));
#endif

	get_dialog_entry(title,
			 _("Enter print command.\nUse '%s' for file name"),
			 _("Print Command:"),
			 128, prefs.printcmd,
			 GTK_SIGNAL_FUNC(print_execute), (gpointer)w,
			 GTK_SIGNAL_FUNC(print_cancel), (gpointer)d,
			 GTK_SIGNAL_FUNC(print_destroy), (gpointer)d,
			 &(d->printwin), &(d->printentry));
	g_free(title);
} /* file_print_cb */


/*** local function definitions ***/
/*
 * PRIVATE: get_filename
 *
 * returns the filename to be printed in a newly allocated buffer.
 * the filename is determined by this algorithm:
 *
 * 1. if the file to be printed has not been updated and exists on disk,
 * then print the file on disk immediately.
 * 2. if the file has been updated, or if it is "Untitled", we should
 * prompt the user whether or not to save the file, and then print it.
 * 2a.  if the user does not want to save it, then this is the only time
 * we actually need to create a temporary file.
 */
#define PRINT_TITLE	"Save before printing?"
#define PRINT_MSG	"has not been saved!"
static char *
get_filename(win_t *w)
{
	doc_t *d;
	char *fname = NULL;
	char *buttons[] =
		{ _("Print anyway"), _(" Save, then print "), _(BUTTON_CANCEL) };
	char *title, *msg;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	if (!d->changed && d->fname) {
		if (d->sb == NULL)
			d->sb = g_new(struct stat, 1);
		if (stat(d->fname, d->sb) == -1) {
			/* print warning */
			g_free(d->sb);
			d->sb = NULL;
			fname = generate_temp_file(d);
		} else
			fname = g_strdup(d->fname);
	} else { /* doc changed or no filename */
		int ret;

		title = g_strdup(PRINT_TITLE);
#ifdef GTK_HAVE_FEATURES_1_1_0
		msg = g_strdup_printf(" '%s' %s ",
				(d->fname) ? d->fname : UNTITLED, PRINT_MSG);
#else
		if (d->fname)
			msg = g_new(char, strlen(PRINT_MSG) +
					  strlen(d->fname) + 6);
		else
			msg = g_new(char, strlen(PRINT_MSG) +
					  strlen(UNTITLED) + 6);
		sprintf(msg, " '%s' %s ", 
			(d->fname) ? d->fname : UNTITLED, PRINT_MSG);
#endif

		ret = do_dialog(title, msg, 3, buttons, 3, NULL, NULL, TRUE);

		g_free(msg);
		g_free(title);

		switch (ret) {
		case 1 :
			fname = generate_temp_file(d);
			break;
		case 2 :
			if (!d->fname)
				(void)doc_saveas_common(w);
			else
				(void)doc_save_common(w);
			fname = g_strdup(d->fname);
			break;
		case 3 :
			fname = NULL;
			break;
		default:
			printf("get_filename: do_dialog returned %d\n", ret);
			exit(-1);
		} /* switch */
	} /* doc changed or no filename */

	return fname;
} /* get_filename */


/*
 * PRIVATE: generate_temp_file
 *
 * create and write to temp file.  returns name of temp file in malloc'd
 * buffer (needs to be freed afterwards).
 *
 */
static char *
generate_temp_file(doc_t *d)
{
	FILE *fp;
	int oldmask;
	char *fname, *buf;

	fname = my_tempname();

	oldmask = umask(0066);
	fp = file_open_fp(fname, "w", OPEN_EXCL, "generate_temp_file");
	if (fp == NULL) {
		umask(oldmask);
		return NULL;
	}

	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 0,
				     gtk_text_get_length(GTK_TEXT(d->data)));
	if (fputs(buf, fp) == EOF) {
		perror("generate_temp_file: can't write to tmp file");
		file_close_fp(fname, fp);
		g_free(fname);
		g_free(buf);
		umask(oldmask);
		return NULL;
	}
	g_free(buf);
	file_close_fp(fname, fp);
	umask(oldmask);

	return fname;
} /* generate_temp_file */


/*
 * PRIVATE: print_cancel
 *
 * destroy the print dialog box
 */
static void
print_cancel(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	g_assert(d != NULL);
	gtk_widget_destroy(d->printwin);
} /* print_cancel */


/*
 * PRIVATE: print_destroy
 *
 * destroy the print dialog box
 */
static void
print_destroy(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	g_assert(d != NULL);
	d->printentry = NULL;
	d->printwin = NULL;
} /* print_destroy */


/*
 * PRIVATE: print_execute
 *
 * actually execute the print command
 */
static void
print_execute(GtkWidget *wgt, gpointer cbdata)
{
	char *scmd, *pcmd, *tmp, *fname;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	/* print using specified command */
	if ((pcmd = gtk_entry_get_text(GTK_ENTRY(d->printentry))) == NULL)
		return;

	/* look for "file variable" and place marker */
	if ((tmp = strstr(pcmd, "%s")) == NULL)
		return;
	*tmp = '\0';
	tmp += 2;

	if ((fname = get_filename(w)) == NULL) {
		gtk_widget_destroy(d->printwin);
		return;	/* canceled */
	}

	/* build command and execute */
#ifdef GTK_HAVE_FEATURES_1_1_0
	scmd = g_strdup_printf("%s%s%s", pcmd, fname, tmp);
#else
	scmd = g_new(char, strlen(pcmd) + strlen(fname) + 1);
	sprintf(scmd, "%s%s%s", pcmd, fname, tmp);
#endif
	GNPDBG_PREFS(("print_execute: print command = '%s'\n", scmd));
	if (system(scmd) == -1)
		perror("print_execute: system() error");
	g_free(scmd);

	if (strncmp(fname, prefs.tmpdir, strlen(prefs.tmpdir)) == 0)
		if (unlink(fname))
			perror("print_execute: unlink() error");
	g_free(fname);

	msgbar_printf(w, _("Printed %s"), (d->fname) ? d->fname : UNTITLED);
	msgbox_printf(_("printed %s"), (d->fname) ? d->fname : UNTITLED);

	gtk_widget_destroy(d->printwin);
} /* print_execute */


/* the end */
