/*
    Copyright (c) 2004        Dale Mellor,  John Darrington
                  1998,  2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License,  or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef CUBE_H
#define CUBE_H


#include "cube_i.h"


/* Cube co-ordinates have their origin at the centre of the cube,  and their
   units are equivalent to one half of the length of one edge of a block. */



/* Unique identifiers for the faces,  which can be used to populate a
   bit-field. */
enum
    {
        FACE_0 = 0x01 << 0,
        FACE_1 = 0x01 << 1,
        FACE_2 = 0x01 << 2,
        FACE_3 = 0x01 << 3,
        FACE_4 = 0x01 << 4,
        FACE_5 = 0x01 << 5
    };


/* Create a new cube object with the given number of faces per side. */
Cube *new_cube (const unsigned cube_dimension);


/* Free up memory used by the cube. */
void free_cube (Cube *const cube);


/* Is the cube solved? */
typedef enum _Cube_Status { NOT_SOLVED=0,  SOLVED,  HALF_SOLVED } Cube_Status;
Cube_Status cube_status_check (const Cube *const cube);


/* Structure used to communicate which blocks of the cube lie in a certain
   slice. These objects should only be constructed by one of the identify_blocks
   functions,  and must be destroyed with the free_slice_blocks function. */
typedef struct _Slice_Blocks
{
    int *blocks;
    unsigned number_blocks;
    unsigned axis;

} Slice_Blocks;


/* Free the resources associated with an instance of the above structure. */
void free_slice_blocks (Slice_Blocks *const slice_blocks);


/* Return the identity of all the blocks in a particular slice (or the one
   containing the block with block_id). The return object must be freed with
   free_slice_blocks. Axis is in the interval [0,  3). */
Slice_Blocks *identify_blocks (Cube *const cube,
                               const unsigned block_id,
                               const unsigned axis);

Slice_Blocks *identify_blocks_2 (Cube *const cube,
                                 const GLfloat slice,
                                 const unsigned axis);


/* As above,  but return all blocks in the surface face corresponding to the axis
   (which may now be anywhere in the interval [0,  6)). */
Slice_Blocks *identify_surface_blocks (Cube *const cube,
                                       const unsigned axis);


/*  Rotate a complete slice,  as identified by a prior call to
    identify_blocks. The return value is zero on success,  one otherwise. The
    slice_blocks object must have been previously created with one of the
    identify_blocks functions. */
int rotate_slice (Cube *const cube,
                  const int turns,
                  const Slice_Blocks *const slice_blocks);


/* Get a copy of the tranformation of the specified block. The return value is
   zero on success,  one otherwise. */
int get_block_transform (const Cube *const cube,
                         const unsigned block,
                         Matrix transform);


/* Return an int identifying all the visible faces in this block. */
int get_visible_faces (const Cube *const cube,
                       const unsigned block);


/* Get a copy of the centre point of the face into p. */
void get_face_centre (const Cube *const cube,
                      const unsigned block,
                      const unsigned face,
                      point p);


/* Set the normal vector for block/face to v. This value gets transformed by
   the MODELVIEW CTM before saving. */
void set_normal_vector (Cube *const cube,
                        const unsigned block,
                        const unsigned face,
                        const vector v);


/* Set the vector for block/face/quadrant to v. This value gets transformed by
   the MODELVIEW CTM before saving. */
void set_quadrant_vector (Cube *const cube,
                          const unsigned block,
                          const unsigned face,
                          const unsigned quadrant,
                          const vector v);


/* Get a copy of the direction vector for the block/face/quadrant. */
void get_quadrant_vector (const Cube *const cube,
                          const unsigned block,
                          const unsigned face,
                          const unsigned quadrant,
                          vector v);


/* Print error messages generated by this library. */
void print_cube_error (const Cube *const cube,
                       const char *const message_prefix);



#include <libguile.h>

   /* Create a scheme object which represents the current state of the given
      cube. */
   SCM make_scm_cube (const Cube *const cube);


/***
    Concession to the non-object-oriented universe (this will go eventually).
***/
extern Cube *the_cube;

int create_the_cube (const unsigned dim);

void destroy_the_cube (void);



#endif /* CUBE_H */
