# GNU Enterprise Datasource Library - Driver for GNUe-AppServer
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Connection.py 7017 2005-02-12 22:45:04Z reinhard $

####                                   ####
#### IF YOU MODIFY ANY CONNECTION      ####
#### ATTRIBUTES, PLEASE UPDATE info.py ####
####                                   ####

import sys

from gnue.common.datasources import Exceptions
from gnue.common.datasources.drivers import Base
from gnue.common.rpc import client
from gnue.common.apps import errors, GConfig

import DataObject

from gnue.common.datasources.drivers.appserver.Schema.Discovery.Introspection import Introspection

# =============================================================================
# Connection class
# =============================================================================

class Connection (Base.Connection):
  """
  Handles a connection to the GNUe-AppServer backend.
  """
  defaultBehavior = Introspection
  supportedDataObjects = {
    'object': DataObject.DataObject
  }

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, name, parameters):
    Base.Connection.__init__ (self, connections, name, parameters)
    self._filters = None
    self._server  = None
    self._sm      = None
    

  # ---------------------------------------------------------------------------
  # Define the needed information to log in
  # ---------------------------------------------------------------------------

  def getLoginFields (self):
    result = []

    cfg = gConfigDict (section = 'appserver')
    dbauth = cfg.get ('authentication', 'False')
    if dbauth.lower () in ['true', 'yes', 'y']:
      result.extend ([['_username', _('User Name'), 0],
                      ['_password', _('Password'), 1]])

    self.__getSessionManager ()
    self._filters = self._sm.getFilters (self.parameters.get ('_language', 'C'))

    for item in self._filters:
      (filterId, filterLabel) = item [0]
      for (label, search, field) in item [1]:
        result.append ([filterId, filterLabel + ':', False])
        break

    return result


  # ---------------------------------------------------------------------------
  # Open a connection
  # ---------------------------------------------------------------------------

  def connect (self, connectData):

    user   = connectData.get ('_username', None)
    passwd = connectData.get ('_password', None)

    self.__getSessionManager ()

    self.__updateFilters (connectData)

    try:
      if connectData.has_key ('_username'):
        del connectData ['_username']
      if connectData.has_key ('_password'):
        del connectData ['_password']
      connectData ['user']     = user
      connectData ['password'] = passwd

      self._sess_id = self._sm.open (connectData)

    except errors.RemoteError, e:
      if e.getName () == 'AuthError':
        raise Exceptions.LoginError, e.getMessage ()
      else:
        raise Exceptions.ConnectionError, e.getMessage ()
      
    except gException, e:
      raise Exceptions.ConnectError, e.getMessage ()

    except:
      raise Exceptions.ConnectError, errors.getException () [2]

    # Can be removed after the call to _dataConnection.cursor() is removed from
    # the Base driver
    self.native = self

  # ---------------------------------------------------------------------------
  # Commit active transaction
  # ---------------------------------------------------------------------------

  def commit (self):
    self._sm.commit (self._sess_id)

  # ---------------------------------------------------------------------------
  # Rollback active transaction
  # ---------------------------------------------------------------------------

  def rollback (self):
    self._sm.rollback (self._sess_id)

  # ---------------------------------------------------------------------------
  # Close connection
  # ---------------------------------------------------------------------------

  def close (self):
    if self._sm is not None:
      self._sm.close (self._sess_id, False)

  # Can be removed after the call to _dataConnection.cursor() is removed from
  # the Base driver
  def cursor(self):
    return None


  # ---------------------------------------------------------------------------
  # Create/return a connection to the appserver
  # ---------------------------------------------------------------------------

  def __getSessionManager (self):

    if self._server is None:
      params = {'host'     : self.parameters.get ('host'),
                'port'     : self.parameters.get ('port'),
                'transport': self.parameters.get ('transport')}
      rpcType = self.parameters.get ('rpctype')
      self._server = client.attach (rpcType, params)

      if self.parameters.has_key ('timeout'):
        self._server.setTimeout (float (self.parameters ['timeout']))

    if self._sm is None:
      self._sm = self._server.request ('Session')

    return self._sm


  # ---------------------------------------------------------------------------
  # Update the given filter values
  # ---------------------------------------------------------------------------

  def __updateFilters (self, connectData):

    for item in self._filters:
      (filterId, filterLabel) = item [0]

      if connectData.has_key (filterId):
        value = connectData [filterId]
        (label, search, field) = item [1][0]

        # if there are no filter values we've to skip replacement. Maybe the
        # user just wants to add new filter values
        if not len (item [3].keys ()):
          continue

        if item [2] is None:
          masterkey = None
        else:
          masterkey = connectData [item [2]]

        found = False
        vDict = item [3][masterkey]
        for record in vDict:
          if record [field] == value:
            connectData [filterId] = record ['gnue_id']
            found = True
            break

        if not found:
          raise Exceptions.LoginError, \
              u_("'%(value)s' is not a valid filter-value for '%(filter)s'") \
              % {'value': value,
                 'filter': label}
