#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: $

import os
from gnue.common.datasources.drivers.DBSIG2.Schema.Creation import \
    Creation as Base


# =============================================================================
# Class implementing schema creation for Oracle
# =============================================================================

class Creation (Base.Creation):

  MAX_NAME_LENGTH = 31
  ALTER_MULTIPLE  = False
  _PK_PRECISION   = 10


  # ---------------------------------------------------------------------------
  # Create a new database
  # ---------------------------------------------------------------------------

  def createDatabase (self):
    """
    """
    raise gException, \
        _("This feature is currently not supported by this driver")


  # ---------------------------------------------------------------------------
  # Handle special defaults
  # ---------------------------------------------------------------------------

  def _defaultwith (self, code, tableName, fieldDef, forAlter):
    """
    This function creates a sequence for 'serials' and sets the default for
    'timestamps' to 'now ()'

    @param code: code-tuple to merge the result in
    @param tableName: name of the table
    @param fieldDef: dictionary describing the field with the default
    @param forAlter: TRUE if the definition is used in a table modification
    """
    if fieldDef ['defaultwith'] == 'serial':
      seq = self._getSequenceName (tableName, fieldDef)
      code [0].append (u"CREATE SEQUENCE %s MAXVALUE %s NOCYCLE%s" \
          % (seq, "9" * self._PK_PRECISION, self.END_COMMAND))

      body = u"IF :new.%(field)s IS NULL THEN" \
              " SELECT %(seq)s.nextval INTO :new.%(field)s FROM dual;" \
              "END IF;" \
            % {'field': fieldDef ['name'],
               'seq': seq}
      self.__addTrigger (tableName, fieldDef ['name'], code, body)


    elif fieldDef ['defaultwith'] == 'timestamp':
      if fieldDef ['type'] == 'date':
        sysdate = "TRUNC (sysdate)"
      else:
        sysdate = "sysdate"

      body = u"IF :new.%(field)s IS NULL THEN" \
              " :new.%(field)s := %(date)s;" \
              "END IF;" \
            % {'field': fieldDef ['name'],
               'date' : sysdate}
      self.__addTrigger (tableName, fieldDef ['name'], code, body)


  # ---------------------------------------------------------------------------
  # Create a trigger code and add it to the epilogue of the given code
  # ---------------------------------------------------------------------------

  def __addTrigger (self, tablename, fieldname, code, body):
    code [2].append (u"CREATE OR REPLACE TRIGGER t__%(table)s_%(field)s_pre"
           "  BEFORE INSERT ON %(table)s"
           "  FOR EACH ROW WHEN (new.%(field)s IS NULL)"
           "  BEGIN %(body)s END; /"
          % {'table': tablename,
             'field': fieldname,
             'body': body})



  # ---------------------------------------------------------------------------
  # A key is an integer
  # ---------------------------------------------------------------------------

  def key (self, fieldDefinition):
    """
    Native datatype for a 'key'-field is 'int8'

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype 'int8'
    """
    return "number (%s)" % self._PK_PRECISION


  # ---------------------------------------------------------------------------
  # String becomes varchar2
  # ---------------------------------------------------------------------------

  def string (self, fieldDefinition):
    """
    """
    length = fieldDefinition.get ('length', 99999)
    if length <= 2000:
      return "varchar2 (%s) % length"
    else:
      return "blob"


  # ---------------------------------------------------------------------------
  # Create an apropriate type for a number
  # ---------------------------------------------------------------------------

  def number (self, fieldDefinition):
    """
    This function returns an apropriate type for a number according to the
    given length and precision.

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype
    """
    scale  = fieldDefinition.get ('precision', 0)
    length = fieldDefinition.get ('length', 0)

    if scale == 0:
      return "number (%s)" % length
    else:
      return "number (%s,%s)" % (length, scale)


  # ---------------------------------------------------------------------------
  # Native datatype for boolean is boolean
  # ---------------------------------------------------------------------------

  def boolean (self, fieldDefinition):
    """
    This funciton returns the native data type for a boolean, which is
    'boolean'

    @param fieldDefinition: dictionary describing the field
    @return: 'boolean'
    """
    nullable = fieldDefinition.get ('nullable', True)

    if nullable:
      return "number (1) CHECK (%(field)s IS NULL OR %(field)s IN (0,1))" \
          % {'field': fieldDefinition ['name']}
    else:
      return "number (1) CHECK (%s IN (0,1))" % fieldDefinition ['name']


  # ---------------------------------------------------------------------------
  # Native datatype for datetime
  # ---------------------------------------------------------------------------

  def datetime (self, fieldDefinition):
    """
    This function returns the native type for a datetime value

    @param fieldDefinition: dictionary describing the field
    @return: 'date'
    """
    return "date"


  # ---------------------------------------------------------------------------
  # Native datatype for tim
  # ---------------------------------------------------------------------------

  def time (self, fieldDefinition):
    """
    This function returns the native type for a time value

    @param fieldDefinition: dictionary describing the field
    @return: 'date'
    """
    return "date"
