# GNU Enterprise Common - GNUe Schema Definition - Sax base XML Parser
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GSParser.py 6851 2005-01-03 20:59:28Z jcater $

import Objects
import copy
import types

from gnue.common.formatting import GTypecast
from gnue.common.definitions import GParser


xmlElements = None

# =============================================================================
# load an XML object tree from a given stream and return it's root object
# =============================================================================

def loadFile (buffer, app = None, initialize = True):
  """
  This function loads an XML object tree from a given stream and return it's
  root object.
  """

  return GParser.loadXMLObject (buffer, xmlSchemaHandler, 'GSSchema', 'schema',
           initialize, attributes = {'_app': app})


# =============================================================================
# Build a dictionary tree with all available XML elements
# =============================================================================

def getXMLelements ():
  """
  This function creates a dictionary tree with all valid xml elements. This
  dictionary tree is available via global variable 'xmlElements'
  """

  global xmlElements

  if xmlElements is None:
    xmlElements = {
      'schema':       {
         'BaseClass': Objects.GSSchema,
         'Required': 1,
         'SingleInstance': 1,
         'Attributes':  {
            'title':       {
               'Typecast': GTypecast.text },
            'author':       {
               'Typecast': GTypecast.text },
            'version':       {
               'Typecast': GTypecast.text },
            'description':       {
               'Typecast': GTypecast.text } } ,
         'ParentTags':  None },

      'tables':  {
         'BaseClass': Objects.GSTables,
         'SingleInstance': 1,
         'ParentTags':  ('schema',) },

      'table':    {
         'BaseClass': Objects.GSTable,
         'Importable': 1,
         'Attributes': {
            'name': {
               'Required': 1,
               'Unique': 1,
               'Typecast': GTypecast.name },
            'description': {
               'Typecast': GTypecast.text },
            'action': {
               'Typecast': GTypecast.text,
               'ValueSet': {
                  'create': {},
                  'extend': {} },
               'Default': 'create' }},
         'ParentTags':  ('tables',) },

      'fields':  {
         'BaseClass': Objects.GSFields,
         'Importable': 1,
         'SingleInstance': 1,
         'ParentTags':  ('table',) },

      'field':   {
         'BaseClass': Objects.GSField,
         'Importable': 1,
         'Attributes': {
            'name':          {
               'Required': 1,
               'Unique': 1,
               'Typecast': GTypecast.name },
            'description': {
               'Typecast': GTypecast.text },
            'type': {
               'Required': 1,
               'Typecast': GTypecast.name },
            'length': {
               'Typecast': GTypecast.whole },
            'precision': {
               'Typecast': GTypecast.whole,
               'Default': 0 },
            'auto': {
               'Typecast': GTypecast.boolean,
               'Default': 0 },
            'nullable':     {
               'Typecast': GTypecast.boolean,
               'Default': 1 },
            'default':     {
               'Typecast': GTypecast.text },
            'defaultwith':     {
               'Typecast': GTypecast.text,
               'ValueSet': {
                  'constant': {},
                  'timestamp': {},
                  'serial': {} },
               'Default': 'constant' },
               },
         'ParentTags':  ('fields',) },

      'primarykey':   {
         'BaseClass': Objects.GSPrimaryKey,
         'SingleInstance': 1,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('table',) },

      'pkfield':   {
         'BaseClass': Objects.GSPKField,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('primarykey',) },

      'constraints':   {
         'BaseClass': Objects.GSConstraints,
         'SingleInstance': 1,
         'ParentTags':  ('table',) },

      'constraint':    {
         'BaseClass': Objects.GSConstraint,
         'Attributes': {
            'name': {
               'Required': 1,
               'Typecast': GTypecast.name },
            'type': {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('constraints',) },

      'constraintfield':   {
         'BaseClass': Objects.GSConstraintField,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('constraint',) },

      'constraintref':   {
         'BaseClass': Objects.GSConstraintRef,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name },
             'table':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('constraint',) },

      'indexes':   {
         'BaseClass': Objects.GSIndexes,
         'SingleInstance': 1,
         'ParentTags':  ('table',) },
      'index':    {
         'BaseClass': Objects.GSIndex,
         'Attributes': {
            'name': {
               'Required': 1,
               'Typecast': GTypecast.name },
            'unique': {
               'Default': 0,
               'Typecast': GTypecast.boolean } },
         'ParentTags':  ('indexes',) },

      'indexfield':   {
         'BaseClass': Objects.GSIndexField,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('index',) },

      'data':   {
         'BaseClass': Objects.GSData,
         'SingleInstance': 1,
         'ParentTags':  ('schema',) },

      'tabledata':   {
         'BaseClass': Objects.GSTableData,
         'Importable': 1,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name },
            'tablename':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('data',) },

      'definition': {
        'BaseClass': Objects.GSDefinition,
        'SingleInstance': True,
        'ParentTags': ('tabledata',) },

      'column': {
        'BaseClass': Objects.GSColumn,
        'Attributes': {
          'field': {
            'Required': True,
            'Typecast': GTypecast.name },
          'type': {
            'Required': True,
            'Typecast': GTypecast.name },
          'key': {
            'Typecast': GTypecast.boolean,
            'Default' : False},
          },
        'ParentTags': ('definition',)},

      'rows':   {
         'BaseClass': Objects.GSRows,
         'SingleInstance': 1,
         'ParentTags':  ('tabledata',) },

      'row':   {
         'BaseClass': Objects.GSRow,
         'ParentTags':  ('rows',) },

      'value':   {
         'BaseClass': Objects.GSValue,
         'Attributes': {
            'field':        {
               'Required': 0,
               'Typecast': GTypecast.name },
            'type':        {
               'Required': 0,
               'Typecast': GTypecast.name,
               'Default':  'text' },
            'key': {
               'Required': 0,
               'Typecast': GTypecast.boolean,
               'Default' : 0}
            },
         'ParentTags':  ('row',),
         'MixedContent': 1,
         'KeepWhitespace': 1},

      'description':   {
         'BaseClass': Objects.GSDescription,
         'SingleInstance': 1,
         'MixedContent': 1,
         'UsableBySiblings': 1,
         'ParentTags':  ('schema',) },

    }

  return GParser.buildImportableTags ('schema', xmlElements)


# =============================================================================
# Class called by the XML parser to process the XML file
# =============================================================================

class xmlSchemaHandler (GParser.xmlHandler):

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self):

    GParser.xmlHandler.__init__(self)
    self.xmlElements = getXMLelements ()

