#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GBaseApp.py
#
# DESCRIPTION:
"""
Class that provides a basis for GNUe applications.

Typically, this class will not be called; rather, a tool will
be a GClientApp or GServerApp.
"""

import sys, os, getopt, string, types
import ConfigParser

from gnue import paths

from gnue.common.apps import GConfig
from gnue.common.apps import GDebug
from gnue.common.datasources import GConnections
from CommandOption import CommandOption

class GBaseApp:
  """
  The base class of the various GNUe application classes.

  GBaseApp Provides the following features
    - Command line argument parsing
    - Run time debug output levels
    - An integrated profiler
    - An integrated debugger
  """
  # Attributes to be overwritten by subclasses
  VERSION = "0.0.0"
  NAME = "GNUe Application"
  COMMAND_OPTIONS = []  # Should be in same format as _base_options below
  SUMMARY = "A brief summary of the program goes here."
  COMMAND = "app"
  USAGE = "[options]"
  USE_CONNECTIONS = 1       # Set to 1 if the program uses dbdrivers
  USE_DATABASE_OPTIONS = 0  # Also implies USE_CONNECTIONS = 1
  USE_RPC_OPTIONS = 0


  # More options, but won't be changed unless
  # this is a non-GNUe app using GNUe-Common
  AUTHOR = "GNU Enterprise Project"
  EMAIL = "info@gnue.org"
  REPORT_BUGS_TO = "Please report any bugs to info@gnue.org."
  CONFIGFILE="gnue.conf"

  #  Run the program
  #  Should be overwritten by subclasses
  def run(self):
    pass


  # Attributes that will be set by GClientApp after __init__ has run
  OPTIONS = {}        # Will contain a hash containing command line options
  ARGUMENTS = []      # Will contain an array of command line arguments
  connections = None  # Will contain a GConnection object

  def __init__(self, connections=None, application=None, defaults=None):

    # Basic options
    self._base_options = [

         ##
         ## Base options
         ##

         CommandOption('version', category="base", action=self.doVersion,
              help=_('Displays the version information for this program.') ),

         CommandOption('debug-level', category="base", default=0, argument=_("level"),
              help=_('Enables debugging messages.  Argument specifies the '
                      'level of messages to display (e.g., "--debug-level 5" displays '
                      'all debugging messages at level 5 or below.)') ),

         CommandOption('debug-file', category="base", argument=_("filename"),
              help=_('Sends all debugging messages to a specified file '
                     '(e.g., "--debug-file trace.log" sends all output to "trace.log")') ),

         CommandOption('help', category="base",  action=self.printHelp,
              help=_('Displays this help screen.') ),

         CommandOption('help-config', category="base",  action=self.doHelpConfig,
              help=_('Displays a list of valid configuration file entries, their '
                     'purpose, and their default values.') ),

         ##
         ## Developer options
         ##

         CommandOption('help-dev', category="base", action=self.printHelpDev,
              help=_('Display all options of interest to core developers. ') ),

         CommandOption('selfdoc', category="dev",  action=self.doSelfDoc,
              argument=_("type[,subtype]"),
              help=_('Generates self-documentation.') ),

         CommandOption('selfdoc-format', category="dev", argument=_("format"),
              help=_('Format to output the self-documentation in. Supported formats '
                     'are dependent on the type of selfdoc being created.') ),

         CommandOption('selfdoc-file', category="dev", argument=_("filename"),
              help=_('Specifies the filename that selfdoc should write to. If not provided, output is sent to stdout.') ),

         CommandOption('selfdoc-options', category="dev", argument=_("options"),
              help=_('Options specific to individual selfdoc types.') ),

         CommandOption('profile', category="dev",
              help=_("Run Python's built-in profiler and display the resulting "
                      "run statistics.") ),

         CommandOption('interactive-debugger', category="dev",
              help=_("Run the app inside Python's built-in debugger ")),
    ]

    if self.USE_DATABASE_OPTIONS:
      self.USE_CONNECTIONS = 1
      self._base_options += [

        CommandOption('username', category="connections", default='',
              argument='name',
              help=_('Username used to log into the database.  Note that if '
                     'specified, this will be used for all databases.  If not '
                     'supplied, the program will prompt for username.') ),

        CommandOption('password', category="connections", default='',
              argument='passwd',
              help=_('Password used to log into the database.  Note that if '
                     'specified, this will be used for all databases.  If not '
                     'supplied, the program will prompt for password if needed.'
                     '\nNOTE: SUPPLYING A PASSWORD VIA THE COMMAND LINE MAY BE '
                     'CONSIDERED A SECURITY RISK AND IS NOT RECOMMENDED.)') )
      ]

    if self.USE_CONNECTIONS:
      self._base_options += [
        CommandOption('help-connections', category="base", action=self.printHelpConn,
              help=_('Display help information related to database '
                     'connections, including a list of available drivers.') ),

        CommandOption('connections', category="connections",
              argument="loc",
              help=_('Specifies the location of the connection definition file. '
                     '<loc> may specify a file name '
                     '(/usr/local/gnue/etc/connections.conf),'
                     'or a URL location '
                     '(http://localhost/connections.conf).'
                     'If this option is not specified, the environent variable '
                     'GNUE_CONNECTIONS is checked.'
                     'If neither of them is set, "%s" is used as a default.') %
             os.path.join (paths.config, "connections.conf")) ]

    # Python version check
    try:
      if sys.hexversion < 0x02010000:
        self.handleStartupError (_('This application requires Python 2.1 or greater.  You are running Python %s') % sys.version[:5])
    except AttributeError:
      # Really, really old version of Python...
      self.handleStartupError (_('This application requires Python 2.1 or greater.'))


    #
    # Get all command line options and arguments
    #
    shortoptions = ""
    longoptions = []
    lookup = {}
    actions = {}

    # Convert old-style options to new-style
    if self.COMMAND_OPTIONS and type(self.COMMAND_OPTIONS[0]) == types.ListType:
      options=self.COMMAND_OPTIONS
      self.COMMAND_OPTIONS = []
      for option in options:
        self.COMMAND_OPTIONS.append(CommandOption(*option))

    for optionset in [self._base_options, self.COMMAND_OPTIONS]:
      for option in optionset:
        self.OPTIONS[option.name] = option.default
        if option.shortOption:
          shortoptions += option.shortOption
          lookup["-" + option.shortOption] = option.name
        lookup["--" + option.longOption] = option.name
        if option.action:
          actions["--" + option.longOption] = option.action
        lo = option.longOption
        if option.acceptsArgument:
          lo += '='
          shortoptions += ':'
        longoptions.append(lo)


    # mod_python apps don't have an argv
    # so create an empty one.
    # TODO: This class needs adjusted to
    #       be more efficent in mod_python cases
    #       But not this close to a release :)
    if not sys.__dict__.has_key('argv'):
      sys.argv=[]

    try:
      opt, self.ARGUMENTS = getopt.getopt(sys.argv[1:], shortoptions, longoptions)
    except getopt.error, msg:
      self.handleStartupError(msg)

    pendingActions = []
    for o in opt:
      if len(o[1]):
        self.OPTIONS[lookup[o[0]]] = o[1]
      else:
        self.OPTIONS[lookup[o[0]]] = True

      # Add any actions to our list
      try:
        pendingActions.append(actions[o[0]])
      except KeyError:
        pass

    for task in pendingActions:
      task()

    self._run = self.run

    # Should we profile?
    if self.OPTIONS['profile']:
      self.run = self._profile

    # Setup debugging
    # Should we run in debugger?
    elif self.OPTIONS['interactive-debugger']:
      self.run = self._debugger

    try:
      GDebug.setDebug(int("%s" % self.OPTIONS['debug-level']),
          self.OPTIONS['debug-file'])
    except ValueError:
      self.handleStartupError(_('The debug_level option ("-d") expects a numerical value.'))



    GDebug.printMesg(1,"Python %s" % sys.version)
    GDebug.printMesg(1,"Run Options: %s" % opt)
    GDebug.printMesg(1,"Run Arguments: %s" % self.ARGUMENTS)

    # Read the config files
    if application:
      try:
        self.configurationManager = GConfig.GConfig(application, defaults, configFilename=self.CONFIGFILE)
      except ConfigParser.NoSectionError, msg:
        self.handleStartupError(_('The gnue.conf file is incomplete: ')\
                                + '\n   %s'         % msg)
      except Exception, msg:
        etype=string.splitfields(str(sys.exc_type),'.')
        etype.reverse()
        self.handleStartupError( \
          _('%s while reading gnue.conf: ') % etype[0] \
          + '\n   %s'         % msg)

    # Add custom import to python's namespace
    try:
      extrapaths = gConfig('ImportPath')
    except:
      extrapaths = ""
    if extrapaths:
      for path in string.split(extrapaths,','):
        p = string.strip(path)
        if not p in sys.path:
          sys.path.append(p)

    # Get the connection definitions
    if connections != None:
      GDebug.printMesg(1,"Reusing connections instance")
      self.connections = connections
    elif self.USE_CONNECTIONS:

      # Check for default username/password
      lhOptions = {}
      if self.USE_DATABASE_OPTIONS:
        if self.OPTIONS['username']:
          lhOptions['_username'] = self.OPTIONS['username']
        if self.OPTIONS['password']:
          lhOptions['_password'] = self.OPTIONS['password']

      if self.OPTIONS['connections']:
        self.connections_file = self.OPTIONS['connections']
      elif os.environ.has_key('GNUE_CONNECTIONS'):
        self.connections_file = os.environ['GNUE_CONNECTIONS']
      else:
        self.connections_file = os.path.join (paths.config, "connections.conf")
#        self.handleStartupError(
#            _('Unable to load the connections definition file.\n') \
#            + _('\n   Please set the environmental variable GNUE_CONNECTIONS or ') \
#            + _('\n   use the "-f" command option.'))

      GDebug.printMesg(1, 'Connection Definition: "%s"' % self.connections_file)

      try:
        self.connections = GConnections.GConnections(self.connections_file, loginOptions=lhOptions)
      except GConnections.InvalidFormatError, msg:
        self.handleStartupError(
            _('Unable to load the connections definition file.\n') \
            + _('\n   The connections file is in an invalid format. ') \
            + '\n   %s' \
               % msg)
      except IOError:
        self.handleStartupError(
             _('Unable to load the connections definition file.\n') \
             + _('\n   The connections file specified either does ') \
             + _('\n   not exist or is not readable by your account.\n') \
             + _('\n   Location: "%s"')
                % self.connections_file)

  def addCommandOption(self, *args, **parms):
    self.COMMAND_OPTIONS.append(CommandOption(*args, **parms))

  #
  #  Display version information for this application
  #
  def printVersion(self):
    from gnue.common import VERSION as commonVersion
    print _("\n%s\nVersion %s\n") % (self.NAME, self.VERSION)
    print _("GNUe Common Version %s\n") % commonVersion

  #
  #  "Help" helpers
  #
  def buildHelpOptions(self, category=None):
    allOptions = {}
    descriptors = {}
    maxLength = 0

    for optionset in [self._base_options, self.COMMAND_OPTIONS]:
      for option in optionset:

        # Limit this to the correct category. A category
        # of None implies all options except "dev".
        if not (  ( category is None and \
                    option.category != "dev" ) or \
                  ( category == option.category ) ):
          continue

        allOptions[string.upper(option.longOption)] = option

        if option.acceptsArgument:
          descr = '--%s <%s>' % (option.longOption, option.argumentName)
        else:
          descr = '--%s' % (option.longOption)
        if option.shortOption:
          descr += ', -%s' % option.shortOption

        descriptors[string.upper(option.longOption)] = descr

        if len(descr) > maxLength:
          maxLength = len(descr)

    if maxLength > 10:
      maxLength = 10

    sorted = allOptions.keys()
    sorted.sort()

    dispOptions = ""

    for optionKey in sorted:

      margin = maxLength + 4
      width = 78 - margin
      pos = 0

      if len(descriptors[optionKey]) > maxLength:
        dispOptions += "\n  %s\n%s" % (descriptors[optionKey], " " * margin)
      else:
        dispOptions += "\n  %s  %s" % (descriptors[optionKey],
               " " * (maxLength - len(descriptors[optionKey])))

      for word in string.split(allOptions[optionKey].help):
        if (len(word) + pos) > width:
          pos = 0
          dispOptions += "\n" + " " * margin

        pos = pos + len(word) + 1

        dispOptions += word + " "

      dispOptions +=  "\n"

    return dispOptions

  def printHelpHeader(self):
    self.printVersion()
    print _("Usage:  ") + self.COMMAND + ' ' + self.USAGE
    print

  def printHelpFooter(self):
    print
    print "%s\n" % self.REPORT_BUGS_TO

  #
  #  Display help information for this program
  #
  def printHelp(self):
    self.printVersion()
    print _("Usage:  ") + self.COMMAND + ' ' + self.USAGE
    print "\n" + self.SUMMARY + '\n'

    print _('Available command line options:')
    print self.buildHelpOptions()
    self.printHelpFooter()
    sys.exit()

  #
  #  Display dev help information for this program
  #
  def printHelpDev(self):
    self.printHelpHeader()
    print _("The following options are mainly of interest to GNUe developers.")
    print _("To view general help, run this command with the --help option.")
    print
    print _('Developer-specific command line options:')
    print self.buildHelpOptions("dev")
    self.printHelpFooter()
    sys.exit()

  def printHelpConn(self):
    self.printHelpHeader()
    print _("The following connection/database-related options are available.")
    print _("To view general help, run this command with the --help option.")
    print
    print _('Database/connection command line options:')
    print self.buildHelpOptions("connections")
    print
    print _('The following database drivers are installed on your system:')
    print "   TODO\n"
    # print self.connections.getAvailableDrivers()
    self.printHelpFooter()
    sys.exit()

  def printHelpRPC(self):
    self.printHelpHeader()
    print _("The following options are mainly of interest to GNUe developers.")
    print _("To view general help, run this command with the --help option.")
    print
    print _('Developer-specific command line options:')
    print self.buildHelpOptions()
    self.printHelpFooter()
    sys.exit()

  def selfdoc(self, command, handle, format=None, options={}):
    if command == 'manpage':
      import manpage
      manpage.ManPage(self, handle, format, options)

  #
  # Converts a
  #
  def getCommandLineParameters(self,paramList):
    parameters = {}
    for param in paramList:
      psplit = string.split(param,'=',1)
      if len(psplit) == 1:
        self.handleStartupError ( \
                                  'Parameter "%s" specified, but no value supplied.' % psplit[0] )
      parameters[string.lower(psplit[0])] = psplit[1]

      GDebug.printMesg(2,'Param "%s"="%s" ' % \
                         (string.lower(psplit[0]), psplit[1]))
    return parameters

  #
  #  Display a startup error and exit gracefully with a message on
  #  how to get help
  #
  def handleStartupError(self, msg):
    self.printVersion()

    # if msg is multiline, then surround with dashes to set it apart
    if string.find("%s" % msg, "\n") + 1:
      print '-' * 60
    print _("Error: %s") % msg
    if string.find("%s" % msg, "\n") + 1:
      print '-' * 60
    print _("\nFor help, type:\n   %s --help\n") % (self.COMMAND)
    sys.exit()


  #
  # Used when interactive debugger in use
  #
  def _debugger(self):

    import pdb
    debugger = pdb.Pdb()
    GDebug.setDebugger(debugger)
    debugger.runctx( 'self._run()', globals(), locals() )

  #
  #  Used when profiling
  #
  def _profile(self):

    import profile
    prof = profile.Profile()
    prof.runctx( 'self._run()', globals(), locals() )

    import pstats
    p = pstats.Stats(prof)
    p.sort_stats('time').print_stats(50)
    p.sort_stats('cumulative').print_stats(50)
    p.sort_stats('calls').print_stats(50)


  #
  # Commands run by the COMMAND_OPTIONS list
  #
  def doVersion(self):
    self.printVersion()
    sys.exit()

  def doSelfDoc(self):
    if self.OPTIONS['selfdoc-file']:
      doprint = False
      handle = open(self.OPTIONS['selfdoc-file'],'w')
    else:
      doprint = True
      import StringIO
      handle = StringIO.StringIO

    self.selfdoc(self.OPTIONS['selfdoc'], handle,
                 self.OPTIONS['selfdoc-format'],
                 self.OPTIONS['selfdoc-options'])
    if doprint:
      handle.seek(0)
      print handle.read()
    handle.close()
    sys.exit()

  def doHelpConfig(self):
    self.printHelpHeader()
    print GConfig.printableConfigOptions(defaults)
    sys.exit()
