#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GConditions.py
#
# DESCRIPTION:
#
#
# NOTES:
#
# HISTORY:
#

from gnue.common.definitions.GObjects import GObj
from gnue.common.formatting import GTypecast
import types,string

class ConditionError (gException):
  pass

class ConditionNotSupported (ConditionError):
  pass


class GCondition(GObj):
  def __init__(self, parent=None, type="GCCondition"):
    GObj.__init__(self, parent, type=type)

class GConditions(GCondition):
  def __init__(self, parent=None, type="GCConditions"):
    GCondition.__init__(self, parent, type=type)

class GConditionElement (GCondition) :
  def __init__(self, parent=None, type="GConditionElement"):
    GCondition.__init__(self, parent, type=type)

class GCField (GConditionElement):
  def __init__(self, parent, name=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCCField')
    self.type = datatype
    self.name = name

class GCParam (GConditionElement):
  def __init__(self, parent, name=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCCParam')
    self.type = datatype
    self.name = name

  def getValue(self):
    return ""

class GCConst (GConditionElement):
  def __init__(self, parent, value=None, datatype="char"):
    GConditionElement.__init__(self, parent, 'GCCConst')
    self.type = datatype
    self.value = value

class GCadd(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCadd')

class GCsub(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCsub')

class GCmul(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCmul')

class GCdiv(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCdiv')

class GCand(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCand')

class GCor(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCor')

class GCnot(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnot')

class GCnegate(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnegate')

class GCeq(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCeq')

class GCne(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCne')

class GCgt(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCgt')

class GCge(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCge')

class GClt(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GClt')

class GCle(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCle')

class GClike(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GClike')

class GCnotlike(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnotlike')

class GCbetween(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCbetween')

class GCnotbetween(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnotbetween')

class GCnull(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnull')

class GCnotnull(GConditionElement):
  def __init__(self, parent=None):
    GConditionElement.__init__(self, parent, 'GCnotnull')



# build an impossible condition  GCondition={0=1}
GCimpossible=GCondition()
_h=GCeq(GCimpossible)
GCConst(_h,1)
GCConst(_h,2)


def getXMLelements(updates={}):
  xmlElements = {
##      'conditions':       {
##         'BaseClass': GConditions,
##         'SingleInstance': 1,
##         'ParentTags':  None },

      'conditions':       {
         'BaseClass': GCondition,
         'ParentTags':  ('conditions','and','or','not','negate'),
         'Deprecated': 'Use the <condition> tag instead.',
         },
      'condition':       {
         'BaseClass': GCondition,
         'ParentTags':  ('conditions','and','or','not','negate') },
      'cfield':       {
         'BaseClass': GCField,
         'Attributes': {
            'name':     {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'cparam':       {
         'BaseClass': GCParam,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Unique':   1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'cconst':       {
         'BaseClass': GCConst,
         'Attributes': {
            'value':     {
               'Required': 1,
               'Typecast': GTypecast.text },
            'type':     {
               'Typecast': GTypecast.text } },
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'add':       {
         'BaseClass': GCadd,
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'sub':       {
         'BaseClass': GCsub,
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'mul':       {
         'BaseClass': GCmul,
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'div':       {
         'BaseClass': GCdiv,
         'ParentTags':  ('eq','ne','lt','le','gt','ge','add','sub','mul',
                         'div','like','notlike','between','notbetween') },
      'and':       {
         'BaseClass': GCand,
         'ParentTags':  ('condition','and','or','not','negate') },
      'or':       {
         'BaseClass': GCor,
         'ParentTags':  ('condition','and','or','not','negate') },
      'not':       {
         'BaseClass': GCnot,
         'ParentTags':  ('condition','and','or','not','negate') },
      'negate':       {
         'BaseClass': GCnegate,
         'ParentTags':  ('condition','and','or','not','negate') },
      'eq':       {
         'BaseClass': GCeq,
         'ParentTags':  ('condition','and','or','not','negate') },
      'ne':       {
         'BaseClass': GCne,
         'ParentTags':  ('condition','and','or','not','negate') },
      'gt':       {
         'BaseClass': GCgt,
         'ParentTags':  ('condition','and','or','not','negate') },
      'ge':       {
         'BaseClass': GCge,
         'ParentTags':  ('condition','and','or','not','negate') },
      'lt':       {
         'BaseClass': GClt,
         'ParentTags':  ('condition','and','or','not','negate') },
      'le':       {
         'BaseClass': GCle,
         'ParentTags':  ('condition','and','or','not','negate') },
      'like':       {
         'BaseClass': GClike,
         'ParentTags':  ('condition','and','or','not','negate') },
      'notlike':       {
         'BaseClass': GCnotlike,
         'ParentTags':  ('condition','and','or','not','negate') },
      'between':       {
         'BaseClass': GCbetween,
         'ParentTags':  ('condition','and','or','not','negate') },
      'notbetween':       {
         'BaseClass': GCnotbetween,
         'ParentTags':  ('condition','and','or','not','negate') },
      'null':      {
         'BaseClass': GCnull,
         'ParentTags': ('condition','and','or','not') },
      'notnull':      {
         'BaseClass': GCnotnull,
         'ParentTags': ('condition','and','or','not') },
      }

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  return xmlElements


#
# a table with extra information about the different condition types
#  (needed for condition tree <-> prefix table conversion)
#
conditionElements = {
  'add':             (2, 999, GCadd ),
  'sub':             (2, 999, GCsub ),
  'mul':             (2, 999, GCmul ),
  'div':             (2, 999, GCdiv ),
  'and':             (1, 999, GCand ),
  'or':              (2, 999, GCor  ),
  'not':             (1,   1, GCnot ),
  'negate':          (1,   1, GCnegate ),
  'eq':              (2,   2, GCeq  ),
  'ne':              (2,   2, GCne  ),
  'gt':              (2,   2, GCgt  ),
  'ge':              (2,   2, GCge  ),
  'lt':              (2,   2, GClt  ),
  'le':              (2,   2, GCle  ),
  'like':            (2,   2, GClike ),
  'notlike':         (2,   2, GCnotlike ),
  'between':         (3,   3, GCbetween )
  }


#############################################################################
#############################################################################
####                       Convenience Methods                             ##
#############################################################################
#############################################################################


#
# Build a condition tree using a dict
# as a source.  Assumes keys are field
# names and values are constants.
#
def buildConditionFromDict (dict, comparison=GCeq, logic=GCand):
  cond = GCondition()
  lastParent = cond

  if len(dict.keys()):
    lastParent = logic(lastParent)

  for key in dict.keys():
    eq = comparison(lastParent)
    GCField(eq, key)

##    if string.split(str(type(dict[key])),"'")[1] in ('int', 'float'):
    if type(dict[key]) in (types.IntType, types.FloatType):
        consttype="number"
    else:
        consttype="char"
    GCConst(eq, dict[key], consttype)

  return cond

#
# Combine two conditions with an and clause.
# NOTE: This modifies cond1 (and also returns it)
#
def combineConditions (cond1, cond2):
  if cond1 == None or cond1 == {}:
    return cond2
  elif cond2 == None or cond2 == {}:
    return cond1

  if type(cond1) == type({}):
    cond1 = buildConditionFromDict(cond1)
  if type(cond2) == type({}):
    cond1 = buildConditionFromDict(cond2)

  if not len(cond1._children):
    cond1._children = cond2._children
    return cond1
  elif len(cond2._children):
    children = cond1._children[:]
    cond1._children = []
    _and = GCand(cond1)
    _and._children = children
    if len(cond2._children) > 1:
      _and2 = GCand(cond1)
      _and2._children = cond2._children[:]
    else:
      cond1._children.append(cond2._children[0])

  return cond1


# creates an GCondition Tree out of an list of tokens in a prefix
# order.

def buildTreeFromPrefix(term):

  # create a GCondition object as top object in the object stack
  parent={0:(GCondition())}

  # GCondition will have only one parameter
  # add paramcount=1 to the parameter count stack
  paramcount={0:1}

  # set start level for stack to zero
  level=0
  for i in term:

    # convert type into an object
    if conditionElements.has_key(i[0]):
      e=conditionElements[i[0]][2](parent[level])
      level=level+1
      # get parameter count
      paramcount[level]=conditionElements[i[0]][0]
      parent[level]=e
    elif i[0]=="field":
      e=GCField(parent[level], i[1])
      paramcount[level]=paramcount[level]-1
      if paramcount[level]==0:
        level=level-1
    elif i[0]=="const":
      e=GCConst(parent[level], i[1])
      paramcount.update({level:(paramcount[level]-1)})
      if paramcount[level]==0:
        level=level-1
#    print "NAME: %s  VALUE: %s  LEVEL: %s PCOUNT: %s" % \
#          (i[0],i[1],level,paramcount[level])

  return parent[0];


def buildPrefixFromTree(conditionTree):
  if type(conditionTree) != types.InstanceType:
    tmsg = u_("No valid condition tree")
    raise ConditionError, tmsg
  else:
    otype = string.lower(conditionTree._type[2:])

    #
    #  care for objects without children
    #
    if otype == 'cfield':
      return [('field',"%s" % conditionTree.name)]

    elif otype == 'cconst':
      return [('const',conditionTree.value)]

    elif otype == 'param':
      return [('param', conditionTree.getValue())]

    #
    #  if its an conditional object, then process it's children
    #
    elif conditionElements.has_key(otype):
      result=[]

      # first add operator to the list
      result.append((otype,''));  #  ,None));


      # change operations with more than there minimal element no into
      # multiple operations with minimal elements
      # reason: to prevent a b c d AND OR being not well defined
      # because it can be a"a b c d AND AND OR" or "a b c d AND OR OR"
      paramcount=len(conditionTree._children)
      while (paramcount > \
             conditionElements[otype][0]):
        paramcount=paramcount-1
        result.append((otype,''));


      # then add children
      for i in range(0, len(conditionTree._children)):
        result = result + \
                 buildPrefixFromTree(conditionTree._children[i])

      #
      #  check for integrity of condition
      #
      if len(conditionTree._children) < conditionElements[otype][0]:
        tmsg = u_('Condition element "%s" expects at least %s arguments; found %s') % \
                 (otype, conditionElements[otype][0], len(conditionTree._children))
        raise ConditionError, tmsg

      if len(conditionTree._children) > conditionElements[otype][1]:
        tmsg = u_('Condition element "%s" expects at most %s arguments; found %s') % \
                (otype, conditionElements[otype][1], len(conditionTree._children))
        raise ConditionError, tmsg


      # return combination
      return result;

    else:
      tmsg = u_('Condition clause "%s" is not supported '
                'by the condition to prefix table conversion.') % otype
      raise ConditionNotSupported, tmsg

