#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GConnection.py
#
# DESCRIPTION:
#
# NOTES:
#

__all__ = ['RecordSet']

from gnue.common.apps import GDebug
from gnue.common.datasources import GConditions, Exceptions
import string

###########################################################
#
#
#
###########################################################
class RecordSet:

  def __init__(self, parent, initialData={}, dbIdentifier=None, defaultData={}):
    self._detailObjects = []
    self._dbIdentifier = dbIdentifier
    self._deleteFlag = 0
    self._updateFlag = 0
    self._parent = parent
    self._fieldOrder = {}
    self._modifiedFlags = {}      # If field name is present as a key,
                                  # then field has been modified

    self._cachedDetailResultSets = {}

    self._initialData = initialData

    if self._initialData and len(self._initialData):
      self._insertFlag = 0
      self._emptyFlag = 0
      self._fields = {}
      self._fields.update(initialData)
    else:
      self._insertFlag = 1
      self._emptyFlag = 1
      self._fields = {}
      self._fields.update(defaultData)

  def __setitem__(self, attr, val):
    self.setField(attr, val)

  def __getitem__(self, attr):
    return self.getField(attr)

  # Returns 1=Record has uncommitted changes
  def isPending(self):

    # The _insertFlag and _deleteFlag takes care of records that
    # were inserted, but then deleted before a save (i.e., nothing to do)
    if self._emptyFlag or self._insertFlag and self._deleteFlag:
      return 0
    else:
      return self._insertFlag or self._deleteFlag or self._updateFlag


  # Returns 1=Record is pending a deletion
  def isDeleted(self):
    if self._emptyFlag:
      return 0
    else:
      return self._deleteFlag and not self._insertFlag


  # Returns 1=Record is pending an update
  def isModified(self):
    if self._emptyFlag or self._insertFlag:
      return 0
    else:
      return self._updateFlag


  # Returns 1=Record is pending an insertion
  def isInserted(self):
    if self._emptyFlag:
      return 0
    else:
      return self._insertFlag and not self._deleteFlag


  # Returns 1=Record is empty (inserted, but no data set)
  def isEmpty(self):
    return self._emptyFlag


  # Returns current value of "field"
  def getField(self, field):
    try:
      return self._fields[field]
    except KeyError:
      try:

        # TODO: When we're confident that
        # TODO: all field names are lowercase,
        # TODO: then this can be removed.

        return self._fields[string.lower(field)]
      except KeyError:
        # If a field value has yet to be set
        # (either from a query or via a setField),
        # then _fields will not contain a key
        # for the requested field even though
        # the field name may still be valid.
        return None


  # Sets current value of "field"
  # If trackMod is set to 0 then the modification flag isn't raised
  def setField(self, field, value, trackMod = 1):
    # If this field is bound to a datasource and the datasource is read only,
    # generate an error.
    if self._parent.isFieldBound(field) and self._parent.isReadOnly():
      # Provide better feedback??
      tmsg = _("Attempted to modify read only field '%s'") % field
      raise Exceptions.ReadOnlyError, tmsg
    else:
      # -- start quote
      # <jamest> change it and see who screams!
      # -- end quote
      # old code:
      # fn = string.lower(field)
      # self._fields[fn] = value
      # new code:
      self._fields[field] = value
      if trackMod == 1:
        if self._parent.isFieldBound(field):
          self._emptyFlag = 0
          self._updateFlag = 1
          # self._modifiedFlags[fn] = 1
          self._modifiedFlags[field] = 1

          try:
            self._parent._dataObject._dataSource._onModification(self)
          except AttributeError:
            pass
    return value

  # Batch mode of above setField method
  # If trackMod is set to 0 then the modification flag isn't raised
  def setFields(self, updateDict, trackMod = 1):
    # If this field is bound to a datasource and the datasource is read only,
    # generate an error.
    for field in updateDict.keys():
      self.setField(field, updateDict[field], trackMod)


  def getFieldsAsDict(self):
    """
    Returns the record set as a dictionary.

    @return: A python dictionary of field name/value pairs.
    """

    results = {}
    for field in self._fields:
      results[field] = self.getField(field)
    return results

  # Returns 1=Field has been modified
  def isFieldModified(self, fieldName):
    if self._modifiedFlags.has_key (fieldName):
      return 1
    else:
      #TODO: the string.lower() line should never be called but is left here
      #TODO: until the code is clean
      return self._modifiedFlags.has_key (string.lower(fieldName))


  # Mark the current record as deleted
  def delete(self):
    if self._parent.isReadOnly():
      # Provide better feedback??
      tmsg = _("Attempted to delete from a read only datasource")
      raise Exceptions.ReadOnlyError, tmsg
    else:
      self._deleteFlag = 1


  # Posts changes to database
  def post(self):
    # Should a post() to a read only datasource cause a ReadOnlyError?
    # It does no harm to attempt to post since nothing will be posted,
    # But does this allow sloppy programming?

    GDebug.printMesg(5,'Preparing to post datasource %s' %  self._parent._dataObject.name)

    # Save the initial status so we know if any triggers changed us
    status = (self._insertFlag, self._deleteFlag, self._updateFlag)

    # Call the hooks for commit-level hooks
    if not self._emptyFlag and hasattr(self._parent._dataObject,'_dataSource'):

      if self._insertFlag and not self._deleteFlag:
        self._parent._dataObject._dataSource._beforeCommitInsert(self)
      elif self._deleteFlag and not self._insertFlag:
        self._parent._dataObject._dataSource._beforeCommitDelete(self)
      elif self._updateFlag:
        self._parent._dataObject._dataSource._beforeCommitUpdate(self)

    #
    # If the record status changed while we were doing the triggers,
    # start from the beginning and run the triggers again.
    #
    if status != (self._insertFlag, self._deleteFlag, self._updateFlag):
      self.post()
      return


    if self.isPending():
      GDebug.printMesg(5,'Posting datasource %s' % self._parent._dataObject.name)

      if self.isPending():
        self._postChanges()


    # Post all detail records
    for child in (self._cachedDetailResultSets.keys()):
      c = self._cachedDetailResultSets[child]._dataObject
      # Set the primary key for any new child records
      fk = {}
      for i in range(len(c._masterfields)):
        fk[c._detailfields[i]] = self.getField(c._masterfields[i])

      self._cachedDetailResultSets[child].post(foreign_keys=fk)


  # Sets the ResultSet associated with this master record
  def addDetailResultSet(self, resultSet):
    self._cachedDetailResultSets[resultSet._dataObject] = resultSet


  ###
  ### Methods below should be over-written by Vendor Specific functions
  ###

  # Post any changes to database
  def _postChanges (self):
    """
    Post any changes (deletes, inserts, and updates) to the database.
    Descendants can either override this function, or the three functions
    _postDelete, _postInsert, and _postUpdate.

    _postChanges is guaranteed to be only called for records that have pending
    changes. Implementations of this function can check the kind of pending
    change by querying the _deleteFlag, _insertFlag, and _updateFlag instance
    variables.
    """
    if self._deleteFlag:
      self._postDelete ()

    elif self._insertFlag:
      self._postInsert (self._fields)

    elif self._updateFlag:
      modifiedFields = {}
      for field in (self._modifiedFlags.keys ()):
        modifiedFields [field] = self._fields [field]
      self._postUpdate (modifiedFields)
      self._modifiedFlags = {}

    self._deleteFlag = False
    self._insertFlag = False
    self._updateFlag = False

  def _postDelete (self):
    """
    Post a deletion to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postDelete is guaranteed to be only called for records that pend a
    deletion.
    """
    pass

  def _postInsert (self, fields):
    """
    Post an insert to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postInsert is guaranteed to be only called for records that pend an
    insert (i.e. that were newly created).

    @param fields: a dictionary with field names as keys and field values as
        values.
    """
    pass

  def _postUpdate (self, fields):
    """
    Post an update to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postUpdate is guaranteed to be only called for records that pend an
    update (i.e. for existing records that were modified).

    @param fields: a dictionary with field names as keys and field values as
        values.
    """
    pass
