#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# datasources/drivers/ldap/Driver.py
#
# DESCRIPTION:
# Implementation of dbdriver for use with LDAP Servers.
#
# NOTES:
#
# to handle the differences between relational and hirachical databases
# a special syntax is required.
#
# PARAMETERS:
#    host   f.e. ldap.gnue.org
#    port   f.e. 631
#    basedn f.e.  dc=gnue,dc=org
#    scope  := [default|subtree|onelevel|base]
#
# HISTORY:
#

VERSION="0.0.1"

try:
  from gnue.common.apps import GDebug
  from gnue.common.datasources import GDataObjects, GConditions, GConnections
except: 
  from gnue.common import GDebug,GDataObjects,GConditions

import string
import types
import md5
import sys

try:
  import ldap
except ImportError:
  raise GConnections.DependencyError, ('python-ldap',
                                       'http://python-ldap.sourceforge.net/')

## TODO: add an option to parse multiple entries


class LDAP_Cursor:
  def __init__(self,dataCon,searchdn,primarykey,data):
    self._dataCon=dataCon
    self._searchdn=searchdn
    self._primarykey=primarykey
    self._data=data

  def fetch(self):    
    # get the next dn in the dataset
    # the LDAP data structure is a list of tuples, each tuple consist of a
    # dn and a dictionary
    
    (dn,valuelist)=self._data.pop()

    # modify valuelist by joining parameterlist into one parameter
    dict={}          
    for f in valuelist.keys():
      # lowercase fieldnames
      dict[string.lower(f)]=string.join(valuelist[f],"|")
   
    # return the data
    return (dn,dict)    
    

class LDAP_DataConnection:
  # TODO: on timeout error: reconnect
  def __init__(self,ldapCon,basedn,scope):
    self._ldapCon=ldapCon
    self._basedn=basedn
    self._scope=scope

  def search(self,searchdn,filter,primarykey):
    
    dn = "%s,%s" % (searchdn, self._basedn)

    try:
      result = self._ldapCon.search_s(dn,self._scope,filter)

    except ldap.NO_SUCH_OBJECT:
      result = {}

    return LDAP_Cursor(self,searchdn,primarykey,result)

  def triggerSearch(self,searchdn,filter):
    
    dn = "%s,%s" % (searchdn, self._basedn)
    
    try:
      result = self._ldapCon.search_s(dn,self._scope,filter)

    except ldap.NO_SUCH_OBJECT:
      result = {}

    return result
      
  def cursor(self):
    return None

  def remove(self,dn):
    
    GDebug.printMesg(6, 'remove entry dn=%s' % dn)
    
    self._ldapCon.delete_s(dn)
    
  def insert(self,data,pk_field,pk_value,searchdn):
    
    # create mod list     
    modlist=[]
    
    for f in (data.keys()):
      # convert multiple fields like 'objectclass=sambaAccount|posixAccount'
      # into a list
      values=string.splitfields('%s' % data[f],"|")      
      modlist.append((f,values))

    # create new DN
    dn="%s=%s,%s,%s" % (pk_field,pk_value,searchdn,self._basedn)
      
    GDebug.printMesg(6, 'add new entry dn=%s fields=%s' % (dn,modlist))
    
    self._ldapCon.add_s(dn,modlist)      

  def changePK(self,dn,pk_field,pk_value,searchdn):

      # build new DN
      newdn="%s=%s,%s,%s" % (pk_field,pk_value,searchdn,self._basedn)
      
      GDebug.printMesg(6, 'FQDN rename operation change fqdn=%s -> %s' % \
                       (dn,newdn))
      
      dataCon.modrdn_s(dn,newdn)

      return newdn
  
  def update(self,dn,data,pk_field,pk_value,searchdn):
        
    # create mod list     
    modlist=[]
    for f in (data.keys()):
      # convert multiple fields like 'objectclass=sambaAccount|posixAccount'
      # into a list
      values=string.splitfields('%s' % data[f],"|")      
      modlist.append((ldap.MOD_REPLACE,f,values))

      
    GDebug.printMesg(6, 'modified entry dn=%s fields=%s' % (dn,modlist))
    
    self._ldapCon.modify_s(dn,modlist)      

## TODO: Modified LDAP Connection or cursor class, which iterates over attributs
##       of one entry instead over a list of entries
      
           
  

class LDAP_RecordSet(GDataObjects.RecordSet):

  
  def setFQDN(self,FQDN):
    self._fqdn=FQDN

  # HELPER FUNCTIONS
  def getFieldValue(self,fieldname):
    try:
      return self._fields[fieldname]
    except:
      try:
        return self._fields[string.lower(fieldname)]
      except:
        return ""
    
    
  def _postChanges(self):
    
    update_dict={}
    dataCon=self._parent._dataObject._dataConnection
    primarykey=self._parent._dataObject.primarykey
    searchdn=self._parent._dataObject.computeSearchDN(\
      self._parent._dataObject.table)
    
    if 1==1:
#    try:
      if self._deleteFlag:
        GDebug.printMesg(5, 'LDAP database driver: Instance deleted')
        # TODO: check result
        dataCon.remove(self._fqdn)
      
      elif self._insertFlag:
      
        data={}
        for field in (self._modifiedFlags.keys()):
          data[field]=self.getFieldValue(field)
          
        dataCon.insert(data,
                       primarykey,
                       self._fields[primarykey],
                       searchdn)
        
        GDebug.printMesg(5, 'LDAP database driver: new Instance ' + \
                         'created and inserted')
      
      elif self._updateFlag:

        ## TEST for modified primary key
        if self._modifiedFlags.has_key(self._parent._dataObject.primarykey):
          
          dataCon.rename(self._fqdn,
                         primarykey,
                         self._fields[primarykey],
                         searchdn)
      
        data={}
        print "UPDATE:"
        for field in (self._modifiedFlags.keys()):
          print "  field : %s " % field,
          print "data:  %s " % self._fields[field]
          data[field]=self._fields[field]
            
        dataCon.update(self._fqdn,
                       data,
                       primarykey,
                       self._fields[primarykey],
                       searchdn)
                  
        GDebug.printMesg(5, 'LDAP database driver: Instance updated')
        

##    except ldap.INVALID_SYNTAX:
##      raise GDataObjects.ConnectionError,_("Invalid Syntax \n%s\n%s ") % \
##            (sys.exc_info()[0], sys.exc_info()[1]) 
##    except:
##      raise GDataObjects.ConnectionError,_("Unknown error: \n%s\n%s ") % \
##            (sys.exc_info()[0], sys.exc_info()[1]) 
      
      
    self._updateFlag = 0
    self._insertFlag = 0
    self._deleteFlag = 0


  
# LDAP_ResultSet
#
# Notes:
# In the GEASv2 driver a CURSOR is simply the List handle returned
# via the query interface
#
class LDAP_ResultSet(GDataObjects.ResultSet): 
  def __init__(self, dataObject, cursor=None, \
        defaultValues={}, masterRecordSet=None): 
    GDataObjects.ResultSet.__init__(
           self,dataObject,cursor,defaultValues,masterRecordSet)    
    self._recordSetClass = LDAP_RecordSet
    
    # Populate field names
    self._fieldNames = self._dataObject.getFields(self._dataObject.table)           

    GDebug.printMesg(5, 'ResultSet created')


  def _loadNextRecord(self):
    retval=0
    # all records has to be loaded in cache during resultset initialisation
    if self._cursor:
       
      # load all data into the cache
      try:
        (dn,data) = self._cursor.fetch()
      except IndexError,TypeError:
        return 
      
      GDebug.printMesg(8, 'Add record %s with data %s' % (dn,data))
      
      record=self._recordSetClass(parent=self,initialData=data)
      
      record.setFQDN(dn)
      
      self._cachedRecords.append (record)
      
      self._recordCount=self._recordCount+1
      
      return 1    


class LDAP_DataObject(GDataObjects.DataObject):
  # ConditionalName (min args, max args, field creation, bypass function )
  #
  # Commented out conditionals are a bad thing

##  4. String Search Filter Definition

##   The string representation of an LDAP search filter is defined by the
##   following grammar, following the ABNF notation defined in [5].  The
##   filter format uses a prefix notation.

##        filter     = "(" filtercomp ")"
##        filtercomp = and / or / not / item
##        and        = "&" filterlist
##        or         = "|" filterlist
##        not        = "!" filter
##        filterlist = 1*filter
##        item       = simple / present / substring / extensible
##        simple     = attr filtertype value
##        filtertype = equal / approx / greater / less
##        equal      = "="
##        approx     = "~="
##        greater    = ">="
##        less       = "<="
##        extensible = attr [":dn"] [":" matchingrule] ":=" value
##                     / [":dn"] ":" matchingrule ":=" value
##        present    = attr "=*"
##        substring  = attr "=" [initial] any [final]
##        initial    = value
##        any        = "*" *(value "*")
##        final      = value
##        attr       = AttributeDescription from Section 4.1.5 of [1]
##        matchingrule = MatchingRuleId from Section 4.1.9 of [1]
##        value      = AttributeValue from Section 4.1.6 of [1]

##   The attr, matchingrule, and value constructs are as described in the
##   corresponding section of [1] given above.

##   If a value should contain any of the following characters

##           Character       ASCII value
##           ---------------------------
##           *               0x2a
##           (               0x28
##           )               0x29
##           \               0x5c
##           NUL             0x00

##   the character must be encoded as the backslash '\' character (ASCII
##   0x5c) followed by the two hexadecimal digits representing the ASCII
##   value of the encoded character. The case of the two hexadecimal
##   digits is not significant.
   
  conditionElements = {
       'add':             (2, 999, None,                    'None'    ),
       'sub':             (2, 999, None,                    'None'    ),
       'mul':             (2, 999, None,                    'None'    ),
       'div':             (2, 999, None,                    'None'    ),
       'and':             (1, 999, '(&%s)',                 '(&%s)'   ),
       'or':              (1, 999, '%s',                    '(|%s)'   ),
       'not':             (1,   1, '(!%s)',                 'None'    ),
       'negate':          (1,   1, '(-%s)',                 'None'    ),
       'eq':              (2,   2, '(%s=%s)',                None     ),
       'ne':              (2,   2, '(%s != %s)',             None     ),
       'gt':              (2,   2, '(%s > %s)',              None     ),
       'ge':              (2,   2, '(%s>=%s)',               None     ),
       'lt':              (2,   2, '(%s < %s)',              None     ),
       'le':              (2,   2, '(%s<=%s)',               None     ),
       'like':            (2,   2, '(%s=%s)',                None     ),
       # there is no LIKE in LDAP
       'notlike':         (2,   2, '(!(%s=%s))',             None     ),
       'between':         (3,   3, '%s BETWEEN %s AND %s',   None     )
       }

  def __init__(self):
    GDataObjects.DataObject.__init__(self)

    GDebug.printMesg (1,"LDAP database driver backend initializing")
    self._resultSetClass = LDAP_ResultSet
    self._DatabaseError = ldap.LDAPError

  def connect(self, connectData={}):
    
    GDebug.printMesg(1,"LDAP database driver connecting...")
    
    try:

      if hasattr(ldap,'initialize'):

        uri = "ldap://%s:%s/" % (connectData['host'],connectData['port'])

        GDebug.printMesg(3,"Initialize connection to %s." % uri)

        ldapCon = ldap.initialize(uri,
                                  trace_level=0,
                                  trace_file=sys.stdout)
      else:
        
        GDebug.printMesg(3,"Open connection to %s." % connectData['host'])

        if connectData.has_key("port"):
          
          ldapCon = ldap.open(connectData['host'],int(connectData['port']))
          
        else:
          
          ldapCon = ldap.open(connectData['host'])
      
      GDebug.printMesg (5,"Connection established.")
      
    except ldap.LDAPError , msg:
      
      print "Error connecting to LDAP server: %s" % msg 

    try:
      
      #
      #  a the moment python-ldap supports SIMPLE_AUTH only
      #
      
      GDebug.printMesg(3,"Authentificate against LDAP server")
      
      GDebug.printMesg(3,"User '%s' Password '%s' " % \
                       (connectData['_username'],
                        connectData['_password']))
      
      if connectData.has_key("bindmethod"):      

        ldapCon.bind_s(connectData['_username'],
                       connectData['_password'],
                       connectData['bindmethod'])
      else:
        ldapCon.simple_bind_s(connectData['_username'],
                              connectData['_password'])
        
      GDebug.printMesg (5,"Successfully bound to LDAP server.")

    except:
      
      raise GDataObjects.ConnectionError, "Error binding to LDAP server"
    
      # TODO: Try to add additional information to the username and bind again:
      # like username="admin" -> "cn=admin,dc=gnue,dc=org"
    


    # set up other important connection variables:
    # could also be read from/setup by an URI attribute

    # 1. BASE DN
        
    if not connectData.has_key('basedn'):
      raise GDataObjects.ConnectionError, "Missing BASEDN setting"

    basedn = connectData['basedn']

    # 2. SEARCH SCOPE
    
    if not connectData.has_key('scope'):
      scope=ldap.SCOPE_ONELEVEL
      
    else:
      if connectData['scope']=="default":
        scope=ldap.SCOPE_DEFAULT
        
      elif connectData['scope']=="base":
        scope=ldap.SCOPE_BASE
        
      elif connectData['scope']=="onelevel":
        scope=ldap.SCOPE_ONELEVEL
        
      elif connectData['scope']=="subtree":
        scope=ldap.SCOPE_SUBTREE
        
      else:
        scope=ldap.SCOPE_ONELEVEL
        GDebug.printMesg(3,"Wrong setting for scope parameter '%s'"\
                         % connectData['scope'])
        
    self._dataConnection = LDAP_DataConnection(ldapCon,
                                               basedn,
                                               scope)
    self._postConnect()
        
    
  def _postConnect(self):
    self.triggerExtensions = TriggerExtensions(self._dataConnection)
    
  # We only need the basics -- username and password -- to log in
  def getLoginFields(self): 
    return [['_username', 'User Name',0],['_password', 'Password',1]]
  

  def _buildQuery(self, conditions={},forDetail=None,additionalSQL=""):
    # Standardize incomming conditions as a GConditions structre
    if type(conditions) == types.DictType:
        cond = GConditions.buildConditionFromDict(conditions)
    else:
        cond = conditions

    query = self.__conditionToLDAPQuery(None,cond._children[0])
    
    GDebug.printMesg(7,'Full query in : %s' % query)
    
    return query


  def __conditionToLDAPQuery(self, queryObject, conditionTree, count=1):
      if type(conditionTree) != types.InstanceType:
        if conditionTree == None:
          return ""
        elif type(conditionTree) == types.StringType:
          return conditionTree
        else:      
          raise GConditions.ConditionNotSupported, \
                "This driver only accepts condition Trees (the type was: %s )"%\
                type(conditionTree)
      else:

        otype = string.lower(conditionTree._type[2:])
        if otype == 'cfield':
          # TODO care for special fields here also
          return "%s" % conditionTree.name
        elif otype == 'cconst':
          # return the value with all % changed to *
          # ldap don't understand LIKE syntax's %
          return string.replace("%s" % conditionTree.value,"%","*")
        
        elif otype == 'param':          
          v = conditionTree.getValue()
          return (v == None and "NULL") or \
                 ("'%s'" % conditionTree.getValue())
          
          # If the object is a conditional object then process
          # it's children to build query
        elif self.conditionElements.has_key(otype):
          # process children
          for i in range(0, len(conditionTree._children)):
            conditionTree._children[i] = \
                 self.__conditionToLDAPQuery(queryObject,
                                             conditionTree._children[i])
          # check for too less parameters 
          if len(conditionTree._children) < self.conditionElements[otype][0]:
            raise GConditions.ConditionError, \
                  _('Condition element "%s" expects at least %s arguments; found %s') % \
                  (otype, self.conditionElements[otype][0], len(conditionTree._children))
          # check for too much parameters 
          if len(conditionTree._children) > self.conditionElements[otype][1]:
            raise GConditions.ConditionError, \
                  _('Condition element "%s" expects at most %s arguments; found %s') % \
                  (otype, self.conditionElements[otype][1], len(conditionTree._children))
          # check
          if len(conditionTree._children) > self.conditionElements[otype][0]:
            return self.conditionElements[otype][2] % \
                 string.join(conditionTree._children,"")
          else:
            return self.conditionElements[otype][2] % \
                 tuple(conditionTree._children)
        else:
          raise GConditions.ConditionNotSupported, \
                _('Condition clause "%s" is not supported by this db driver.') % otype

  def computeSearchDN(self, tablename):
    #
    # split table into single attributs (cn_ou_People%cn
    #                                  -> ou=People+basedn
    #                                   + objectclass=posixAccount
    #
    #  TODO: allow more than one deeper search layer, possibly split
    #        layer and search class by other character than "_" or
    #        even remove search class out of table name.
    
    searchdn=""

    a=string.splitfields(tablename,'_')
    
    if len(a) % 2 == 1:
      raise GDataObjects.ConnectionError, "Cannot create right search dn "+\
            " from table name, there should be an odd number of parts"+\
            " seperated by '_' in the table name."
      # TODO: Replace this with a way to access attributs

    count=len(a)

    seperator = ""
    
    if count>0:
      searchdn="%s=%s%s%s" % (a[count-2],a[count-1],seperator,searchdn)
      seperator=", "
      count = count - 2

    GDebug.printMesg (5,"Converting tablename '%s' into search DN '%s'." % \
                      (tablename,searchdn))
  
    return searchdn
  

                      
  def _createEmptyResultSet(self, readOnly=0, masterRecordSet=None):
    return self.createResultSet(readOnly=readOnly,\
                                conditions=GConditions.GCimpossible,\
                                masterRecordSet=masterRecordSet)

  def _createResultSet(self, conditions={}, readOnly=0, masterRecordSet=None,sql=""):

    searchdn = self.computeSearchDN(self.table)
    filter   = ""
    
    if conditions:
      GDebug.printMesg (5,"Setting Conditions ...")
      
      filter=self._buildQuery(conditions)
      
      GDebug.printMesg (6," %s -> %s" % (conditions, filter))

    if hasattr(self,"order_by"):
      GDebug.printMesg (5,"FIXME: Sorting for LDAP not implemented "+\
                        "(order_by='%s')" % self.order_by)

    if not hasattr(self,"primarykey"):
      self.primarykey = 'cn'
      
    GDebug.printMesg (5,"Use '%s' as FQDN base field" %\
                      self.primarykey)

    try:
      GDebug.printMesg (0,"Started search for matching LDAP entries " +\
                        "(DN=%s,Filter=%s)" % (searchdn , filter))

      cursor=self._dataConnection.search(searchdn, filter, self.primarykey)
            
    except:      
      raise GDataObjects.ConnectionError, "Error searching ldap database \n"+\
            "(%s,%s)" % (sys.exc_info()[0], sys.exc_info()[1])

    rs = self._resultSetClass(self, cursor=cursor,
                              masterRecordSet=None)

    # set Resultset to readonly 
    rs._readonly = readOnly
      
    return rs

  #
  # ATTENTION !!!
  #
  # LDAP has no transaction support. that means that changes
  # will only be written to LDAP is commit is called. i.e.
  # through the _post method of the RecordSet class. 
  
  def commit(self): 
    GDebug.printMesg (5,"LDAP database driver: fake commit")

  def rollback(self): 
    GDebug.printMesg (5,"LDAP database driver: fake rollback")



##  TODO: Introspection should return a list of table names, which are similar to
##  the tree structure of the ldap database, i.e.
##    base dn: dc=gnue,dc=org
##    tree:         |-- People
##                  |     |--- VIP
##                  |     |--- Users
##                  |-- Groups

##    -> ou_People, ou_People_ou_VIP,  ou_People_ou_Users, ou_Groups




  def getSchemaTypes(self):
      return [('object',_('Objects'),1)]

  # Return a list of Schema objects
  def getSchemaList(self, type=None):
      includeObjects = (type in ('object','sources', None))
      
      list = []
      for classname in self._dataConnection.classes:
        list.append(self.getSchemaByName(classname))
      return list

  def getSchemaByName(self, name, type=None): 
      classdef = self._dataConnection.getFullClassDefinition(str(name))
      schema = GDataObjects.Schema(attrs={'id':string.lower(classdef.name),
                                          'name':classdef.name,
                                          'type':'object'},
                                   getChildSchema=self.__getFieldSchema)
      return schema
  
  # Get fields/methods of a GEAS object
  def __getChildSchema(self, parent):
      list = []
      c = con.getFullClassDefinition( classnames[idx-1] )
      
  def getFields(self, name):
      # no introspection support for now
      return self._fieldReferences.keys()



supportedDataObjects = { 
  'object': LDAP_DataObject
}


class TriggerExtensions:

  def __init__(self, connection):
    self.__connection = connection
    

  def ldapsearch(self,searchdn,filter):
    return self.__connection.triggerSearch(searchdn,filter)
