# GNU Enterprise RPC interface - Client adapter
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: client.py 5657 2004-04-06 20:45:41Z reinhard $

import string

from gnue.common.apps import GDebug, plugin

# =============================================================================
# Public functions
# =============================================================================

# -----------------------------------------------------------------------------
# Attach to a client driver
# -----------------------------------------------------------------------------

def attach (interface, params):
  driver = plugin.find (interface, 'gnue.common.rpc.drivers', 'ClientAdapter')
  return driver.ClientAdapter (params)

# =============================================================================
# Exceptions
# =============================================================================

# -----------------------------------------------------------------------------
# The basis of all rpc.client exceptions
# -----------------------------------------------------------------------------

class Error (gException):
  pass

# -----------------------------------------------------------------------------
# Requested adapter does not exist
# -----------------------------------------------------------------------------

class InvalidAdapter (Error):
  pass

# -----------------------------------------------------------------------------
# The requested adapter could not initialize. Perhaps the
# supplied parameters do not point to a valid server, etc.
# -----------------------------------------------------------------------------

class AdapterInitializationError (Error):
  pass

# -----------------------------------------------------------------------------
# The parameters supplied to the adapter are not in the
# correct format, or all the needed parameters were not
# supplied.
# -----------------------------------------------------------------------------

class AdapterConfigurationError (AdapterInitializationError):
  pass

# -----------------------------------------------------------------------------
# Parent for all caller errors
# -----------------------------------------------------------------------------

class ProgrammingError (Error):
  pass

# -----------------------------------------------------------------------------
# A client application requested a service that is not
# exposed by the server.
# -----------------------------------------------------------------------------

class InvalidService (ProgrammingError):
  pass

# -----------------------------------------------------------------------------
# An invalid parameter was passed to a RPC
# -----------------------------------------------------------------------------

class InvalidParameter (ProgrammingError):
  pass

# -----------------------------------------------------------------------------
# An error which has occured on the server and is raised again on the client
# -----------------------------------------------------------------------------

class DistantError (Error):
  """
  Indicates an exception that happened on the server side.

  If e is an exception of this type, then e.message is the message text of the
  exception that occured on the server side, e.type is a string containing the
  name of the type of the server side exception, and e.detail is the complete
  server side traceback.
  """
  def __init__ (self, traceback):
    self.detail = traceback
    self.type = ''
    message = ''

    # Search for start of Exception type/value part
    list = string.split (traceback, '\n')
    print list
    for i in range (1, len (list) - 1):
      if len (list [i]) > 0 and list [i] [0] != ' ':
        info = string.split (list [i], ': ', 1)
        if len (info) == 2:
          self.type = info [0]
          message = string.join ([info [1]] + list [i+1:], '\n')
          break
    Error.__init__ (self, message)

# =============================================================================
# Self test code - requires server.py running
# =============================================================================

if __name__ == '__main__':

  import traceback
  from gnue.common.rpc import client # to recogize DistantError

  connection = attach ('xmlrpc', {})

  obj = connection.request ('test')

  print 'stringtest:', repr (obj.stringtest ('This'))
  print 'ustringtest:', (obj.ustringtest (u'\xe0\xe2\xe1\xe4')).encode ('utf-8')
  print 'inttest: 21 * 2 =', repr (obj.inttest (21))
  print 'floattest: 123.45 * 2 =', repr (obj.floattest (123.45))
  print 'datetimetest:', repr (obj.datetimetest ())
  print 'booltest:', repr (obj.booltest ())
  subobj = obj.objtest ()
  print 'objtest:', repr (subobj)
# print 'subobj.test'
# subobj.test ()
  del subobj

  print 'testing exception ...'
  try:
    obj.exceptiontest ()
  except client.DistantError, e:
    print "-" * 70
    print "exception message:", e
    print "-" * 70
    print "local traceback:"
    traceback.print_exc ()
    print "-" * 70
    print "remote exception type:", e.type
    print "-" * 70
    print "remote exception detail:", e.detail
    print "-" * 70

  print 'shutting donwn server ...'
  try:
    # This will raise an exception because the server will not even answer any
    # more. Need to find a better way to shutdown the server.
    obj.shutdown ()
  except:
    pass
