#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: Base.py 5665 2004-04-07 11:04:32Z reinhard $

import string
import types
from mx.DateTime import now

# =============================================================================
# Exceptions
# =============================================================================

class ProcessorError (gException):
  """
  This is the base exception class for all processor related exceptions.
  """
  pass



# =============================================================================
# Base class for GNUe Schema Definition processors
# =============================================================================
class BaseProcessor:
  """
  This is the base class for GNUe Schema Definition processors. Such a
  processor will be fed with TableDefinition- and DataDefinition-objects and
  translates them according to it's language/dialect. It's doing this by
  populating the Definition-object's list-properties. 
  (see gnue.common.schema.scripter.Definition)
  """

  MAX_NAME_LENGTH    = 30
  MAX_LINE_LENGTH    = 78

  COMMENT_BEGIN      = "-- "
  COMMENT_END        = ""
  COMMENT_SINGLELINE = 1

  END_COMMAND        = ""       # Symbol used to terminate a command
  END_BATCH          = ""       # Symbol used to terminate a command-sequence

  QUOTE_CHAR         = ""
  ALTER_MULTIPLE     = True     # ALTER TABLE can handle multiple fields


  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  def __init__ (self, destination, source = None):
    self.destination = destination
    self.source      = source
    self.encoding    = "UTF-8"


  # ---------------------------------------------------------------------------
  # Find a method implementation in this class or any superclass of this class
  # ---------------------------------------------------------------------------
  def __findMethod (self, aClass, aMethod):
    if aClass.__dict__.has_key (aMethod):
      return aClass.__dict__ [aMethod]
    else:
      for base in aClass.__bases__:
        result = self.__findMethod (base, aMethod)
        if result is not None:
          return result

    return None
  

  # ---------------------------------------------------------------------------
  # Create an identifier for a sequence-like thing
  # ---------------------------------------------------------------------------
  def _getSequenceName (self, tablename, gsObject):
    """
    Create a name for a sequence-like object using 'tablename' and 'gsObject'.
    """
    res = ""

    if hasattr (gsObject, "name"):
      res = "%s_%s_seq" % (tablename, gsObject.name)

    if res == "" or len (res) > self.MAX_NAME_LENGTH:
      res = "%s_%d_seq" % (tablename, 
                           gsObject._parent._children.index (gsObject))

    if len (res) > self.MAX_NAME_LENGTH:
      res = "%s_%s_seq" % (tablename, id (gsObject))

    if len (res) > self.MAX_NAME_LENGTH:
      res = "seq_%s" % id (gsObject)

    return res


  # ---------------------------------------------------------------------------
  # Write a string to the destination using the specified encoding
  # ---------------------------------------------------------------------------
  
  def _writeText (self, text):
    """
    This method writes the 'text' (string/list) to the destination using 
    encoding. If text is a sequence it will be joined with newlines first.
    """

    if isinstance (text, types.ListType):
      astr = string.join (text, "\n") + u"\n"
    else:
      astr = text

    self.destination.write (astr.encode (self.encoding))

  
  # ===========================================================================
  # Schema support methods
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # fully qualify a field with name and datatype
  # ---------------------------------------------------------------------------

  def _qualify (self, gsField):
    """
    This method qualifies 'gsField' by concatenating the fieldname and it's
    translated datatype (see _translateType ()).
    """
    return "%s %s" % (gsField.name, self._translateType (gsField))


  # ---------------------------------------------------------------------------
  # get an apropriate representation for gsField's datatype
  # ---------------------------------------------------------------------------

  def _translateType (self, gsField):
    """
    Find a method for datatype translation of gsField.type in the current
    class or any of it's superclasses and return it's result.
    """
    aMethod = self.__findMethod (self.__class__, gsField.type)
    if aMethod is None:
      raise AttributeError, _("%s instance has no attribute %s.") % \
                             (self.__class__.__name__, gsField.type)
    else:
      return aMethod (self, gsField)


  # ---------------------------------------------------------------------------
  # Process the fields-sequence of a table definition 
  # ---------------------------------------------------------------------------
  
  def _processFields (self, tableDef):
    """
    This function iterates over all fields in table definition and calls the
    _processField () function on it. A processor has to override this last
    function to do the actual work on fields.
    """
    for field in tableDef.fields:
      self._processField (tableDef, field, field == tableDef.fields [-1])


  # ---------------------------------------------------------------------------
  # Virtual: Process a single field of a table definition
  # ---------------------------------------------------------------------------

  def _processField (self, tableDef, gsField, isLast):
    """
    A processor can override this method to translate a single field. The
    argument 'isLast' is set to True if gsField is the last field in the
    collection.
    """
    pass


  # ---------------------------------------------------------------------------
  # Virtual: Process the primary key of a table definition
  # ---------------------------------------------------------------------------

  def _processPrimaryKey (self, tableDef):
    """
    A processor can override this method to translate a primary key definition.
    """
    pass

  
  # ---------------------------------------------------------------------------
  # Process the indices of a table definition
  # ---------------------------------------------------------------------------

  def _processIndices (self, tableDef):
    """
    A processor can override this method to translate all index definitions of
    a table definition.
    """
    for index in tableDef.indices.values ():
      self._processIndex (tableDef, index)


  # ---------------------------------------------------------------------------
  # Virtual: Process a index definition
  # ---------------------------------------------------------------------------
  
  def _processIndex (self, tableDef, indexDef):
    """
    A processor can override this method to translate a single index
    definition.
    """
    pass


  # ---------------------------------------------------------------------------
  # Process the constraints of a table definition
  # ---------------------------------------------------------------------------

  def _processConstraints (self, tableDef):
    """
    A processor can override this method to translate all constraints of a
    table definition.
    """
    for constraint in tableDef.constraints.values ():
      self._processConstraint (tableDef, constraint)


  # ---------------------------------------------------------------------------
  # Virtual: process a single constraint of a table definition
  # ---------------------------------------------------------------------------

  def _processConstraint (self, tableDef, constraint):
    """
    A processor can override this method to translate a single
    constraintdefinition.
    """
    pass


  # ---------------------------------------------------------------------------
  # Translate a table-definition into it's phase instances
  # ---------------------------------------------------------------------------

  def translateTableDefinition (self, tableDef):
    """
    This function calls all _process*-functions on the given table definition
    and finally writes the definition to the destination.
    """
    # Process all parts of the table definition
    self._processFields (tableDef)

    if tableDef.action == 'create' and tableDef.primaryKey is not None:
      self._processPrimaryKey (tableDef)

    if len (tableDef.indices.keys ()):
      self._processIndices (tableDef)

    if len (tableDef.constraints.keys ()):
      self._processConstraints (tableDef)

    if len (self.END_BATCH):
      for phase in tableDef.phases.values ():
        phase.epilogue.append (self.END_BATCH)


  # ---------------------------------------------------------------------------
  # write a phase definition to the destination if it exists
  # ---------------------------------------------------------------------------
  def writePhase (self, tableDef, phase):
    """
    """
    if tableDef.phases.has_key (phase):
      tableDef.phases [phase].writeDefinition (self.destination, self.encoding)


  # ===========================================================================
  # Data support methods
  # ===========================================================================


  # ---------------------------------------------------------------------------
  # Find a Data transformation service handler for a fields data-type
  # ---------------------------------------------------------------------------

  def _dts_type (self, gsField):
    """
    This function looks for a data transformation handler function for the
    gsField's data type. The function first looks in the current class, and if
    not successfull asks all superclasses for such a method. On success the
    method will be called with gsField returning it's result. If no
    handler was found, the native value object will be returned as is.

    NOTE: data transformation service functions must have a name of
    'dts_<type>' where <type> stands for the acutal datatype, e.g. "dts_date"
    is a transformation handler for date-values.
    """
    aMethod = self.__findMethod (self.__class__, "dts_%s" % gsField.dataType)
    if aMethod is not None:
      return aMethod (self, gsField)
    else:
      return gsField.value


  # ---------------------------------------------------------------------------
  # Virtual: Process all rows of a data definition
  # ---------------------------------------------------------------------------

  def _processDataRows (self, dataDef, tableDef):
    """
    A processor can override this method to translate all data rows held
    by the data definition object.
    """
    pass


  # ---------------------------------------------------------------------------
  # Write a data definition to the destination
  # ---------------------------------------------------------------------------
  def writeData (self, dataDef, tableDef = None):
    """
    Process all data rows in the data definition and writes it to the
    destination.
    """
    self._processDataRows (dataDef, tableDef)

    if len (self.END_BATCH):
      dataDef.epilogue.append (self.END_BATCH)

    dataDef.writeDefinition (self.destination, self.encoding)




  # ===========================================================================
  # Miscellaneous public methods
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Set the client encoding
  # ---------------------------------------------------------------------------

  def client_encoding (self, encoding = None):
    """
    This function creates a comment describing the current encoding of the SQL
    script generated by the processor. A processor would like to override this
    function for changing the client encoding. This function will be called by
    the scripter.
    """
    if encoding is not None:
      self.encoding = encoding

    self._writeText (self.comment (u_("Client encoding set to '%s'") % \
                                   self.encoding))

  
  # ---------------------------------------------------------------------------
  # Comment all lines from text
  # ---------------------------------------------------------------------------

  def comment (self, text):
    """
    Create a sequence of 'commented' lines given in the sequence 'text'. Use
    the COMMENT_* constants to control this functions behaviour.
    """
    body   = []
    result = []
    
    ruler = u"=" * (self.MAX_LINE_LENGTH - len (self.COMMENT_BEGIN) - \
                                           len (self.COMMENT_END))
    body.append (ruler)
    if isinstance (text, types.ListType):
      body.extend (text)
    else:
      body.extend (text.split ("\n"))
    body.append (ruler)

    if self.COMMENT_SINGLELINE:
      for line in body:
        result.append ("%s%s%s" % (self.COMMENT_BEGIN, line, self.COMMENT_END))

    else:
      space = " " * len (self.COMMENT_BEGIN)
      first = True
      for line in body:
        if first:
          line = "%s%s" % (self.COMMENT_BEGIN, line)
          first = False
        else:
          if len (line):
            line = "%s%s" % (space, line)

        result.append (line)

      if len (result):
        result [-1] += " %s" % self.COMMENT_END

    return result


  # ---------------------------------------------------------------------------
  # Virtual: called on start of a dump
  # ---------------------------------------------------------------------------

  def startDump (self):
    """
    This method is called by the scripter on start of a dump. Use it to do 
    'per-generation' actions, e.g. set encoding 
    """
    today = "%s UTC" % now ()
    text = u_("This file was generated by gnue-schema\nfrom %s on %s.") % \
           (self.source, today)

    self._writeText (u"\n")
    self._writeText (self.comment (text.splitlines () + \
                     ["", u_("Do NOT edit manually!")]))
    self._writeText (u"\n")



  # ---------------------------------------------------------------------------
  # Virtual: called on end of a schema dump
  # ---------------------------------------------------------------------------

  def finishDump (self):
    """
    This method is called by the scripter at the end of a dump.
    """
    pass


  # ---------------------------------------------------------------------------
  # Virtual: called on start of a schema dump
  # ---------------------------------------------------------------------------

  def startSchema (self):
    """
    This method is called by the scripter on start of a schema dump. Use it to
    take initial actions.
    """
    pass


  # ---------------------------------------------------------------------------
  # Virtual: called on start of a data dump
  # ---------------------------------------------------------------------------

  def startData (self):
    """
    This method is called by the scripter on start of a data dump. Use it to
    take initial actions.
    """
    pass


  # ---------------------------------------------------------------------------
  # Depreciated 'timestamp': we won't use timestamp any longer
  # ---------------------------------------------------------------------------

  def timestamp (self, gsField):
    """
    Depreciated - use datatype 'datetime' instead
    """
    print _("WARNING: datatype 'timestamp' is depreciated. "
            "Use datetime instead.")
    return self.datetime (gsField)


  # ---------------------------------------------------------------------------
  # Depreciated 'text': we won't use text any longer
  # ---------------------------------------------------------------------------

  def text (self, gsField):
    """
    Depreciated - use datatype 'string' without a length attribute instead
    """
    print _("WARNING: datatype 'text' is depreciated. "
            "Use 'string' without length instead.")
    gsField.type = "string"
    return self.string (gsField)


  # ---------------------------------------------------------------------------
  # quote a string
  # ---------------------------------------------------------------------------

  def quoteString (self, aString):
    """
    This function escapes the given string and puts it into quotes.
    """
    return "%s%s%s" % (self.QUOTECHAR, self.escapeString (aString),
                       self.QUOTECHAR)


  # ---------------------------------------------------------------------------
  # escape a string
  # ---------------------------------------------------------------------------

  def escapeString (self, aString):
    """
    Processors override this function to make a string save for output, e.g. by
    escaping all special characters.
    """
    return aString
