#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: Base.py 7204 2005-03-14 19:16:03Z johannes $

import sys
import traceback
import string

from gnue.common.logic.language import ImplementationError, AbortRequest
from gnue.common.logic.NamespaceCore import NamespaceElement, NamespaceFunction


# =============================================================================
# Base class for LanguageAdapters
# =============================================================================

class BaseLanguageAdapter:
  """
  This is the base class for language adapters. A language adapter has to
  provide a public function for creation of new execution contexts.
  """

  # ---------------------------------------------------------------------------
  # Create and return a new execution context
  # ---------------------------------------------------------------------------

  def createNewContext (self):
    """
    Abstract: Create a new execution context.
    """
    raise ImplementationError, (self.__class__, 'createNewContext ()')



# =============================================================================
# Base class for execution contexts
# =============================================================================

class BaseExecutionContext:
  """
  An execution context provides an environment where code can be transformed
  and executed.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, runtime = None):
    self._runtime    = runtime
    self.shortname   = "unknown_executioncontext"
    self.description = "There is no description provided"


  # ---------------------------------------------------------------------------
  # Merge a namespace into the execution contexts namespace
  # ---------------------------------------------------------------------------

  def defineNamespace (self, addNS, asGlobal = False):
    """
    Merge the given namespace @addNS into the execution context. This function
    is doing this using bindFunction () and bindObject () depeding on the
    namespace elements type.
    """
    for (name, value) in addNS.items ():
      if name is not None:
        if isinstance (value, NamespaceFunction):
          self.bindFunction (name, value, asGlobal)

        if isinstance (value, NamespaceElement):
          self.bindObject (name, value, asGlobal)


  # ---------------------------------------------------------------------------
  # Bind an object into the contexts namespace using the given name
  # ---------------------------------------------------------------------------

  def bindObject (self, name, aObject, asGlobal = False):
    """
    Abstract: A descendant overrides this function to bind a given object into
    the local or global namespace.
    """
    raise ImplementationError, (self.__class__, 'bindObject ()')


  # ---------------------------------------------------------------------------
  # Bind a function into the contexts namespace using the given name
  # ---------------------------------------------------------------------------

  def bindFunction (self, name, aFunction, asGlobal = False):
    """
    Abstract: A descendant overrides this function to bind a given function 
    with into the local or global namespace.
    """
    raise ImplementationError, (self.__class__, 'bindFunction ()')


  # ---------------------------------------------------------------------------
  # Bind an element into the contexts builtin-namespace
  # ---------------------------------------------------------------------------

  def bindBuiltin (self, name, anElement):
    """
    Abstract: A descendant overrides this function to bind a given element into
    the builtin-namespace of the context.
    """

    raise ImplementationError, (self.__class__, 'bindBuiltin ()')


  # ---------------------------------------------------------------------------
  # Create a new function instance 
  # ---------------------------------------------------------------------------

  def buildFunction (self, name, code, parameters = {}):
    """
    Abstract: Create a new instance of a virtual function and prepare it's 
    code.
    """
    raise ImplementationError, (self.__class__, 'buildFunction ()')


  # ---------------------------------------------------------------------------
  # Release an execution context
  # ---------------------------------------------------------------------------

  def release (self):
    """
    Release an execution context: remove references in the namespace and the
    like.
    """

    pass



# =============================================================================
# Base class of a virtual function
# =============================================================================

class VirtualFunction:
  """
  This is the base class of virtual functions. Such an instance must be able to
  prepare a sourcecode and to execute it.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, context, name, code, parameters):
    self._context    = context
    self._name       = name
    self._code       = code
    self._parameters = parameters
    self._prepare ()


  # ---------------------------------------------------------------------------
  # Execute the function using the given arguments
  # ---------------------------------------------------------------------------

  def execute (self, *args, **params):
    """
    Execute the function using the given arguments. A descendant must override
    this function.
    """
    raise ImplementationError, (self.__class__, 'execute ()')



  # ---------------------------------------------------------------------------
  # Generate a user requested abort 
  # ---------------------------------------------------------------------------

  def requestAbort (self, message):
    """
    Abstract: Generate a user abort request.  Linked into the execution
    namespace so that scripts can request an abort.
    """
    raise AbortRequest(message)
  
  # ---------------------------------------------------------------------------
  # redirect a call of an instance to the execute () function.
  # ---------------------------------------------------------------------------

  def __call__ (self, *args, **params):
    """
    If an instance gets called, redirect this call to the execute () function.
    """
    return apply (self.execute, args, params)


  # ---------------------------------------------------------------------------
  # Prepare the given source code
  # ---------------------------------------------------------------------------

  def _prepare (self):
    """
    This function gets called on constructing an instance at could be used by a
    descendant to prepare the sourcecode, i.e. compile it.
    """
    pass

