# GNU Enterprise RPC interface - Server adapter
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: server.py 7460 2005-04-22 14:53:23Z reinhard $

from types import *

import string

from gnue.common.apps import errors, plugin
from gnue.common.utils.FileUtils import openResource

# =============================================================================
# Public functions
# =============================================================================

# -----------------------------------------------------------------------------
# Bind/export our services
# -----------------------------------------------------------------------------

# Attributes:
#
# rpcdef     The RPC definition loaded from an XML file. This can be
#              1) a string that contains the location of the XML file
#                 (can be a URL or pathname)
#              2) a file-handle (or file-like handle; e.g. StringBuffer)
#                 referencing an XML definition
#              3) a GnuRpc object created from GComm.loadDefinition()
#
# drivers    A dictionary that defines all the transports to be used. The
#            dictionary is of the form: {driver:params} where:
#              1) driver: A string containing the name of the commdriver
#                 (e.g., 'xmlrpc' or 'corba')
#              2) params: A dictionary containing parameters specific to
#                 the commdriver (e.g., {'port':'8888'} )
#
# bindings   A dictionary containing binding definitions. The dictionary
#            is of the form: {server:handler} where:
#              1) service: Name of the service being bound/exposed
#              2) handler: Method that when called returns a class
#                 instance i.e. {'DonutPlace',getDonutPlaceObj}

def bind (rpcdef, drivers, bindings):

  if isinstance (rpcdef, StringType):

    # Application supplied the location of their rpcdef.. load it
    fin = openResource (rpcdef)
    mapping = loadDefinition (fin, 1)
    fin.close ()

  elif hasattr (rpcdef, 'read'):

    # Application provided us a file-like object
    mapping = loadDefinition (rpcdef)

  else:

    # Otherwise, they must have specified a GnuRpc object
    mapping = rpcdef


  servers = {}

  for interface in drivers.keys ():
    params = drivers [interface]

    driver = plugin.find (interface, 'gnue.common.rpc.drivers', 'ServerAdapter')
    servers [interface] = driver.ServerAdapter (mapping, bindings, params)

  return servers

# To keep everything that a client/server app will need in this
# file, any convenience functions from the commdrivers directory.
# (i.e., a user app should never need to access the commdrivers
# directory.)

# -----------------------------------------------------------------------------
# Load an XML-based interface definition
# -----------------------------------------------------------------------------

from gnue.common.rpc.parser.Parser import loadDefinition

# =============================================================================
# Exceptions
# =============================================================================

# -----------------------------------------------------------------------------
# The requested adapter does not exist...
# -----------------------------------------------------------------------------

class InvalidAdapter (errors.AdminError):
  pass

# -----------------------------------------------------------------------------
# The requested adapter could not initialize. Perhaps the
# supplied parameters do not point to a valid server, etc.
# -----------------------------------------------------------------------------

class AdapterInitializationError (errors.AdminError):
  pass

# -----------------------------------------------------------------------------
# The parameters supplied to the adapter are not in the
# correct format, or all the needed parameters were not
# supplied.
# -----------------------------------------------------------------------------

class AdapterConfigurationError (AdapterInitializationError):
  pass

# -----------------------------------------------------------------------------
# Parent for all caller errors
# -----------------------------------------------------------------------------

class ProgrammingError (errors.SystemError):
  pass

# -----------------------------------------------------------------------------
# An invalid parameter was passed to a RPC
# -----------------------------------------------------------------------------

class InvalidParameter (ProgrammingError):
  pass


# =============================================================================
# Self test code
# =============================================================================

if __name__ == '__main__':

  import sys
  import time
  import StringIO
  import mx.DateTime

  class TestError (Exception):
    pass

  class subobject:
    def __init__ (self):
      print "Initializing subobject"

    def test (self):
      print "Testing subobject"

    def __del__ (self):
      print "Destroying subobject"

  class servertest:

    def stringtest (self, s):
      return '"%s" is a string' % s

    def ustringtest (self, u):
      return u'"%s" is a unicode string' % u

    def inttest (self, i):
      return i * 2

    def floattest (self, f):
      return f * 2

    def datetimetest (self):
      return mx.DateTime.now ()

    def booltest (self):
      return True

    def objtest (self):
      return subobject ()

    def exceptiontest (self):
      raise TestError, 'Message string'

    def shutdown (self):
      sys.exit (0)

  rpcdef = StringIO.StringIO ('<gnurpc>'
                                '<service name="test" binding="test">'
                                  '<method name="stringtest"/>'
                                  '<method name="ustringtest"/>'
                                  '<method name="inttest"/>'
                                  '<method name="floattest"/>'
                                  '<method name="datetimetest"/>'
                                  '<method name="booltest"/>'
                                  '<method name="objtest"/>'
                                  '<method name="exceptiontest"/>'
                                  '<method name="shutdown"/>'
                                '</service>'
#                               '<service name="subobject" binding="subobject">'
#                                 '<method name="test"/>'
#                               '</service>'
                              '</gnurpc>')

  if (len(sys.argv)==2) and \
     (sys.argv[1] in ['pyro','xmlrpc','socket','soap','xmlrpc.pw_xmlrpc','xmlrpc.py_xmlrpc']):
    drivers = {sys.argv[1]: {'port': 8765}}
  else:
    print """
GNUe RPC test server

Usage: gcvs server.py <transport>

"""
    raise SystemExit

  bindings = {'test': servertest}

  servers = bind (rpcdef, drivers, bindings)

  print 'Starting server ...'

  (servers.values () [0]).serve ()
