#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can` redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# SchemaNavigator.py
#
# DESCRIPTION:
# Class that implements a "Schema Viewer" (introspection) Tool Panel
#
# NOTES:
#
#  Schemas
#     +- gnue (GNUe Test Connection / PostgreSQL)
#         +-- Tables
#              +-- Table1
#                   +-- Field1
#                   +-- Field2
#                   +-- Field3
#                   +-- Field4
#              +-- Table2
#         +-- Views
#              +-- View1
#                   +-- Field1
#                   +-- Field2
#                   +-- Field3
#                   +-- Field4
#
#
#

__all__ = ['SchemaNavigator']

import sys, os, cPickle, traceback, StringIO, string, types
from wxPython.wx import *
from wxPython.gizmos import *
from wxPython.wx import __version__ as wxversion
from gnue.common.apps import GDebug
from gnue.common.datasources import GDataSource
from gnue.common.apps import RuntimeSettings
from gnue.common import events
from gnue.designer.base.ToolBase import *
from gnue.designer.base.ToolBar import *

class SchemaNavigator(ToolBase):

  runtime_section = "SchemaNavigator"
  default_dock = 'left-5'
  uses_toolbar = 1
  uses_feedback_bar = 1

  def init(self):
    self.connections = self.instance._app.connections

    self.connectionName = None
    self.connectionNames = self.connections.getConnectionNames()
    self.connectionNames.sort()
    self.connectionCache = {}

    self.current = None

    self.tree = tree = TreeCtrl(self, self)
    root = tree.AddRoot('Schema')
    tree.SetItemHasChildren(root, 1)
    for connection in self.connectionNames:
       ConnectionNode(self, tree, root, connection=connection)
##    tree.Expand(root)

    sizer = wxBoxSizer(wxVERTICAL)
    sizer.Add(tree, 1, wxEXPAND|wxALL, 6)
    self.SetAutoLayout(True)
    self.SetSizer(sizer)


    # EventAware provided by ToolBase
##    self.registerEventListeners({
##                       'ObjectSelected'      : self.onSetCurrentObject,
##                       'ObjectCreated'       : self.onCreateObject,
##                       'ObjectModified'      : self.onModifyObject,
##                       'ObjectDeleted'       : self.onDeleteObject,
##                      })

    self.expanded = 1
    EVT_TREE_SEL_CHANGED(self, self.tree.GetId(), self.OnSelectionChange)
    EVT_TREE_BEGIN_DRAG(self, self.tree.GetId(), self.OnBeginDrag)
    EVT_TREE_BEGIN_RDRAG(self, self.tree.GetId(), self.OnBeginRDrag)

    self.Layout()

  def createToolbar(self, parent):
    return _SchemaToolbar(parent, self.instance)

  # Since this class is the root node, we need to
  # emulate part of the Node class
  def getColumn(self, i=0):
    return ""

  def OnBeginDrag(self, event, right=0):
    do = wxCustomDataObject(wxCustomDataFormat("application/x-gnue-designer"))
    do.SetData(cPickle.dumps(self.buildDataObject(right),1))

    dropSource = wxDropSource(self)
    dropSource.SetData(do)
    self.instance.dispatchEvent('BeginUndoGroup')
    # win32 needs wxDrag_AllowMove to really enable DnD...
    result = dropSource.DoDragDrop(wxDrag_CopyOnly | wxDrag_AllowMove)
    self.instance.dispatchEvent('EndUndoGroup')

  def OnBeginRDrag(self, event):
    self.OnBeginDrag(event, right=1)

  def OnSelectionChange(self, event):
    try:
      o = self.tree.GetPyData(event.GetItem())
      if o.getColumn():
        self.setFeedback(o.getLabel() + ':  ' + o.getColumn())
      else:
        self.setFeedback('')
    except:
      pass

  def __addTable(self, object):
    # Find the "connection" node
    parent = self.tree.GetPyData(self.tree.GetItemParent(object.node))
    while not isinstance(parent, ConnectionNode):
      parent = self.tree.GetPyData(self.tree.GetItemParent(parent.node))

    rs = {}
    for attr, val in object.schema.__dict__.items():
      if attr[0] != '_' and type(val) != types.MethodType:
        rs[attr] = val

    rs['connection'] = parent.connection
    return rs


  def __addField(self, object):
    rs = {}
    for attr, val in object.field.__dict__.items():
      if attr[0] != '_' and type(val) != types.MethodType:
        rs[attr] = val
    return rs


  def buildDataObject(self, right=0):
    rs = []
    tableCache = {}

    # Add all our tables/fields to a list
    # in the format [ (table, [field, field]),
    #                 (table, [field, field]) ]
    for tid in self.tree.GetSelections():
      node = self.tree.GetPyData(tid)
      if isinstance(node, TableNode):
        tn = tableCache[node] = []
        rs.append((self.__addTable(node), tn))
      elif isinstance(node, FieldNode):
        table = self.tree.GetPyData(self.tree.GetItemParent(tid))
        try:
          tn = tableCache[table]
        except KeyError:
          tn = tableCache[table] = []
          rs.append((self.__addTable(table), tn))
        tn.append(self.__addField(node))

    # If a table was selected, but no fields,
    # then add all the fields to the cache
    for table, fields in tableCache.items():
      if not fields:
        if not table.expanded:
          table.expand()
        cookie = 0
        if wxversion[:3] == '2.6':
          child, cookie = self.tree.GetFirstChild(table.node)
        else:
          child, cookie = self.tree.GetFirstChild(table.node, cookie)
        while child.IsOk():
          fields.append(self.__addField(self.tree.GetPyData(child)))
          child, cookie = self.tree.GetNextChild(table.node, cookie)

    return [{'Type': 'SchemaDrop', 'Data': rs, 'ShowMenu': right}]


class TreeCtrl(wxTreeCtrl):

  def __init__(self, parent, tool):
    wxTreeCtrl.__init__(self, parent, -1,
##           style=wxTR_HAS_BUTTONS|wxTR_NO_LINES| wxTR_HIDE_ROOT|
##                 wxNO_BORDER|wxTR_MULTIPLE|wxTR_ROW_LINES)
           style=wxTR_HAS_BUTTONS|wxTR_HIDE_ROOT|wxTR_LINES_AT_ROOT|
                 wxSIMPLE_BORDER|wxTR_MULTIPLE|wxTR_FULL_ROW_HIGHLIGHT )

    self.tool = tool
    EVT_TREE_ITEM_EXPANDING(self, self.GetId(), self.__OnExpand)


  def __OnExpand(self, event):
    parent = event.GetItem()
    if self.GetChildrenCount(parent) > 0:
      return

    node = self.GetPyData(parent)
    if node and not node.expanded:
      node.expand()

    event.Skip()




######################################################################
#
# Various node classes
#

class Node:
  isleaf = 0
  def __init__(self, navigator, tree, parent, **params):
    self.__dict__.update(params)
    self.expanded = 0
    self.navigator = navigator
    self.tree = tree
    self.node = node = tree.AppendItem(parent, self.getLabel())
    tree.SetItemHasChildren(node, not self.isleaf)
    tree.SetPyData(node, self)

class ConnectionNode(Node):

  def expand(self):
    self.expanded = 1
    connection = self.navigator.connections.getConnection(self.connection, login=1)

    schema = connection.readSchema ()

    if schema is None:
      return

    for tables in schema.findChildrenOfType ('GSTables', False, True):
      SchemaTypeNode (self.navigator, self.tree, self.node, tables = tables)


  def getLabel (self):
    return self.connection

  def getColumn (self, i=0):
    data = self.navigator.connections.getConnectionParameters (self.connection)
    try:
      return data['comment'] + ' (' + data['provider'] + ')'
    except:
      return '(' + data['provider'] + ')'


class SchemaTypeNode (Node):
  def getLabel (self):
    return self.tables.name
  def getColumn (self, i=0):
    return ""
  def expand (self):
    self.expanded = 1

    for table in self.tables.findChildrenOfType ('GSTable', False, True):
      TableNode (self.navigator, self.tree, self.node, schema = table)


class TableNode(Node):
  def getLabel(self):
    return self.schema.name
  def getColumn(self, i=0):
    return ""
  def expand(self):
    self.expanded = 1
    for field in self.schema.findChildrenOfType ('GSField', False, True):
      FieldNode (self.navigator, self.tree, self.node, field = field)


class FieldNode (Node):
  isleaf = 1
  def getLabel(self):
    return self.field.name

  def getColumn(self):
    rv = ""
    data = []
    rv += self.field.type.capitalize ()
    try:
      data.append (str (self.field.length))
    except:
      pass
    try:
      data.append (str (self.field.precision))
    except:
      pass
##    if data:
##      rv += '; Size: ' + string.join(data,',')

    if not self.field.nullable:
      rv += "; Not Null"

    return rv

  def expand(self):
    self.expanded = 1



######################################################################
#
#  Toolbar
#

ToolbarMapping = 'SchemaNavigator:Refresh,tb_refresh'

class _SchemaToolbar(BaseToolBar):
##  ICON_SIZE=(16,16)
  def init(self):
    self.addToolbarMapping(ToolbarMapping)

