# GNU Enterprise Forms - Curses UI Driver - User Interface
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: UIdriver.py 7770 2005-08-02 02:43:03Z jamest $

import atexit
import curses
import string

from gnue.common.apps import i18n
from gnue.common.datasources import GLoginHandler
from gnue.forms.GFKeyMapper import vk, KeyMapper
from gnue.forms.uidrivers._base.UIdriver import GFUserInterfaceBase
from gnue.forms.uidrivers.curses import dialogs

# =============================================================================
# User interface object
# =============================================================================

class GFUserInterface (GFUserInterfaceBase):

  # ---------------------------------------------------------------------------
  # Initialize library
  # ---------------------------------------------------------------------------

  def __init__ (self, eventHandler, name = "Undefined", disableSplash = None,
                parentContainer = None, moduleName = None):

    GFUserInterfaceBase.__init__ (self, eventHandler, name, disableSplash,
                                  parentContainer, moduleName)

    self.__screen = curses.initscr ()
    atexit.register (curses.endwin)
    curses.raw ()
    curses.start_color ()

    # Define colors
    curses.init_pair (1, curses.COLOR_WHITE, curses.COLOR_BLUE)
    curses.init_pair (2, curses.COLOR_BLACK, curses.COLOR_WHITE)
    curses.init_pair (3, curses.COLOR_BLACK, curses.COLOR_CYAN)
    curses.init_pair (4, curses.COLOR_BLACK, curses.COLOR_WHITE)
    curses.init_pair (5, curses.COLOR_BLACK, curses.COLOR_CYAN)
    curses.init_pair (6, curses.COLOR_WHITE, curses.COLOR_RED)
    curses.init_pair (7, curses.COLOR_WHITE, curses.COLOR_BLUE)
    curses.init_pair (8, curses.COLOR_WHITE, curses.COLOR_YELLOW)
    curses.init_pair (9, curses.COLOR_WHITE, curses.COLOR_RED)

    self.attr = {}
    self.attr ['title']       = curses.color_pair (1)
    self.attr ['page']        = curses.color_pair (2)
    self.attr ['currentpage'] = curses.color_pair (3)
    self.attr ['background']  = curses.color_pair (4)
    self.attr ['entry']       = curses.color_pair (5)
    self.attr ['focusentry']  = curses.color_pair (6) + curses.A_BOLD
    self.attr ['disabled']    = curses.color_pair (5)
    self.attr ['status']      = curses.color_pair (7)
    self.attr ['fkeys']       = curses.color_pair (4)
    self.attr ['infomsg']     = curses.color_pair (8) + curses.A_BOLD
    self.attr ['warnmsg']     = curses.color_pair (9) + curses.A_BOLD
    self.attr ['errormsg']    = curses.color_pair (9) + curses.A_BOLD \
                                                      + curses.A_BLINK
    self.attr ['window 1']    = curses.color_pair (5)

    self.__currentForm = None

    # TODO: Add a concept of 'lookups' to the GF* layer (like canCOMMIT, ...)
    #       Meanwhile we're using Ctrl-W (hardcoded)
    self.lookupKey = 23


  # ---------------------------------------------------------------------------
  # Initialize user interface
  # ---------------------------------------------------------------------------

  def _initialize (self):

    self.widgetWidth = 1
    self.widgetHeight = 1
    self.textWidth = 1
    self.textHeight = 1

    self.__exiting = False

    KeyMapper.setUIKeyMap (self._keymap)
    KeyMapper.loadUserKeyMap ({'JUMPPROMPT': 'Ctrl-r'})

    # Find out all functions mapped to function keys
                                        # Both hold [eventname, enabled] pairs:
    self.__events_fkey = {}             #  by curses keycode
    self.__events_name = {}             #  by event name
    for fkey in range (curses.KEY_F1, curses.KEY_F20):
      if self.__shiftkeys.has_key (fkey):
        (key, shift, ctrl, meta) = self.__shiftkeys [fkey]
      else:
        (key, shift, ctrl, meta) = (fkey, False, False, False)
      name = KeyMapper.getEvent (key, shift, ctrl, meta)
      if name:
        event = [name, False]
        self.__events_fkey [fkey] = event
        self.__events_name [name] = event
        self.registerEventListeners ({'can'    + name: self.__canEvent,
                                      'cannot' + name: self.__cannotEvent})


  # ---------------------------------------------------------------------------
  # Activate the given form
  # ---------------------------------------------------------------------------

  def _activateForm (self, form, modal):

    self.__currentForm = form

  # ---------------------------------------------------------------------------
  # Key mapping for converting e.g. F13 into Shift-F3
  # ---------------------------------------------------------------------------

  __shiftkeys = {
    #                   unshifted         Shift  Ctrl   Meta
    #                   ------------------------------------
    curses.KEY_F13:    (curses.KEY_F3,    True,  False, False),
    curses.KEY_F14:    (curses.KEY_F4,    True,  False, False),
    curses.KEY_F15:    (curses.KEY_F5,    True,  False, False),
    curses.KEY_F16:    (curses.KEY_F6,    True,  False, False),
    curses.KEY_F17:    (curses.KEY_F7,    True,  False, False),
    curses.KEY_F18:    (curses.KEY_F8,    True,  False, False),
    curses.KEY_F19:    (curses.KEY_F9,    True,  False, False),
    curses.KEY_F20:    (curses.KEY_F10,   True,  False, False),
    curses.KEY_SRIGHT: (curses.KEY_RIGHT, True,  False, False),
    curses.KEY_SLEFT:  (curses.KEY_LEFT,  True,  False, False),
    curses.KEY_SIC:    (curses.KEY_IC,    True,  False, False),
    curses.KEY_SDC:    (curses.KEY_DC,    True,  False, False),
    curses.KEY_SHOME:  (curses.KEY_HOME,  True,  False, False),
    curses.KEY_SEND:   (curses.KEY_END,   True,  False, False),
    curses.KEY_BTAB:   (9             ,   True,  False, False)
  }

  # ---------------------------------------------------------------------------
  # Key mapping between curses codes and gnue codes
  # ---------------------------------------------------------------------------

  _keymap = {
    vk.F1        : curses.KEY_F1,
    vk.F2        : curses.KEY_F2,
    vk.F3        : curses.KEY_F3,
    vk.F4        : curses.KEY_F4,
    vk.F5        : curses.KEY_F5,
    vk.F6        : curses.KEY_F6,
    vk.F7        : curses.KEY_F7,
    vk.F8        : curses.KEY_F8,
    vk.F9        : curses.KEY_F9,
    vk.F10       : curses.KEY_F10,
    vk.F11       : curses.KEY_F11,
    vk.F12       : curses.KEY_F12,
    vk.INSERT    : curses.KEY_IC,
    vk.DELETE    : curses.KEY_DC,
    vk.HOME      : curses.KEY_HOME,
    vk.END       : curses.KEY_END,
    vk.PAGEUP    : curses.KEY_PPAGE,
    vk.PAGEDOWN  : curses.KEY_NPAGE,
    vk.UP        : curses.KEY_UP,
    vk.DOWN      : curses.KEY_DOWN,
    vk.LEFT      : curses.KEY_LEFT,
    vk.RIGHT     : curses.KEY_RIGHT,
    vk.TAB       : 9,
    vk.ENTER     : 10,
    vk.BACKSPACE : curses.KEY_BACKSPACE
  }

  # ---------------------------------------------------------------------------
  # Main loop
  # ---------------------------------------------------------------------------

  def mainLoop (self):

    while not self.__exiting:
      try:
        key = self.__currentForm.wait ()

        if key >= 0 and key <= 255:
          self._uiFocusWidget._keypress (unicode (chr (key), i18n.encoding))
        else:
          if self.__shiftkeys.has_key (key):      # translate shifted f-key
            (key, shift, ctrl, meta) = self.__shiftkeys [key]
          else:
            (shift, ctrl, meta) = (False, False, False)

          self._uiFocusWidget._fkeypress (key, shift, ctrl, meta)
      except:
        self.showException ()

  # ---------------------------------------------------------------------------
  # Set title of a form
  # ---------------------------------------------------------------------------

  def setTitle (self, event):

    # Exactly this should actually be done by the base ui driver
    uiForm = self._gfObjToUIWidget [event._form]
    uiForm.setTitle (event.title)

  # ---------------------------------------------------------------------------
  # Display warning message
  # ---------------------------------------------------------------------------

  def formAlert (self, event):

    # Exactly this should actually be done by the base ui driver
    uiForm = self._gfObjToUIWidget [event._form]
    uiForm.statusMessage (event.data)

    # Curses specific
    curses.beep ()

  # ---------------------------------------------------------------------------
  # Cause console to beep
  # ---------------------------------------------------------------------------

  def formBeep (self, event):
    curses.beep ()

  # ---------------------------------------------------------------------------
  # Indicate start of nonresponsive phase
  # ---------------------------------------------------------------------------

  def beginWait (self, event):

    self.__currentForm.statusMessage (u_("processing..."))

  # ---------------------------------------------------------------------------
  # Indicate end of nonresponsive phase
  # ---------------------------------------------------------------------------

  def endWait (self, event):

    self.__currentForm.statusMessage (None)

  # ---------------------------------------------------------------------------
  # Show a message
  # ---------------------------------------------------------------------------

  def _showMessage (self, message, kind, title, cancel):

    attr = {
      'Question': self.attr ['infomsg'],
      'Info':     self.attr ['infomsg'],
      'Warning':  self.attr ['warnmsg'],
      'Error':    self.attr ['errormsg']
    }

    yes = _("Yes")
    no  = _("No")
    ok  = _("Ok")

    self.__screen.bkgdset (' ', attr [kind])

    (y, x) = self.__screen.getmaxyx ()

    self.__screen.move (y - 2, 0)
    self.__screen.clrtobot ()

    self.__screen.addstr (y - 2, 1, o(message))

    if kind == 'Question':
      self.__screen.addstr (y - 1, 1, '(' + yes [:1] + ')' + yes [1:] + '/' + \
                                      '(' + no  [:1] + ')' + no  [1:] + ' ? ')
    else:
      self.__screen.addstr (y - 1, 1, '[' + ok + ']')
      self.__screen.move (y - 1, 2)

    if kind == 'Question':
      validKeys = {ord (yes [0]): True,
                   ord (no [0]):  False}
    else:
      validKeys = {10: True}

    if cancel:
      validKeys [27] = None

    while True:
      key = self.__screen.getch ()
      if validKeys.has_key (key):
        result = validKeys [key]
        break

    if self.__currentForm:              # repaint status line of form
      self.__currentForm.statusMessage (None)

    return result

  # ---------------------------------------------------------------------------
  # Show exception information
  # ---------------------------------------------------------------------------

  def _showException (self, group, name, message, detail):

    # Give us a chance to debug exceptions until we have a 'good' exception
    # dialog
    gDebug (2, "MESSAGE: %s" % message)
    gDebug (2, "Detail : %s" % detail)
    self.showMessage (message, kind = 'Error')


  # ---------------------------------------------------------------------------
  # Select one of the given options
  # ---------------------------------------------------------------------------

  def getOption (self, title, options):
    """
    Create a dialog box for selecting one of the given options
    @param title: title for the dialog box
    @param options: dictionary to select a value from

    @return: the selected option (dict-value) or None if cancelled
    """

    if self.__currentForm:
      (left, top, right, bottom) = self.__currentForm.getCanvas ()
    else:
      (right, bottom) = self.__screen.getmaxyx ()
      left = top = 0

    dialog = dialogs.OptionsDialog (title, options, self.attr, self.lookupKey,
                                    left, top, right, bottom)
    try:
      return dialog.run ()

    finally:
      self.__screen.refresh ()


  # ---------------------------------------------------------------------------
  # Clean up everything
  # ---------------------------------------------------------------------------

  def _exit (self, formName):

    self.__exiting = True


  # ---------------------------------------------------------------------------
  # Get input data for a given set of fields
  # ---------------------------------------------------------------------------

  def _getInput (self, title, fields, cancel = True):

    if self.__currentForm:
      (left, top, right, bottom) = self.__currentForm.getCanvas ()
    else:
      (right, bottom) = self.__screen.getmaxyx ()
      left = top = 0

    dialog = dialogs.InputDialog (title, fields, self.attr, cancel, left, top,
                                  right, bottom)
    dialog.run ()
    return dialog.inputData


  # ---------------------------------------------------------------------------
  # Helper method for forms to get screen size
  # ---------------------------------------------------------------------------

  def screenSize (self):

    (y, x) = self.__screen.getmaxyx ()
    return (x, y)

  # ---------------------------------------------------------------------------
  # Helper method for forms to get active function keys
  # ---------------------------------------------------------------------------

  # Remember what events are enabled/disabled
  def __canEvent (self, event):
    name = event.__event__ [3:]
    if self.__events_name.has_key (name):
      self.__events_name [name] [1] = True

  def __cannotEvent (self, event):
    name = event.__event__ [6:]
    if self.__events_name.has_key (name):
      self.__events_name [name] [1] = False

  # Texts to display in function key bar
  __functionText = {
    'COMMIT':        u_("Save"),
    'ROLLBACK':      u_("Revert"),
    'NEWRECORD':     u_("Insert"),
    'MARKFORDELETE': u_("Delete"),
    'ENTERQUERY':    u_("Query"),
    'EXECQUERY':     u_("Execute"),
    'CANCELQUERY':   u_("Cancel"),
    'EXIT':          u_("Exit"),
  }

  # Now the real function
  def getFunctionKeyLine (self):

    keys = []
    for fkey in range (curses.KEY_F1, curses.KEY_F20):
      if self.__events_fkey.has_key (fkey):
        (name, active) = self.__events_fkey [fkey]
        if active and self.__functionText.has_key (name):
          keys.append (KeyMapper.getEventKeystrokeRepr (name) + "=" + \
                       self.__functionText [name])
    if not self.__events_name.has_key ('EXIT'):
      name = "EXIT"                     # Always show shortcut for Exit
      keys.append (KeyMapper.getEventKeystrokeRepr (name) + "=" + \
                   self.__functionText [name])
    return string.join (keys, ' ')
