# GNU Enterprise Forms - GF Class Hierarchy - Scrollbar
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFScrollBar.py 7954 2005-09-21 18:10:06Z reinhard $
"""
Class implementing scrollbars for controlling record positions within blocks
"""

from gnue.common.apps import errors
from gnue.forms.GFObjects.GFObj import GFObj

# =============================================================================
# Exceptions
# =============================================================================

class MissingBlockError (errors.ApplicationError):
  def __init__ (self, block):
    msg = u_("Scrollbar references a non-existent block '%s'") % block
    errors.ApplicationError.__init__ (self, msg)


# =============================================================================
# GFScrollBar
# =============================================================================

class GFScrollBar (GFObj):
  
  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent = None):
    GFObj.__init__ (self, parent)
    self._type = "GFScrollBar"
    self.label = ""
    self.rows  = 1

    self._inits = [self.__initialize]


  # ---------------------------------------------------------------------------
  # Phase I init: find the parent GFForm and the bound GFBlock
  # ---------------------------------------------------------------------------

  def __initialize (self):

    self._form = self.findParentOfType ('GFForm')

    # ...Block
    if not self.block in self._form._logic._blockMap:
      raise MissingBlockError, self.block

    self._block = self._form._logic._blockMap [self.block]
    self._block.registerScrollbar (self)


  # ---------------------------------------------------------------------------
  # Get the number of records and rows of the wrapped block
  # ---------------------------------------------------------------------------

  def getRecordAndPageCount (self):
    """
    Get the current record, the number of records and the number of rows of the
    underlying block. The number of rows is usable as a page size.

    @return: tuple (current-record, record-count, rows)
    """

    return (self._block._currentRecord, self._block._recordCount,
        self._block._rows)


  # ---------------------------------------------------------------------------
  # Adjust the ui scrollbar to reflect new position and/or record-cound
  # ---------------------------------------------------------------------------

  def adjustScrollbar (self, current, count):
    """
    Update the UI scrollbar to reflect a new position or record-count.

    @param current: current first visible record number in the block
    @param count: number of records in the block
    """

    if self.uiWidget is not None:
      self.uiWidget.adjustScrollbar (current, count)


  # ---------------------------------------------------------------------------
  # Jump to a record relative to the current record
  # ---------------------------------------------------------------------------

  def scrollRecords (self, adjustment):
    """
    Move the current record by a given adjustment

    @param adjustment: number of records to be moved (can be negative)
    """

    self._block.scrollRecords (adjustment)


  # ---------------------------------------------------------------------------
  # Jump to a given record number
  # ---------------------------------------------------------------------------

  def scrollToRecord (self, recordNumber):
    """
    Jump to the given record in the underlying block.

    @param recordNumber: number of record to jump to.
    """

    self._block.scrollToRecord (recordNumber)
