# GNU Enterprise Forms - wx UI Driver - UI specific dialogs
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: dialogs.py 7663 2005-06-27 13:41:52Z btami $

import textwrap

from wxPython import __version__ as WXVERSION
from wxPython.wx import *

from gnue.forms import VERSION
from gnue.common.apps import GConfig
from gnue.forms.uidrivers.wx.common import wxEncode

# =============================================================================
# This class implements an about dialog for the wx UI driver
# =============================================================================

class AboutBox (wxDialog):
  """
  Displays an about dialog for the current application as defined by the given
  arguments.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, name = "Unknown", appversion = VERSION,
      formversion = "?", author = "Unknown", description = 'n/a'):
    """
    @param name: name of the application
    @param appversion: version of the application (GNUe Forms)
    @param formversion: version of the form
    @param author: author of the form
    @param description: text describing the form
    """

    title = wxEncode (u_("About %s") % name)
    wxDialog.__init__ (self, None, -1, title, size = wxSize (400, 200))

    topSizer   = wxBoxSizer (wxVERTICAL)
    innerSizer = wxBoxSizer (wxVERTICAL)

    gfBox = wxStaticBox (self, -1, ' GNUe Forms ')
    gfBoxSizer = wxStaticBoxSizer (gfBox, wxVERTICAL)

    gfFlex = wxFlexGridSizer (2, 2, 4, 10)
    gfFlex.Add (wxStaticText (self, -1, wxEncode (u_("Version:"))))
    gfFlex.Add (wxStaticText (self, -1, appversion))
    gfFlex.Add (wxStaticText (self, -1, wxEncode (u_("Driver:"))))
    gfFlex.Add (wxStaticText (self, -1, "wxPython %s" % WXVERSION))

    gfBoxSizer.Add (gfFlex, 1, wxALL, 8)

    innerSizer.Add (gfBoxSizer, 0, wxEXPAND | wxBOTTOM, 4)

    fiBox = wxStaticBox (self, -1, wxEncode (u_(' Form Information ')))
    fiBoxSizer = wxStaticBoxSizer (fiBox, wxVERTICAL)

    # Make sure to have a seriously sized description
    descr = '\n'.join (textwrap.wrap (description, 78))

    fiFlex = wxFlexGridSizer (4, 2, 4, 10)
    fiFlex.Add (wxStaticText (self, -1, wxEncode (u_("Name:"))))
    fiFlex.Add (wxStaticText (self, -1, name))
    fiFlex.Add (wxStaticText (self, -1, wxEncode (u_("Version:"))))
    fiFlex.Add (wxStaticText (self, -1, formversion))
    fiFlex.Add (wxStaticText (self, -1, wxEncode (u_("Author:"))))
    fiFlex.Add (wxStaticText (self, -1, author))
    fiFlex.Add (wxStaticText (self, -1, wxEncode (u_("Description:"))))
    fiFlex.Add (wxStaticText (self, -1, descr))

    fiBoxSizer.Add (fiFlex, 1, wxALL, 8)

    innerSizer.Add (fiBoxSizer, 1, wxEXPAND | wxALL)

    buttonSizer = wxBoxSizer (wxHORIZONTAL)
    buttonSizer.Add (wxButton (self, wxID_OK, "OK"), 0, wxALL, 8)

    topSizer.Add (innerSizer, 1, wxEXPAND | wxALL, 8)

    topSizer.Add (buttonSizer, 0, wxALIGN_RIGHT)

    self.SetSizer (topSizer)
    topSizer.SetSizeHints (self)


# =============================================================================
# Class implementing a versatile input dialog
# =============================================================================

class InputDialog (wxDialog):
  """
  Dialog class prompting the user for a given number of fields. These field
  definitions are specified as follows:

  A field definition is a tuple having these elements:
  - fieldlabel: This text will be used as label in the left column
  - fieldname: This is the key in the result-dictionary to contain the value
      entered by the user
  - fieldtype: Currently these types are supported:
      - label: The contents of 'fieldlabel' as static text
      - warning: The contents of 'fieldlabel' as static text, formatted as
          warning
      - string: A text entry control
      - password: A text entry control with obscured characters
      - dropdown: Foreach element given in 'elements' a separate ComboBox
          control will be created, where each one has it's own dictionary of
          allowed values. If a value is selected in one control, all others are
          synchronized to represent the same key-value.
  - default: Default value to use
  - masterfield: Used for 'dropdowns'. This item specifies another field
      definition acting as master field. If this master field is changed, the
      allowedValues of this dropdown will be changed accordingly. If a
      masterfield is specified the 'allowedValues' dictionaries are built like
      {master1: {key: value, key: value, ...}, master2: {key: value, ...}}
  - elements: sequence of input element tuples (label, allowedValues). This is
      used for dropdowns only. 'label' will be used as ToolTip for the control
      and 'allowedValues' gives a dictionary with all valid keys to be selected
      in the dropdown.

  @return: If closed by 'Ok' the result is a dictionary with all values entered
    by the user, where the "fieldname"s will be used as keys. If the user has
    not selected a value from a dropdown (i.e. it has no values to select)
    there will be no such key in the result dictionary. If the dialog is
    canceled ('Cancel'-Button) the result will be None.
  """

  _FIELDTYPES = ['label', 'warning', 'string', 'password', 'dropdown', 'image']
  _NO_LABELS  = ['label', 'warning', 'image']


  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, title, fields, cancel = True):
    """
    Create a new input dialog

    @param title: Dialog title
    @param fields: sequence of field definition tuples
    @param cancel: If True add a Cancel button to the dialog
    """

    wxDialog.__init__ (self, None, -1, title)

    topSizer = wxBoxSizer (wxVERTICAL)
    self.innerSizer = wxBoxSizer (wxVERTICAL)
    
    buttonSizer = wxBoxSizer (wxHORIZONTAL)
    okButton = wxButton (self, wxID_OK, "Ok")
    buttonSizer.Add (okButton, 0, wxALL, 8)

    if cancel:
      cancelButton = wxButton (self, wxID_CANCEL, "Cancel")
      buttonSizer.Add (cancelButton, 0, wxALL, 8)

    self.inputData   = {}
    self.__dropdowns = {}
    self.__labels    = []
    self.__controls  = []
    self.__lastEntry = None

    # Build all labels first, so we know the maximum label width
    self.__maxLabelWidth = 0
    for (label, name, fieldtype, default, master, elements) in fields:
      if not fieldtype.lower () in self._NO_LABELS:
        newLabel = wxStaticText (self, -1, wxEncode (label))
        self.__maxLabelWidth = max (self.__maxLabelWidth,
                                    newLabel.GetSizeTuple () [0])
        self.__labels.append (newLabel)

    self.__nextLabel = 0

    # Now build and add all controls
    for (label, name, fieldtype, default, master, elements) in fields:
      ftype = fieldtype.lower ()
      if not ftype in self._FIELDTYPES:
        raise InvalidFieldTypeError, fieldtype

      if ftype in ['label', 'warning']:
        self.__add_text (label, ftype == 'warning')

      elif ftype == 'image':
        self.__addImage (name)

      else:
        if ftype == 'string':
          self.__add_string (label, name, default, elements)

        elif ftype == 'password':
          self.__add_string (label, name, default, elements, True)

        elif ftype == 'dropdown':
          self.__add_dropdown (label, name, default, master, elements)

    self.__addToControls (okButton)
    if cancel:
      self.__addToControls (cancelButton)

    topSizer.Add (self.innerSizer, 1, wxEXPAND | wxALL, 8)
    topSizer.Add (buttonSizer, 0, wxALIGN_RIGHT)

    self.SetSizer (topSizer)
    topSizer.SetSizeHints (self)

    firstControl = self.__controls [0]
    firstControl.SetFocus ()

    if isinstance (firstControl, wxTextCtrl):
      firstControl.SetSelection (-1, -1)


  # ---------------------------------------------------------------------------
  # Add a centered, static label or warning
  # ---------------------------------------------------------------------------

  def __add_text (self, label, warning = False):

    text = wxStaticText (self, -1, label, style = wxALIGN_CENTER)

    if warning:
      text.SetForegroundColour (wxRED)

    self.innerSizer.Add (text, 1, wxALIGN_CENTER)


  # ---------------------------------------------------------------------------
  # Add a text control for a string or a password
  # ---------------------------------------------------------------------------

  def __add_string (self, label, name, default, elements, password = False):

    rowSizer = wxBoxSizer (wxHORIZONTAL)
    self.__addLabel (label, rowSizer)

    style = wxTE_PROCESS_ENTER
    if password:
      style |= wxTE_PASSWORD

    self.inputData [name] = default or ''

    entry = wxTextCtrl (self, -1, default or '', style = style)
    entry._field = name

    if elements and elements [0][0]:
      entry.SetToolTip (wxToolTip (elements [0][0]))

    EVT_TEXT (entry, -1, self.__entryChanged)
    EVT_TEXT_ENTER (entry, -1, self.__enterPressed)

    rowSizer.Add (entry, 1, style)

    self.innerSizer.Add (rowSizer, 1, wxEXPAND)
    self.__addToControls (entry)


  # ---------------------------------------------------------------------------
  # Add a series of dropdowns into a single row
  # ---------------------------------------------------------------------------

  def __add_dropdown (self, label, name, default, master, elements):

    rowSizer = wxBoxSizer (wxHORIZONTAL)
    self.__addLabel (label, rowSizer)

    perMaster = self.__dropdowns.setdefault (master, {})
    drops     = perMaster.setdefault (name, [])

    for (tip, allowedValues) in elements:
      (keys, data, defItem) = self.__getModel (master, allowedValues, default)

      data = [str(d) for d in data]
      combo = wxComboBox (self, -1, defItem or '', choices = data,
          style = wxCB_READONLY)

      combo._keys    = keys
      combo._data    = data
      combo._default = default
      combo._field   = name
      combo._master  = master
      combo._allowedValues = allowedValues

      if tip:
        combo.SetToolTip (wxToolTip (wxEncode(tip)))

      drops.append (combo)
      EVT_COMBOBOX (combo, -1, self.__comboChanged)

      if defItem:
        self.inputData [name] = default

      rowSizer.Add (combo, 1, wxEXPAND | wxRIGHT, 4)
      self.__addToControls (combo)

    self.innerSizer.Add (rowSizer, 1, wxEXPAND)


  # ---------------------------------------------------------------------------
  # Add a centered image to the dialog
  # ---------------------------------------------------------------------------

  def __addImage (self, imageURL):

    image = wxImage (imageURL).ConvertToBitmap ()
    bitmap = wxStaticBitmap (self, -1, image)
    self.innerSizer.Add (bitmap, 0, wxALIGN_CENTER)


  # ---------------------------------------------------------------------------
  # Add the next label to the given sizer and increment the label-index
  # ---------------------------------------------------------------------------

  def __addLabel (self, label, sizer):

    cLabel = self.__labels [self.__nextLabel]
    self.__nextLabel += 1

    style = wxALIGN_LEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL
    sizer.Add (cLabel, 0, style, 10)
    sizer.SetItemMinSize (cLabel, self.__maxLabelWidth,
                                  cLabel.GetSizeTuple () [1])


  # ---------------------------------------------------------------------------
  # Whenever an entry will be changed, keep inputData in snyc
  # ---------------------------------------------------------------------------

  def __entryChanged (self, event):

    entry = event.GetEventObject ()
    self.inputData [entry._field] = entry.GetValue ()


  # ---------------------------------------------------------------------------
  # If <Enter> is pressed within a text control, move the focus
  # ---------------------------------------------------------------------------
  
  def __enterPressed (self, event):

    entry = event.GetEventObject ()
    if entry == self.__lastEntry:
      self.EndModal (wxID_OK)

    for item in self.__controls:
      if item._index == entry._index + 1:
        item.SetFocus ()


  # ---------------------------------------------------------------------------
  # On change of a combo, make sure to keep all others and depending in sync
  # ---------------------------------------------------------------------------
   
  def __comboChanged (self, event):

    combo  = event.GetEventObject ()
    value  = combo.GetStringSelection()
    newKey = combo._keys[ combo._data.index (value)]

    self.inputData [combo._field] = newKey
    self.__updateDeps (combo, combo._master, combo._field, newKey)


  # ---------------------------------------------------------------------------
  # Get the data model for a given master from a data dictionary
  # ---------------------------------------------------------------------------

  def __getModel (self, master, dataDict, default = None):

    defaultItem = None

    if master:
      values = dataDict.get (self.inputData.get (master), {})
    else:
      values = dataDict

    if values:
      (keys, data) = (values.keys (), values.values ())
      if default and default in keys:
        defaultItem = data [keys.index (default)]
    else:
      keys = data = []

    return (keys, data, defaultItem)


  # ---------------------------------------------------------------------------
  # Keep all combos in sync and make sure to have depending combos up to date
  # ---------------------------------------------------------------------------

  def __updateDeps (self, combo, master, element, newKey):
    
    drops = self.__dropdowns [master] [element]

    # First synchronize all dropdowns of the same element
    if len (drops) > 1:
      for item in drops:
        item.SetValue (item._data [item._keys.index (newKey)])

    # If this dropdown is master of others, make sure to keep them in sync too
    if self.__dropdowns.has_key (element):
      self.__updateDepending (element, newKey)


  # ---------------------------------------------------------------------------
  # Update all depending combos of a given master
  # ---------------------------------------------------------------------------

  def __updateDepending (self, master, masterKey):

    drops = self.__dropdowns [master]

    for (datakey, combos) in drops.items ():
      for combo in combos:
        (keys, data, defItem) = self.__getModel (master, combo._allowedValues,
                                                 combo._default)
        data = [str(d) for d in data]
        if keys or combo._keys:
          combo.Enable (True)
          combo.Clear ()
          for description in data:
            combo.Append (str(description))

        combo._keys = keys
        combo._data = data

        if not keys:
          combo.SetValue ('')
          combo.Disable ()
          if datakey in self.inputData:
            del self.inputData [datakey]
        else:
          if defItem is not None:
            combo.SetValue (defItem)
            self.inputData [datakey] = combo._default

          else:
            ckey = combo._keys [combo._data.index (combo.GetValue ())]
            self.inputData [datakey] = ckey


  # ---------------------------------------------------------------------------
  # Add a widget to the list of controls, and give it an index
  # ---------------------------------------------------------------------------

  def __addToControls (self, widget):

    self.__controls.append (widget)
    widget._index = len (self.__controls)

    if not isinstance (widget, wxButton):
      self.__lastEntry = widget


  # ---------------------------------------------------------------------------
  # Show the modal dialog and clear inputData on cancelling
  # ---------------------------------------------------------------------------

  def ShowModal (self):
    """
    Starts the modal dialog. If it get's cancelled inputData will be cleared.
    """

    result = wxDialog.ShowModal (self)
    if result == wxID_CANCEL:
      self.inputData = None



# =============================================================================
# Module self test
# =============================================================================

if __name__ == '__main__':
  app = wxPySimpleApp ()

  desc = "This is a quite long description of the application.\n" \
         "It also contains newlines as well as a lot of text. This text " \
         "get's continued in the third line too.\n" \
         "WWWWWWW WWWWWWW WWWWWWWWWWWW WWWWWWW WWWWWWWWWW WWWWWWW WWWWWWW" \
         "WWWW WWWWW WWWW! And here comes the rest. Here we go."

  dialog = AboutBox ('FooBar', author = 'BarBaz', description = desc)
  try:
    dialog.ShowModal ()
  finally:
    dialog.Destroy ()

  # ---------------------------------------------------------------------------

  cname = {'c1': 'demoa', 'c2': 'demob'}
  ckey  = {'c1': 'ck-A', 'c2': 'ck-B'}

  wija = {'c1': {'04': '2004', '05': '2005'},
          'c2': {'24': '2024', '25': '2025', '26': '2026'}}

  codes = {'24': {'241': 'c-24-1', '242': 'c-24-2'},
           '25': {'251': 'c-25-1'}}

  fields = [('Foo!', '/home/tamas/gnue/share/gnue/images/gnue.png', 'image',
             None, None, []),
            ('Username', '_username', 'string', 'frodo', None, \
              [('Name of the user', None)]),
            ('Password', '_password', 'password', 'foo', None, [('yeah',1)]),
            ('Foobar', '_foobar', 'dropdown', 'frob', None, \
                [('single', {'trash': 'Da Trash', 'frob': 'Frob'})]),
            ('Multi', '_multi', 'dropdown', '100', None, \
                [('name', {'50': 'A 50', '100': 'B 100', '9': 'C 9'}),
                ('sepp', {'50': 'se 50', '100': 'se 100', '9': 'se 9'})]),
            ('Noe', '_depp', 'label', 'furz', None, []),
            ('Das ist jetzt ein Fehler', None, 'warning', None, None, []),
            ('Firma', 'company', 'dropdown', 'c1', None,
                [('Name', cname), ('Code', ckey)]),
            ('Wirtschaftsjahr', 'wija', 'dropdown', '05', 'company',
                [('Jahr', wija)]),
            ('Codes', 'codes', 'dropdown', None, 'wija',
                [('Code', codes)])]

  dialog = InputDialog ('Foobar', fields)
  try:
    dialog.ShowModal ()
    print "Result:", dialog.inputData

  finally:
    dialog.Destroy ()

  app.MainLoop ()
