#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# DESCRIPTION:
# A wxPython based user interface driver for GNUe forms.
#
# NOTES:
#
# $Id: wrappers.py 7589 2005-06-10 18:09:10Z btami $

import string, sys
from gnue.common import events
from gnue.forms.uidrivers.wx.common import *

from wxPython.wx import *
from wxPython.wx import __version__ as wxversion

# How sad...
if wxversion[:3] == '2.2':
  USING_WX_22 = 1
else:
  USING_WX_22 = 0


# ============================================================================
# Wrap a scrollable window
# ============================================================================
class ScrollableWrapper(wxScrolledWindow):
  """
  Wrapper class for scrollable windows.
  """

  # --------------------------------------------------------------------------
  # Constructor
  # --------------------------------------------------------------------------
  def __init__(self, uiform):
    self.uiform = uiform
    self.frame = frame = uiform.containerFrame
    self.form = form = uiform._form
    self.margin = 6
    if sys.platform == 'win32':
      pos = wxPoint(0, 35)
    else:
      pos = wxDefaultPosition
    wxScrolledWindow.__init__(self, frame, -1, pos)
    self.spacerpane = wxPanel(self, -1)

    EVT_SIZE(self.frame, self.OnFrameSize)
    formWidth = int(form._layout.Char__width)
    formHeight = int(form._layout.Char__height)
    self.layoutSize = wxSize(formWidth * uiform._uiDriver.widgetWidth,
                      formHeight * uiform._uiDriver.widgetHeight+45)
    self._fitted = 0
    self._pageList = []


  # --------------------------------------------------------------------------
  # Initial panel window setup during init
  # not a wxPanel set per page right after this
  # --------------------------------------------------------------------------
  def finalize(self):
    """
    Initial panel window setup during init.
    """
    w, h = self.pane.GetSizeTuple()
    self.spacerpane.SetSize(wxSize(w + self.margin * 2, h + self.margin * 2))
    self.spacerpane.SetPosition(wxPoint(self.margin, self.margin))
    try:
      self.SetVirtualSize(self.spacerpane.GetSize())
    except AttributeError:
      pass # Yes, it is more wx2.2 goodness!!
    self.Show(1)
    self.fixScrollbars()


  # --------------------------------------------------------------------------
  # Handles frame resizing
  # --------------------------------------------------------------------------
  def OnFrameSize(self, event):
    """
    Handles frame resize request.
    """
    self.SetSize(self.frame.GetClientSize())
    self.fixScrollbars()


  # --------------------------------------------------------------------------
  # Sets scrollbar pos and size correctly
  # --------------------------------------------------------------------------
  def fixScrollbars(self):
    """
    Sets correct scrollbar position and size.
    """
    x1, y1 = self.frame.GetClientSizeTuple()
    x2, y2 = self.spacerpane.GetSizeTuple()
    if x1 < x2 or y1 < y2:
      self.SetScrollbars(5, 5, x2, y2)
    else:
      self.SetScrollbars(0, 0, 0, 0)


# ============================================================================
# Wrap a tabbed (notebook-paged) window
# ============================================================================
class TabbedWrapper(ScrollableWrapper):
  """
  Wrapper class for tabbed windows.
  """

  # --------------------------------------------------------------------------
  # Constructor
  # --------------------------------------------------------------------------
  def __init__(self, *args, **parms):
    ScrollableWrapper.__init__(self, *args, **parms)
    tabstyle  = _tabStyles[self.form._layout.tabbed]
    self.pane = notebook = wxNotebook(self.spacerpane, -1, pos=wxPoint(0,0),
                                      style=tabstyle)
    self.notebooksizer = wxNotebookSizer(notebook)

    sizer2 = wxBoxSizer(wxVERTICAL)
    self.frame.SetAutoLayout(1)
    self.spacerpane.SetSizer(sizer2)

    notebook.Show(1)
    self.__notebookLoopTrap = 0
    EVT_NOTEBOOK_PAGE_CHANGING(notebook, notebook.GetId(), self.notebookTabHandler)


  # --------------------------------------------------------------------------
  # Create and add a new page
  # --------------------------------------------------------------------------
  def createPage(self, object):
    """
    Creates a new page (tab) and adds it to the pages collection.
    """
    sizer = wxBoxSizer(wxVERTICAL)
    newWidget = wxPanel(self.pane, -1, size=self.layoutSize)
    newWidget.SetAutoLayout(1)
    newWidget.SetSizer(sizer)

    self._pageList.append(newWidget)
    tabtitle = hasattr(object,'caption') and object.caption or object.name
    self.pane.AddPage(newWidget, wxEncode(tabtitle))
    self.pane.SetClientSize(self.layoutSize)
    self.fit()
    return newWidget


  # --------------------------------------------------------------------------
  # Fit window size
  # --------------------------------------------------------------------------
  def fit(self):
    """
    Fits window size to client area.
    """
    if self._fitted: return

    self.SetClientSize(self.layoutSize)

    if not USING_WX_22:
      w, h = self.pane.GetSizeTuple()
      w += self.margin * 2
      h += self.margin * 2
      self.spacerpane.SetSize(wxSize(w, h))
      self.SetSize(self.spacerpane.GetSize())
    else:
      # TODO: How fscking sad to have to do this.
      # TODO: but, I can't get wx 2.2 to cooperate
      w, h = self.GetClientSizeTuple()
      w += 20
      h += 50
      self.spacerpane.SetSize(wxSize(w, h))
      self.SetSize(wxSize(w, h))
    self.frame.SetClientSize(self.GetSize())
    self._fitted = 1


  # --------------------------------------------------------------------------
  # Catch the tab changes
  # --------------------------------------------------------------------------
  def notebookTabHandler(self, event):
    """
    Handles tab change requests.
    """
    if event.GetEventType() == wxEVT_COMMAND_NOTEBOOK_PAGE_CHANGING:
      self.__alreadyChanged = True
      self.uiform._eventHandler('requestPAGE', event.GetSelection(), _form=self.form)
      del self.__alreadyChanged


  # --------------------------------------------------------------------------
  # Trick to trap mouse page changes that causes SetSelection to be called 2x
  # --------------------------------------------------------------------------
  def setPage(self, page):
    """
    Handles double SetSelection calls during mouse page changes.
    """
    try:
      self.__alreadyChanged
      return
    except AttributeError:
      pageId = self._pageList.index(page)
      self.pane.SetSelection(pageId)


# ============================================================================
# Wrap a single page window
# ============================================================================
class PlainWrapper(ScrollableWrapper):
  """
  Wrapper class for single page windows.
  """

  # --------------------------------------------------------------------------
  # Constructor
  # --------------------------------------------------------------------------
  def __init__(self, *args, **parms):
    ScrollableWrapper.__init__(self, *args, **parms)
    self.pane = wxPanel(self.spacerpane, -1, size=self.layoutSize, pos=wxPoint(self.margin, self.margin))
    self.pane.Show(1)


  # --------------------------------------------------------------------------
  # Create and add a new page
  # --------------------------------------------------------------------------
  def createPage(self, object):
    """
    Creates a new page (tab) and adds it to the pages collection.
    """
    newWidget = wxPanel(self.pane, -1,  size=self.layoutSize)

    # Only show first page
    if not self._fitted:
      newWidget.Show(1)
    else:
      newWidget.Hide()

    self._pageList.append(newWidget)
    self.fit()
    return newWidget


  # --------------------------------------------------------------------------
  # Fit window size
  # --------------------------------------------------------------------------
  def fit(self):
    """
    Fits window size to client area.
    """
    if self._fitted: return

    self.pane.SetClientSize(self.layoutSize)
    self.pane.Fit()
    self.SetClientSize(self.pane.GetSize())
    self.SetSize(self.pane.GetSize())
    self.frame.Fit()
    self._fitted = 1


  # --------------------------------------------------------------------------
  # Set current page
  # --------------------------------------------------------------------------
  def setPage(self, page):
    """
    Sets current page.
    """
    for p in self._pageList:
      if p != page:
        p.Show(0)
      else:
        p.Show(1)


_tabStyles = {'left':wxNB_LEFT,
              'right':wxNB_RIGHT,
              'bottom':wxNB_BOTTOM,
              'top':0,
              }
