#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# GFInstance.py
#
# DESCRIPTION:
"""
GFInstance manages forms instances in a 1:N relationship.
It sits between the UI and the form and passes events
between the two in a semi-intelligent manner.
"""
# NOTES:
#  Once all the events are moved back in here
#    make the next/prec methods more generic in the GFForm
#    change self._form in here to a list

import os
import sys
import dircache

from gnue.forms.GFForm import *
from gnue.forms.GFParser import loadFile
from gnue.forms import VERSION
from gnue.forms import GFKeyMapper
from gnue.common.apps import GDebug
from gnue.common.datasources import GDataObjects, GConnections
from gnue.common import events
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.utils.FileUtils import openResource, openBuffer
from gnue.common.logic.language import AbortRequest
from gnue.common.apps.GClientApp import *

class GFInstance(events.EventAware):
  #
  # Initialize the class
  #
  def __init__(self, manager, connections, ui, disableSplash=0,
               parameters={}, parentContainer=None):
    #
    # Configure event handling
    #
    self.eventController = events.EventController()
    events.EventAware.__init__(self, self.eventController)
    self.registerEventListeners( {
                           # Datasource trigger events
                           'preCommit'           : self.preCommit,
                           'preUpdate'           : self.preUpdate,
                           'preInsert'           : self.preInsert,
                           'preDelete'           : self.preDelete,
                           'onRecordTouched'     : self.onRecordTouched,
                           'onQueryNewRecord'    : self.onQueryNewRecord,

                           # Focus-related events
                           'requestNEXTENTRY'    : self.nextEntry,
                           'requestPREVENTRY'    : self.previousEntry,
                           'requestNEXTPAGE'     : self.nextPage,
                           'requestPREVPAGE'     : self.previousPage,
                           'requestNEXTBLOCK'    : self.nextBlock,
                           'requestPREVBLOCK'    : self.previousBlock,
                           'requestFIRSTRECORD'  : self.firstRecord,
                           'requestLASTRECORD'   : self.lastRecord,
                           'requestPREVRECORD'   : self.prevRecord,
                           'requestNEXTRECORD'   : self.nextRecord,
                           'requestRECORDNUMBER' : self.jumpToRecord,
                           'requestJUMPPROMPT'   : self.requestJumpTo,
                           'requestJUMPRECORD'   : self.jumpRecords,
                           'requestJUMPROWSUP'   : self.jumpRowsUp,
                           'requestJUMPROWSDOWN' : self.jumpRowsDown,
                           'requestPAGE'         : self.gotoPage,
                           'requestFOCUS'        : self.changeFocus,

                           # Data set manipulation
                           'requestNEWRECORD'    : self.newRecord,
                           'requestMARKFORDELETE': self.deleteRecord,
                           'requestCOMMIT'       : self.executeCommit,
                           'requestPRINTOUT'     : self.executePrintout,
                           'requestROLLBACK'     : self.executeRollback,
                           'requestENTERQUERY'   : self.requestQuery,
                           'requestEXECQUERY'    : self.executeQuery,
                           'requestCANCELQUERY'  : self.cancelQuery,
                           'requestCOPYQUERY'    : self.copyQuery,

                           # Miscellaneous stuff
                           'requestEXIT'         : self.executeExit,
                           'requestABOUT'        : self.executeAbout,
                           'requestMODETOGGLE'   : self.toggleInsertMode,

                           'fireTRIGGER'         : self.fireTrigger,
                           'buttonActivated'     : self.fireButton,

##                            # Dialog Support
##                            'requestDIALOG'        : self.activateDialog,

                           # "Entry" events (proxied)
                           'requestKEYPRESS'     : self.proxyEntryEvent,
                           'requestREPLACEVALUE' : self.proxyEntryEvent,
                           'requestCURSORMOVE'   : self.proxyEntryEvent,
                           'requestCURSORLEFT'   : self.proxyEntryEvent,
                           'requestCURSORRIGHT'  : self.proxyEntryEvent,
                           'requestCURSOREND'    : self.proxyEntryEvent,
                           'requestCURSORHOME'   : self.proxyEntryEvent,
                           'requestBACKSPACE'    : self.proxyEntryEvent,
                           'requestDELETE'       : self.proxyEntryEvent,
                           'requestTOGGLECHKBOX' : self.proxyEntryEvent,
                           'beginEDITMODE'       : self.proxyEntryEvent,
                           'endEDITMODE'         : self.proxyEntryEvent,
                           'requestINSERTAT'     : self.proxyEntryEvent,
                           'requestDELETERANGE'  : self.proxyEntryEvent,
                           'requestCOMBODROPPED' : self.proxyEntryEvent,

                           # Selection/clipboard events (proxied)
                           'requestSELECTWITHMOUSE' : self.proxyEntryEvent,
                           'requestSELECTALL'    : self.proxyEntryEvent,
                           'requestSELECTTOHOME' : self.proxyEntryEvent,
                           'requestSELECTTOEND'  : self.proxyEntryEvent,
                           'requestSELECTLEFT'   : self.proxyEntryEvent,
                           'requestSELECTRIGHT'  : self.proxyEntryEvent,
                           'requestCOPY'         : self.proxyEntryEvent,
                           'requestCUT'          : self.proxyEntryEvent,
                           'requestPASTE'        : self.proxyEntryEvent,
                     } )

    self.connections = connections       # Link to the GBaseApp's GConnections
    self.manager = manager               # Link to the GBaseApp Instance that
                                         #   created this GFInstance
    self._uimodule = ui                  # The UI created in the GBaseApp
    self._disableSplash = disableSplash  # Disable splashscreen
    self._parameters = parameters        # The parameters passed to the
                                         # GBaseApp instance
    self._formsDictionary = {}           # A dictionary containing all the
                                         # forms loaded from a file

    self._parentContainer = parentContainer

    #
    # Load user customized key mappings
    #
    options = gConfigDict()
    mapping = {}

    for key in options.keys():
      if string.lower(key)[:4] == 'key_':
        mapping[key[4:]] = options[key]

    GFKeyMapper.KeyMapper.loadUserKeyMap(mapping)
  
  
  def addDialogs(self):
    """
    Loads the base dialogs into memory.
    
    Base dialogs include items such as the jump to record
    dialog and error dialogs.
    """
    
    
    #
    # Import and register dialogs
    #
    basedir  = os.path.dirname(sys.modules[self.__module__].__file__)
    basedir +='/dialogs/'
    for dialogName in dircache.listdir(basedir):
       #try:
         if dialogName[0] != '_' and dialogName != '.svn':

           if os.path.isfile(basedir+dialogName) and os.path.splitext(dialogName)[1] == ".py":
             dialogName = os.path.splitext(dialogName)[0]
             dialog = dyn_import('gnue.forms.dialogs.%s' %(dialogName))
           elif os.path.isdir(basedir+dialogName):
             dialog = dyn_import('gnue.forms.dialogs.%s' %(dialogName))
           try:
             self.addFormFromBuffer(dialog.buildForm())
           except StandardError, mesg:
             print "WARNING: Cannot build %s form \n%s" % (dialogName,mesg)

       #except ImportError, mesg:
       #  GDebug.printMesg(2,"dialogs.%s doesn't appear to be a valid dialog" % (dialogName))
       #  GDebug.printMesg(5,' --> %s' % (mesg))


  def addFormFromBuffer(self,buffer):
    """
    Loads a GObj based form tree when passed a string containing 
    gfd markup.
    
    @param fileName: A string containing a URI
    """
    try:
      fileHandle=openBuffer(buffer)
      self.addFormFromFilehandle(fileHandle)
      fileHandle.close()
    except IOError, mesg:
      self.manager.handleStartupError(_("Unable to open file\n\n     %s")%mesg)

  def addFormFromFile(self,fileName):
    """
    Loads a GObj based form tree when passed a file name.
    
    @param fileName: A string containing a URI
    """
    try:
      fileHandle=openResource(fileName)
      self.addFormFromFilehandle(fileHandle)
      fileHandle.close()
    except IOError, mesg:
      self.manager.handleStartupError(_("Unable to open file\n\n     %s")%mesg)

  def addFormFromFilehandle(self,fileHandle):
    """
    Loads a GObj based form tree when passed a valid python file handle.
    
    A copy of the instance is passed into the parser so that
    it can work with things like the GConnections stored in
    the base app
    
    @param fileHandle: A python file handle.
    """
    
    #
    # Load the file bypassing the initialization
    # We bypass the initialization because <dialog>s are
    # really <form>s and they don't like being children
    # of another form
    #
    form = loadFile (fileHandle, self, initialize=0)

    #
    # Extract the child <dialog>s from the main form tree
    #
    self.reapSubforms(form)

    #
    # Add the main form into the dictionary
    #
    self._formsDictionary[form.name] = form

  #
  # removes the subforms from the main tree before
  # any UI stuff is constructed.
  #
  # NOTE: using list.remove() will not work in a
  #       for loop as it skips the very next item
  #       in the list
  #
  def reapSubforms(self,formTree):
    childList = []
    for child in formTree._children:
      if isinstance(child, GFForm):
        child._parent = None
        self._formsDictionary[child.name] = child
      else:
        childList.append(child)
    formTree._children = childList

  #
  # activate
  #
  # Tells the instance to bring up it's UI, activate the __main__ form,
  # and pass control to the UI
  #
  def activate(self):
    #
    # Initialize all the forms loaded into memory
    #
    try:
      for dialog in self._formsDictionary.keys():
        object = self._formsDictionary[dialog]
        object.phaseInit()

    except GConnections.Error, mesg:
      self.manager.handleStartupError(mesg)

    except GDataObjects.ConnectError, mesg:
      self.manager.handleStartupError(
        _("Unable to login to datasource.\n\n       %s") %mesg)

    except GDataObjects.ConnectionError, mesg:
      self.manager.handleStartupError(
        _("Error while communicating with datasource.\n\n       %s") %mesg)

    except GDataObjects.Error, mesg:
      self.manager.handleStartupError(mesg)


    #
    # Construct an instance of the UI driver
    self._uiinstance =  self._uimodule.GFUserInterface(self.eventController, disableSplash=self._disableSplash, parentContainer=self._parentContainer)
    self._uiinstance.initialize()

    #
    # Build the UIs for all the forms
    #
    for dialog in self._formsDictionary.keys():
      self.buildForm(dialog)

    #
    # Bring up the main form
    #
    self.activateForm('__main__')
    self._uiinstance.mainLoop()

  #
  #  Builds a UI based upon a specific form and
  #  sets that form to active status
  #
  def activateForm(self,formName='__main__',parameters={}, modal=0):
    form = self._formsDictionary[formName]

    if parameters != {}:
      form._parameters=parameters

    if not form._currentEntry:
      raise  _('There are no navigable widgets in this form.  Unable to display.')

    form.processTrigger('On-Activation')

    self._uiinstance.activateForm(formName,modal)

    self.dispatchEvent('gotoENTRY',object=form._currentEntry,
                       _form=form)

    form.refreshDisplay(form)
    self.updateStatus(form)

  #
  # buildForm
  #
  # Requests that the ui driver build a user interface
  # for a form tree.
  #
  def buildForm(self,formName='__main__'):
    form = self._formsDictionary[formName]

    self._uiinstance.buildForm(form, formName)

  #
  #
  # UI FUNCTIONS
  #
  # Routines to update parts of the UI's status bar
  #
  def updateInsertMode(self, form):
    self.updateStatusBar(insertMode=form._insertMode, form=form)

  def updateRecordCounter(self,form):
    self.updateStatusBar(currentRecord=form._currentBlock._currentRecord+1,
                         maxRecord=form._currentBlock._recordCount, form=form)

  def updatePageCounter(self, form):
    maxPages = len(form._layout._pageList)
    count = form._layout._pageList.index(form._currentPage)
    self.updateStatusBar(currentPage=count+1,maxPage=maxPages, form=form)

  def updateTip(self, form):
    tip = ''
    if form._currentEntry:
      if form._currentEntry.getOption('tip'):
        tip = form._currentEntry.getOption('tip')
    self.updateStatusBar(tip=tip, form=form)

  #
  # updateRecordStatus
  #
  # Determine the status of the current record
  # in the current block.
  #
  def updateRecordStatus(self, form):
    if form._currentBlock._resultSet.current.isDeleted():
      status = 'deleted'
    elif form._currentBlock._resultSet.current.isPending():
      status = 'modified'
    elif form._currentBlock.mode == 'query':
      status = 'query'
    else:
      status = 'saved'
    self.updateStatusBar(recordStatus=status, form= form)

  #
  # updateStatus
  #
  # Calls all the various update status bar
  # functions to ensure everything is refreshed
  #
  def updateStatus(self, form):
    self.updateTip(form=form)
    self.updateInsertMode(form=form)
    self.updateRecordCounter(form=form)
    self.updateRecordStatus(form=form)
    self.updatePageCounter(form=form)

  #
  # updateStatusBar
  #
  # generates the event to the UI that
  # tells it to update it's status bar
  #
  def updateStatusBar(self, tip=None, recordStatus=None, insertMode=None,
                      currentRecord=None,maxRecord=None, currentPage=None,
                      maxPage=None, form=None):
    self.dispatchEvent('uiUPDATESTATUS',tip=tip, recordStatus=recordStatus,
                       insertMode=insertMode, currentRecord=currentRecord,
                       maxRecord=maxRecord, currentPage=currentPage,
                       maxPage=maxPage, _form=form)

  #
  #
  # EVENT FUNCTIONS
  #
  # From here down should be nothing but eventListeners listed
  # in the __init__ above.
  #

  #
  # _entryUpdated
  #
  # Common code snipped called when something has changed with an
  # entry and it has to update the stat bar.  Used to cut down on
  # repeated code.
  #
  def _entryUpdated(self, form):
    self.dispatchEvent('gotoENTRY',object=form._currentEntry, _form=form)
    self.updateRecordCounter(form)
    self.updateRecordStatus(form)

  
  def proxyEntryEvent(self, event):
    """
    This serves as a proxy for any entry-level events.
    Since we don't want all entries listening for all
    events (only the *current* entry), we will capture
    any relevant events here and pass on to the entry.

    @param event: The event currently being processed.
    """
    if event._form._currentEntry and \
       event._form._currentEntry._type != 'GFButton':

      handler = event._form._currentEntry._displayHandler

      # If the display will need to be refreshed,
      # then the proxied event should set this to 1
      event.refreshDisplay = 0

      # Pass off the event to the current entry's event handler
      handler.subevents.dispatchEvent(event)

      # Refresh entry display if appropriate
      if event.refreshDisplay:
        handler.generateRefreshEvent()
        event._form.refreshUIEvents()

      # If the entry needs an error message displayed,
      # then the proxied event should set this to the
      # message text
      if event.__errortext__:
        self.displayMessageBox(event.__errortext__)

  def nextEntry(self, event):
    """
    Called whenever an event source has requested that the
    focus change to the next data entry object
    
    @param event: The event currently being processed.
    """
    if not event._form.endEditing():
      return
    message = event._form.nextEntry()
    if message:
      self.displayMessageBox( message)
      return
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateRecordStatus(event._form)
    self.updateTip(event._form)

  def previousEntry(self, event):
    """
    Called whenever an event source has requested that the
    focus change to the next data entry object
    
    param event: The event that requested the previous entry.
    """
    
    if not event._form.endEditing():
      return

    event._form.previousEntry()

    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateRecordStatus(event._form)
    self.updateTip(event._form)

  #
  # nextBlock
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry block
  #
  def nextBlock(self,event):
    if not event._form.endEditing():
      return
    event._form.nextBlock()
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # previousBlock
  #
  # Called whenever an event source has requested that the
  # focus change to the next data entry block
  #
  def previousBlock(self, event):
    if not event._form.endEditing():
      return
    event._form.previousBlock()
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)


  #
  # nextPage
  #
  # Called to make the form jump to the next page
  #
  def nextPage(self,event):
    if not event._form.endEditing():
      return
    currentIndex = event._form._layout._pageList.index(event._form._currentPage)
    if currentIndex == len(event._form._layout._pageList)-1:
      nextIndex = 0
    else:
      nextIndex = currentIndex+1
    event._form.findAndChangeFocus(event._form._layout._pageList[nextIndex])
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # previousPage
  #
  # Called to make the form jump to the previous page
  #
  def previousPage(self,event):
    if not event._form.endEditing():
      return
    currentIndex = event._form._layout._pageList.index(event._form._currentPage)
    event._form.findAndChangeFocus(event._form._layout._pageList[currentIndex-1])
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # gotoPage
  #
  # Called to make the form jump to a specific page
  #
  def gotoPage(self,event):
    if not event._form.endEditing():
      return
    newpage = event._form._layout._pageList[event.data]
    event._form.findAndChangeFocus(newpage)
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # previousRecord
  #
  # Called enever an event source has requested that the
  # form advance to the next record in memory
  #
  def prevRecord(self, event):
    if not event._form.endEditing():
      return
    message = event._form.prevRecord()
    if message:
      self.displayMessageBox(message)

      return
    self._entryUpdated(event._form)

  #
  # nextRecord
  #
  # Called whenever an event source has requested that the
  # form advance to the next record in memory
  #
  def nextRecord(self, event):
    if not event._form.endEditing():
      return
    message = event._form.nextRecord()
    if message:
      self.displayMessageBox(message)

      return
    self._entryUpdated(event._form)

  #
  # firstRecord
  #
  # Called enever an event source has requested that the
  # form advance to the first record in memory
  #
  def firstRecord(self, event):
    if not event._form.endEditing():
      return
    message = event._form.firstRecord()
    if message:
      self.displayMessageBox(message)

      return
    self._entryUpdated(event._form)

  #
  # lastRecord
  #
  # Called enever an event source has requested that the
  # form advance to the last record in memory
  #
  def lastRecord(self, event):
    if not event._form.endEditing():
      return
    message = event._form.lastRecord()
    if message:
      self.displayMessageBox(message)

      return
    self._entryUpdated(event._form)

  #
  # jumpToRecord
  #
  # Called whenever an event source has requested that the
  # form move a specific record
  #
  def jumpToRecord(self,event):
    if not event._form.endEditing():
      return
    try:
      count = abs(int(event.data))-1
    except ValueError:
      message = _("Invalid numeric value entered.")
    else:
      message = event._form.jumpRecord(count)

    if message:
      self.displayMessageBox(message)

      return

    self._entryUpdated(event._form)

  #
  # jumpRecords
  #
  # jump a specified number of records
  # forward or backward.
  #
  def jumpRecords(self,event):
    if not event._form.endEditing():
      return

    # Manipulate event and redirect to jumpToRecord
    rows = event.data
    curRec = event._form._currentBlock._currentRecord + 1
    recCount = event._form._currentBlock._recordCount
    if curRec+rows < 1:
      event.data = 1
    elif curRec+rows > recCount:
      event.data = recCount
    else:
      event.data = curRec+rows
    self.jumpToRecord(event)

    self._entryUpdated(event._form)

  #
  # jumpRowsUp
  #
  # jump a rows="xx" number of records
  # backward.
  #
  def jumpRowsUp(self,event):
    if not event._form.endEditing():
      return

    # Manipulate event and redirect to jumpToRecord
    if event._form._currentEntry._rows > 1:
      event.data = -event._form._currentEntry._rows + 1
    elif event._form._currentBlock._rows >1:
      event.data = -event._form._currentBlock._rows + 1
    else:
      event.data = 0
    if event.data:
      self.jumpRecords(event)

  # jumpRowsDown
  #
  # jump a rows="xx" number of records
  # forward.
  #
  def jumpRowsDown(self,event):
    if not event._form.endEditing():
      return

    # Manipulate event and redirect to jumpToRecord
    if event._form._currentEntry._rows > 1:
      event.data = event._form._currentEntry._rows - 1
    elif event._form._currentBlock._rows >1:
      event.data = event._form._currentBlock._rows - 1
    else:
      event.data = 0
    if event.data:
      self.jumpRecords(event)

  #
  # executeExit
  #
  # When exit is requested verify that the data
  # has been saved
  #
  def executeExit(self, event):
    try:
      event._form.processTrigger('Pre-Exit', ignoreAbort=False)
    
      if not event._form.isSaved():
        self.displayMessageBox( _('Data not saved. Save changes or clear the form to proceed.'))
        event._form.refreshUIEvents()
      else:
        event._form.processTrigger('On-Exit')
        self.dispatchEvent('exitApplication',_('Current data is saved'),_formName=event._form.name)
    except AbortRequest, t:
      self.displayMessageBox(t)
      event._form.refreshUIEvents()
  
  #
  # deleteRecord
  #
  # Tells the form to mark a record for delete
  #
  def deleteRecord(self, event):
    if event._form.readonly:
      self.dispatchEvent('formALERT',_('Form is readonly'),_form=event._form)
    elif event._form._currentBlock.restrictDelete:
      self.dispatchEvent('formALERT',_('Block does not allow delete'),_form=event._form)
    else:
      event._form.deleteRecord()
      self.updateRecordStatus(event._form)

  #
  # newRecord
  #
  # Tells the form to create a new record
  #
  def newRecord(self, event):
    if not event._form.endEditing():
      return
    if event._form.readonly:
      self.dispatchEvent('formALERT',_('Form is readonly'),_form=event._form)
    elif event._form._currentBlock.restrictInsert:
      self.dispatchEvent('formALERT',_('Block does not allow insert'),_form=event._form)
    else:
      event._form.newRecord()
      self._entryUpdated(event._form)

  #
  # executeAbout
  #
  # Displays the about dialog
  #
  def executeAbout(self,event):
    parameters = {
      'appversion'  : VERSION,
      'ui_driver'    : self._uiinstance.name,
      'name'        : event._form.title or "Unknown",
      'formversion' : event._form.getOption('version') or "Unknown",
      'author'      : event._form.getOption('author') or "Unknown",
      'description' : event._form.getOption('description') or "Unknown",
      }
    self.activateForm('_about', parameters, modal=1)


  #
  # displayMessageBox
  #
  # Displays a generic message box
  #
  def displayMessageBox(self, message='', caption='GNUe Message'):
     parameters = {
       'caption':caption,
       'message':message
       }
     self.activateForm('_messageBox', parameters, modal=1)


  #
  # requestJumpTo
  #
  # Displays a dialog box prompting for record to jump to
  # then jumps to the requested record
  #
  def requestJumpTo(self,event):
    parameters = {
      'recordnumber': ''
      }
    self.activateForm('_jumpto', parameters, modal=1)

    if not parameters.has_key('cancel'):
      if not event._form.endEditing():
        return
      try:
        count = abs(int(float(parameters['recordnumber'])))-1
      except ValueError:
        message = _("Invalid numeric value entered.")
      else:
        message = event._form.jumpRecord(count)

      if message:
        self.displayMessageBox(message)

        return

      self._entryUpdated(event._form)

  #
  #
  # toggleInsertMode(self):
  #
  # tells the form to toggle mode
  #
  def toggleInsertMode(self, event):
    event._form.toggleInsertMode()
    self.updateInsertMode(event._form)

  #
  # executePrintout
  #
  # If the form has a trigger named process-printout then fire it
  #
  def executePrintout(self,event):
    if event._form._triggerDictionary.has_key('process-printout'):
      event._form.fireTrigger('process-printout')
    else:
      # TODO: should probably do something else as a default if trigger not available
      # TODO: like a screen print
      self.dispatchEvent('uiPRINTOUT', _form=event._form)

  #
  # executeRollback
  #
  # Tells the form to rollback everything it contains
  #
  def executeRollback(self,event):
    event._form.rollback()
    self._entryUpdated(event._form)

##   #
##   # activateDialog
##   #
##   def activateDialog(self,event):
##     pass
##     #self.activateForm(event.dialogName)


  #
  # changeFocus
  #
  # changes to the requested entry object requested by
  # an event source
  #
  def changeFocus(self, event):
    if not event._form.endEditing():
      return
    if event.data._type=='GFEntry' or event.data._type=='GFImage':
      if not event.data._navigable:
        # is it possible ever to get into non-navigable field?
        return
      if (event.data.readonly and event._form._currentBlock.mode!='query'):
        GDebug.printMesg(2,"You can focus readonly field, but you won't be able to change it.")

      newEntry = event.data
      message = event._form.changeFocus(newEntry)
      if message:
        self.displayMessageBox( message)

        return

      self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
      self.updateRecordStatus(event._form)
      self.updateTip(event._form)

  #
  # requestQuery
  #
  def requestQuery(self, event):
    if not event._form.endEditing():
      return

    message = event._form.initQuery()

    if message:
      self.displayMessageBox(message)

      return

    event._form.refreshDisplay(event._form)
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # cancelQuery
  #
  def cancelQuery(self, event):
    if not event._form.endEditing():
      return

    message = event._form.cancelQuery()

    if message:
      self.displayMessageBox(message)

      return

    event._form.refreshDisplay(event._form)
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # copyQuery
  #
  def copyQuery(self, event):
    if not event._form.endEditing():
      return

    message = event._form.copyQuery()

    if message:
      self.displayMessageBox(message)

      return

    event._form.refreshDisplay(event._form)
    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    self.updateStatus(event._form)

  #
  # executeQuery
  #
  def executeQuery(self, event):
    if not event._form.endEditing():
      return
    message = event._form.executeQuery()
    if message:
      self.displayMessageBox(message)

    self.dispatchEvent('gotoENTRY',object=event._form._currentEntry, _form=event._form)
    event._form.refreshDisplay(event._form._currentBlock)
    self._entryUpdated(event._form)

  #
  # executeCommit
  #
  def executeCommit(self, event):
    if not event._form.endEditing():
      return
    message = event._form.commit()
    if message:
      self.displayMessageBox(message)

      return
    self._entryUpdated(event._form)
    event._form.refreshDisplay(event._form._currentBlock)


  def preCommit(self, event):
    self._handlePreCommit(event, 'PRE-COMMIT')

  def preInsert(self, event):
    self._handlePreCommit(event, 'PRE-INSERT')


  def preUpdate(self, event):
    self._handlePreCommit(event, 'PRE-UPDATE')


  def preDelete(self, event):
    self._handlePreCommit(event, 'PRE-DELETE')


  def onRecordTouched(self, event):
    self._handlePreCommit(event, 'PRE-CHANGE')


  def _handlePreCommit(self, event, trigger):
    try:
      block = event.record._parent._block
    except AttributeError:
      return

    saveMode = block.mode
    block._preCommitWorkingRecord = event.record
    block.mode = 'precommit'
    block.processTrigger(trigger)
    for child in block._fieldList:
      child.processTrigger(trigger)
    block.mode = saveMode


  def onQueryNewRecord(self, event):
    try:
      block = event.record._parent._block
    except AttributeError:
      return

    #
    # TODO: Running this here causes the trigger
    # TODO: to fire prior to all blocks being filled
    #
    #block.processTrigger('POST-QUERY')
    #for child in block._fieldList:
    #  child.processTrigger('POST-QUERY')

  #
  # fireTrigger
  #
  def fireTrigger(self, event):
    event._form.fireTrigger(event.data)

  #
  # fireButton
  #
  def fireButton(self, event):
    event.data.processTrigger('On-Action')

