#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# GFField.py
#
# DESCRIPTION:
"""
The primary data entry widget in forms
"""
# NOTES:
#

from gnue.common.apps import GDebug
from gnue.forms import GFDisplayHandler
from GFValue import GFValue
import string

############################################################
# GFField
#
# Matches the Entry form tag in the gfd
#
# It send events to its parent GFBlock
#
class GFField(GFValue):
  def __init__(self, parent=None, value=None):
    GFValue.__init__(self, parent, value, 'GFField')

    # Default attributes (these may be replaced by parser)
    self.typecast="text"
    self.case="mixed"
    self.style="default"
    self.defaultToLast = False

    # Runtime variables
    self._uppercase = False
    self._lowercase = False
    self._numeric = False
    self._queryable = True

    self._inits = [self.initialize]
    self._allowedValues = None
    self._allowedValuesDescr = [""]

    self._rows = 1
    self._gap = 0

    # This will be populated by GFEntry's initialize
    self._entryList = []

    #
    # Trigger exposure
    #
    self._validTriggers ={ 'PRE-FOCUSOUT':   'Pre-FocusOut',
                           'POST-FOCUSOUT':  'Post-FocusOut',
                           'PRE-FOCUSIN':    'Pre-FocusIn',
                           'POST-FOCUSIN':   'Post-FocusIn',
                           'PRE-COMMIT':     'Pre-Commit',
                           'POST-COMMIT':    'Post-Commit',
                           'PRE-QUERY':      'Pre-Query',
                           'POST-QUERY':     'Post-Query',
##                           'PRE-MODIFY':     'Pre-Modify', #This must be a typo
                           'PRE-INSERT':     'Pre-Insert',
                           'PRE-DELETE':     'Pre-Delete',
                           'PRE-UPDATE':     'Pre-Update',
                           'PRE-CHANGE':     'Pre-Change',
                           'POST-CHANGE':    'Post-Change',
                         }

    self._triggerFunctions={'allowedValues':{'function':self.allowedValues,},
                            'autofillBySequence':{'function':self.triggerAutofillBySequence},
                            'isEmpty':{'function':self.isEmpty},
                            'resetForeignKey':{'function':self.resetForeignKey},
                            'set':{'function':self.triggerSetValue},
                            'get':{'function':self.getValue},
                            'clear': {'function': self.resetToDefault}
                           }
    self._triggerProperties={'readonly':{'set':self.setReadonly,
                                         'get':self.getReadonly
                                         },
                             'parent':  {'get':self.getParent}
                            }
    self._triggerSet = self.triggerSetValue
    self._triggerGet = self.getValue


  def triggerAutofillBySequence(self,sequenceName):
    if (not self.getValue()) or self.getValue()=="":
      sequenceNumber = self._block._dataSourceLink._dataObject.triggerExtensions.getSequence(sequenceName)
      self.setValue(sequenceNumber)

  def isEmpty(self):
    return self.getValue() in ("",None)

  def getReadonly(self):
    return self.readonly

  def setReadonly(self,value):
    self.readonly = value

  def _buildObject(self):
    # Convert deprecated attributes

    # Foreign keys
    # DEPRECATED with 0.4.x
    if hasattr(self,'foreign_key') and self.foreign_key:
      (self.fk_source, self.fk_key) = string.split(self.foreign_key,'.')
      del self.foreign_key
    if hasattr(self,'foreign_key_description') and self.foreign_key_description:
      self.fk_description = self.foreign_key_description
      del self.foreign_key_description


    return GFValue._buildObject(self)

  def initialize(self):

    self._block = block = self.findParentOfType('GFBlock')
    block._fieldMap[self.name] = self
    block._fieldList.append(self)

    if self.typecast == 'number':
      self._numeric = True
      self._allowFormulas = gConfigForms("AllowNumericFormulas")
    if self.case == 'upper':
      self._uppercase = True
    elif self.case == 'lower':
      self._lowercase = True


##    self._formatmask = ""
##    self._inputmask = hasattr(self,'inputmask') and self.inputmask or ""
##    self._displaymask = hasattr(self,'displaymask') and self.displaymask or ""


    if hasattr(self, 'default') and self.default != None and len(self.default):
      default = self.default
    else:
      default = None

    if not hasattr(self,'field') or not len(self.field):
      self.field = "__GNUe__%s" % self.name
      self._block._dataSourceLink.referenceUnboundField(self.field, default)
      self._bound = False
    else:
      self._block._dataSourceLink.referenceField(self.field, default)
      self._bound = True

    # Initialize the foreign key lookup (if necessary)
    if hasattr(self,'fk_source'):
      try:
        self.fk_key
      except NameError:
        raise "fk_source specified, but no fk_key"

      try:
        self._fk_descr = string.split(self.fk_description,',')
      except NameError:
        self.fk_description = self.fk_key
        self._fk_descr = string.split(self.fk_key,',')

      self._fk_datasource = \
         block._form._datasourceDictionary[string.lower(self.fk_source)]

      # Reference the foreign keys to their datasources
      # (so they are selected)
      for field in [self.fk_key] + self._fk_descr:
        self._fk_datasource.referenceField(field, None)

      # Register as a result set listener
      # if self._fk_datasource.hasMaster (): - not usable before phase 2 init
      if hasattr (self._fk_datasource, 'master') and self._fk_datasource.master:
        self._fk_datasource.registerResultSetListener (self.__loadAllowedValues)
      else:
        self.__loadAllowedValues (self._fk_datasource.createResultSet ())

    if hasattr(self, 'queryDefault') and \
         self.queryDefault != None and \
         self._bound and \
         len(self.queryDefault):
      block._queryDefaults[self] = self.queryDefault


  def isQueryable(self):
    return self._queryable


  #
  # Pulls the proper value from the resultSet
  #
  def getValue(self):
    # TODO: No longer true...
    # We have to check to see if the _block has been
    # setup yet.  The Trigger system calls getValue
    # before it's ready so we fake it
    try:
      mode = self._block.mode
    except AttributeError:
      # Not initialized yet.
      return

    if mode == 'query':
          try:
            value = self._block._queryValues[self]
          except KeyError:
            value = None
    elif mode == 'precommit':
        value = self._block._preCommitWorkingRecord.getField(self.field)

    else:
        value = self._block._resultSet.current.getField(self.field)
    # TODO: part of the removed block check from above
    #else:
    #  value = None

    if value == None:       value = ''

    # TODO: check to see if this is still needed
    #CheckBox does not work if comment following "if" block (ra3vat)
    if self.style == 'checkbox' and value == '':      value = 0

    return value

  #
  # triggerSetValue
  #
  # does proper typecasting when value is set via a trigger
  #
  def triggerSetValue(self, value):
    if self.style=='checkbox' and value =='':
      value = 0
    if self.typecast == "number" and value != None:
      value = float(value)
    self.setValue(value)

  # Clear out the current value (setting to null, or default value)
  def resetToDefault(self):
    try:
      default = self._block._lastValues[self.field]
    except KeyError:
      if hasattr(self, 'default') and self.default != None and len(self.default):
        default = self.default
      else:
        default = None
    self.setValue(default)


  def setValue(self, value):
    # TODO: This is now being done in the display handler
    #if self.style=='checkbox' and value =='':
    #  value = 0
    #
    # Number fields
    #if self.typecast == "number" and value != None:
    #  value = float(value)

    try:
      mode = self._block.mode
    except AttributeError:
      # Not initialized yet.
      return

    if mode == 'query':
      self._block._queryValues[self] = value

    elif mode == 'precommit':
      value = self._block._preCommitWorkingRecord.setField(self.field,value)

    else:
      self._block.processTrigger('Pre-Change')
      self.processTrigger('Pre-Change')
      self._block._resultSet.current[self.field] = value

      if self.defaultToLast:
        self._block._lastValues[self.field] = value

      self._block.processTrigger('Post-Change')
      self.processTrigger('Post-Change')

###      if self._block._resultSet.current.isPending():
###        self.dispatchEvent('canCOMMIT')
###        self.dispatchEvent('canROLLBACK')

    # If the field value has changed do to a trigger
    # then update the UI
    self._block._form.updateUIEntry(self)


  #
  # allowedValues
  #
  def __loadAllowedValues (self, resultSet):
    self._allowedValues = {"":""}
    self._allowedValuesDescr = [""]
    self._allowedValuesReverse = {"":""}
    dropdownSeparator = gConfigForms('DropdownSeparator')
    more = resultSet.firstRecord()
    while more:
      try:
        key = "%s" % resultSet.current.getField(self.fk_key)
        descriptions = []
        for i in self._fk_descr:
          descriptions.append("%s" % resultSet.current.getField(i))
        descr = string.join(descriptions,dropdownSeparator)

        self._allowedValues[key] = descr
        self._allowedValuesDescr.append(descr)
        self._allowedValuesReverse[descr] = key
      except AttributeError:
        # TODO: What's this for?
        self._allowedValues[resultSet.current.getField(self.fk_key)] = None
      more = resultSet.nextRecord()
    GDebug.printMesg (5,'Created for DropDown: %s' % self._allowedValues)

  def allowedValues(self):
    return (self._allowedValues, self._allowedValuesDescr)

  def resetForeignKey(self):
    # does nothing at all, because it's done automatically now
    # self._allowedValues = None
    # self.allowedValues()
    pass
