#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# Adapter.py
#
# DESCRIPTION:
# Transforms adapter for SimpleTabulation
#
# NOTES:
#

import string, sys, os
import xml.sax

from gnue.common.formatting import GTypecast
from gnue.reports.base.GROutputAdapter import TransformAdapter as Base
import formatters

class TransformAdapter(Base, xml.sax.ContentHandler):

  def __init__(self, *args, **params):
    Base.__init__(self, *args, **params)
    

  ###
  ### Methods for TransformAdapter support
  ### (these over-ride TransformAdapter methods)
  ###

  def open(self):
    # We need a temp file
    self.input, self.infile = self.createTempFile()
    return self.input


  def close(self):
    stylesheet = os.path.join(self.parameters['_filterbase'], 'SimpleTabulation','tabulator.css')

    # We are finished with the intermediate file, so
    # close in order for sablotron to be able to open it.
    self.input.close()

    # Get a file for output from the destination adapter.
    outfile = self.destination.getOutputHandle()

    # Get the name of the transform script
    try:
      module = formatters.loadFormatter(self.parameters['formatter'])
    except KeyError:
      raise "Filter configuration file is missing the 'formatter=' parameter"

    infile = open(self.infile,'r')

    self.formatter = module.DirectFormatter(outfile, stylesheet, self.parameters)

    # Create a parser
    parser = xml.sax.make_parser()

    # Tell the parser to use our handler
    parser.setContentHandler(self)
    parser.parse(infile)

    infile.close()

    try:
      mimetype = self.parameters['mimetype']
    except:
      mimetype = self.MIMETYPE


    # Let the destination adapter do its thing
    self.destination.close(mimetype=mimetype)

    # clean up our temp files
#    self.deleteTempFile(self.infile)


  # XML handling stuff
  def startElement(self, name, attrs):
    if name == "col":
      self._colindex += 1
      self.formatter.BeginCol()
      
    elif name == "row":
      self._colindex = -1
      try:
        self._rowtype = attrs['type']
      except KeyError:
        self._rowtype = None
      self.formatter.BeginRow(rowtype=self._rowtype)
      
    elif name == "colhead":
      self._previousRow.append(None)
      self.formatter.BeginColHead()
      
    elif name == "tablehead":
      self._previousRow = []
      self.formatter.BeginTableHead()
      
    elif name == "table":
      self.formatter.BeginTable()
      
    elif name == "sectionTitle":
      self.formatter.BeginSectionTitle()
      
    elif name == "section":
      self.formatter.BeginSection()
      
    elif name == "reportTitle":
      self.formatter.BeginReportTitle()
      
    elif name == "report":
      self.formatter.BeginReport()

    self._attrs = attrs
    self._text = ""


  def characters(self, text):
    self._text += text


  def endElement(self, name):
    if name == "col":
      try:
        supr = self._attrs['supressRepeated'] == 'Y'
      except KeyError:
        supr = 0

      if supr and self._text == self._previousRow[self._colindex]:
        c = ''
      else:
        c = self._text

      self.formatter.EndCol(c, self._colindex)
      self._previousRow[self._colindex] = self._text

    elif name == "row":
      self.formatter.EndRow()
      
    elif name == "colhead":
      try:
        colwidth = GTypecast.integer(self._attrs['width'])
      except KeyError: 
        colwidth = None
      try:
        align = self._attrs['align']
      except KeyError:
        align = 'left'
      self.formatter.EndColHead(self._text, colwidth, alignment=align)
      
    elif name == "tablehead":
      self.formatter.EndTableHead()
      
    elif name == "table":
      self.formatter.EndTable()
      
    elif name == "sectionTitle":
      self.formatter.EndSectionTitle(self._text)
      
    elif name == "section":
      self.formatter.EndSection()
      
    elif name == "reportTitle":
      self.formatter.EndReportTitle(self._text)
      
    elif name == "report":
      self.formatter.EndReport()
