/*
 * gkacct.h
 *
 * Accounting modules for GNU Gatekeeper. Provides generic
 * support for accounting to the gatekeeper.
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: gkacct.h,v $
 * Revision 1.1.2.6  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.5  2003/10/27 20:27:52  zvision
 * Improved handling of multiple accounting modules and better tracing
 *
 * Revision 1.1.2.4  2003/10/07 15:22:28  zvision
 * Added support for accounting session updates
 *
 * Revision 1.1.2.3  2003/09/18 01:18:24  zvision
 * Merged accounting code parts from 2.2
 *
 * Revision 1.1.2.2  2003/09/14 21:13:06  zvision
 * Added new FileAcct logger from Tamas Jalsovszky. Thanks!
 * Fixed module stacking. API redesigned.
 *
 * Revision 1.1.2.1  2003/06/19 15:36:04  zvision
 * Initial generic accounting support for GNU GK.
 *
 */
#ifndef __GKACCT_H_
#define __GKACCT_H_
#ifdef HAS_ACCT

#include <ptlib.h>
#include "RasTbl.h"

/** Module for logging accounting events
	generated by the gatekeeper.
*/
class GkAcctLogger : public PObject
{
	PCLASSINFO(GkAcctLogger,PObject)
public:
	/// Processing type for this module
	enum Control 
	{
		/// if cannot log an accounting event
		/// silently ignore the module and process remaining acct modules
		Optional, 
		/// if cannot log an accounting event do not allow futher 
		/// call processing (e.g. call should not be connected, etc.)
		/// always process remaining acct modules
		Required,
		/// if cannot log an accounting event do not allow futher 
		/// call processing (e.g. call should not be connected, etc.)
		/// - always process remaining acct modules,
		/// if the event has been logged, do not process remaining acct modules
		/// and allow further call processing
		Sufficient,
		/// if cannot log an accounting event ignore the module
		/// and process remaining acct modules,
		/// if the event has been logged, do not process remaining acct modules
		/// and allow further call processing
		Alternative
	};

	/// status of the acct event processing
	enum Status 
	{
		Ok = 1,		/// acct event has been logged
		Fail = -1,	/// acct event has not been logged (failure)
		Next = 0	/// acct event has not been logged because the event type
					/// is not supported/not configured for this module
	};

	/// accounting event types
	enum AcctEvent 
	{
		AcctStart = 0x0001, /// log call start (ACF or Q.931 Setup)
		AcctStop = 0x0002, /// log call stop (disconnect)
		AcctUpdate = 0x0004, /// call progress update
		AcctOn = 0x0008, /// accounting enabled (GK start)
		AcctOff = 0x0010, /// accounting disabled (GK stop)
		AcctAll = -1,
		AcctNone = 0
	};
	
	/// Construct new accounting logger object.
	GkAcctLogger(
		/// config to be read for module settings
		PConfig& cfg, 
		/// module name (should be unique among different module types)
		const PString& moduleName,
		/// name for a config section with logger settings
		/// pass NULL to use the moduleName as the section name
		const char* cfgSecName = NULL
		);
		
	virtual ~GkAcctLogger();

	/** @return
		Name for this accounting module.
	*/
	PString GetName() const { return name; }

	/** @return
		Control flag determining processing behaviour for this module
		(optional,sufficient,required).
	*/
	Control GetControlFlag() const { return controlFlag; }

	/** @return
		Flags signalling which accounting events (see #AcctEvent enum#)
		should be logged. The flags are ORed.
	*/
	int GetEnabledEvents() const { return enabledEvents; }

	/** @return
		All events supported by the module ORed together.
	*/
	int GetSupportedEvents() const { return supportedEvents; }
	
	/** Log accounting event.
	
		@return
		Status of this logging operation (see #Status enum#)
	*/
	virtual Status LogAcctEvent( 
		AcctEvent evt, /// accounting event to log
		callptr& call /// additional data for the event
		);

	/** Format time string suitable for accounting logs.
		Currently NTP format conversion occurs.
		
		@return
		Formatted time string.
	*/ 
	static PString AsString( 
		const PTime& tm /// time value to be formatted
		);

	/** Format time string suitable for accounting logs.
		Currently NTP format conversion occurs.
		
		@return
		Formatted time string.
	*/ 
	static PString AsString( 
		const time_t& tm /// time value to be formatted
		);
		
protected:
	/** @return
		Default status that should be returned when the request
		cannot be determined.
	*/
	Status GetDefaultStatus() const { return defaultStatus; }
	
	/** @return
		A name of the config file section with settings for the logger module.
	*/
	const PString& GetConfigSectionName() const { return configSectionName; }

	/** Set which accounting events should be processed by this module.
		It is important to call this from derived module constructor
		to set which accounting events are recognized by this module.
	*/
	void SetSupportedEvents(
		const int events
		)
	{
		supportedEvents = events;
	}

	/** Read a list of events to be logged (ORed #AccEvent enum# constants) 
		from the passed tokens. Override this method if new event types
		are being introduced in derived loggers.
	
		@return
		Resulting event mask.
	*/
	int GetEvents( 
		const PStringArray& tokens /// event names (start from index 1)
		) const;
		
private:
		
	GkAcctLogger(const GkAcctLogger &);
	GkAcctLogger & operator=(const GkAcctLogger &);
	 
private:
	/// processing behaviour (see #Control enum#)
	Control controlFlag;
	/// default processing status (see #Status enum#)
	Status defaultStatus;
	/// ORed #AcctEvent enum# constants - define which events are logged
	int enabledEvents;
	/// all supported (recongized) event types ORed together
	int supportedEvents;
	/// module name
	PString name;
	/// name for the config section with logger settings
	PString configSectionName;
};

/**
	Plain text file accounting module for GNU Gatekeeper.
	Based on source source code from Tamas Jalsovszky
		Copyright (c) 2003, eWorld Com, Tamas Jalsovszky
*/
class FileAcct : public GkAcctLogger
{
	PCLASSINFO(FileAcct,GkAcctLogger)
public:
	enum Constants
	{
		/// events recognized by this module
		FileAcctEvents = AcctStop
	};
	
	/** Create GkAcctLogger for plain text file accounting
	*/
	FileAcct( 
		/// config to be read for module settings
		PConfig& cfg, 
		/// name from Gatekeeper::Acct section
		const PString& moduleName,
		/// name for a config section with logger settings
		/// pass NULL to use the moduleName as the section name
		const char* cfgSecName = NULL
		);
		
	/// Destroy the accounting logger
	virtual ~FileAcct();

	/** Log accounting event.
	
		@return
		Status of this logging operation (see #Status enum#)
	*/
	virtual Status LogAcctEvent( 
		AcctEvent evt, /// accounting event to log
		callptr& call /// additional data for the event
		) ;
		
	/** Rotate the detail file, saving old file contents to a different
		file and starting with a new one.
	*/
	void Rotate();
	
protected:
	/** Called to get CDR text to be stored in the CDR file.
		Can be overriden to provide custom CDR text.
		
		@return
		true if the text is available, false otherwise
	*/
	virtual bool GetCDRText(
		PString& cdrString, /// PString for the resulting CDR line
		AcctEvent evt, /// accounting event being processed
		callptr& call /// call associated with this request (if any)
		);

private:
	/* No copy constructor allowed */
	FileAcct( const FileAcct& );
	/* No operator= allowed */
	FileAcct& operator=( const FileAcct& );

private:
	/// Plain text file name
	PString cdrFilename;
	/// false to append cdr data, true to rotate cdr files
	bool rotateCdrFile;
	/// File object
	PTextFile *cdrFile;
	/// for mutual file access
	PMutex cdrFileMutex;
};

/// Factory for instances of GkAcctLogger-derived classes
class GkAcctFactoryBase : public PObject
{
	PCLASSINFO(GkAcctFactoryBase,PObject)
protected:
	/** Consturct new factory and register its instance
		on the global list of all available acct logger
		factories.
	*/
	GkAcctFactoryBase( 
		/// name for the acct modules this factory creates
		const PString& moduleName = PString::Empty()
		);

public:	
	virtual ~GkAcctFactoryBase();
	
	/** Create new instance of the GkAcctLogger module.
	
		@return
		The new accounting logger module.
	*/
	virtual GkAcctLogger* CreateAcctLogger(
		PConfig& cfg /// settings for the new acct module
		) = 0;

	/** Compare two GkAcctLogger factories. Equality is defined
		as equality of factory names (case-sensitive). See #PObject#
		for more comments.
	*/
	virtual Comparison Compare(
		const PObject& obj
		) const;		

	/** @return
		A name for acct modules generates by this factory.
	*/
	PString GetName() const { return name; }

	/** Find factory object for specified GkAcctLogger modules.
	
		@return
		Factory object or NULL, if not matching factory found.
	*/
	static GkAcctFactoryBase* FindFactoryFor(
		const PString& moduleName /// name for acct modules
		);
		
private:
	/// name for acct modules generated by this factory
	PString name;

	PDECLARE_BASEARRAY(GkAcctFactoryArray,GkAcctFactoryBase*)
	};
	/// static list of registered logger factories
	static GkAcctFactoryArray* acctLoggerFactories;
};

/** Specialized factory template for easy declaration
	of new acct logger factories
*/
template<class _GkAcctLogger_> 
class GkAcctFactory : public GkAcctFactoryBase
{
	PCLASSINFO(GkAcctFactory,GkAcctFactoryBase)
public:
	GkAcctFactory(
		const PString& moduleName = PString::Empty()
		) 
		: 
		GkAcctFactoryBase(moduleName) 
	{
	}
	
	virtual GkAcctLogger* CreateAcctLogger(
		PConfig& cfg
		)
	{ 
		return new _GkAcctLogger_( cfg, GetName() ); 
	}
};

/** Build a list of accounting logger modules
	from the registered GkAcctFactories.
*/
PDECLARE_ARRAY(GkAcctLoggers,GkAcctLogger)
public:
	GkAcctLoggers(
		PConfig& cfg /// config file with module names and settings
		);
		
	virtual ~GkAcctLoggers();
	
	/** Log accounting event #evt# with all accounting logger modules
		on the list.
		
		@return
		TRUE if the event has been successfully logger. FALSE usually
		means to stop processing of further activities regarding the event
		(call should be terminated, rejected, etc.).
	*/
	virtual BOOL LogAcctEvent( 
		GkAcctLogger::AcctEvent evt, /// accounting event to log
		callptr& call, /// additional data for the event
		const time_t now = 0 /// "now" timestamp for AcctUpdate events
		);
		
private:
	/// interval (seconds) for accounting update events to be logged (0 to skip updates)
	long acctUpdateInterval;	
};

#endif /* HAS_ACCT */
#endif /* __GKACCT_H_ */
