/*
 * radacct.cxx
 *
 * RADIUS protocol accounting logger module for GNU Gatekeeper. 
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: radacct.cxx,v $
 * Revision 1.1.2.13  2003/12/26 13:59:31  zvision
 * Fixed destination call signaling address handling
 *
 * Revision 1.1.2.12  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.11  2003/12/04 15:54:12  zvision
 * Better Framed-IP-Address determination
 *
 * Revision 1.1.2.10  2003/11/24 22:06:15  zvision
 * Small optimizations (NASIdentifier and localInterfaceAddr variables added)
 *
 * Revision 1.1.2.9  2003/10/27 20:27:52  zvision
 * Improved handling of multiple accounting modules and better tracing
 *
 * Revision 1.1.2.8  2003/10/07 15:22:28  zvision
 * Added support for accounting session updates
 *
 * Revision 1.1.2.7  2003/09/18 01:18:24  zvision
 * Merged accounting code parts from 2.2
 *
 * Revision 1.1.2.6  2003/09/14 21:13:48  zvision
 * Changes due to generic accounting API redesign
 *
 * Revision 1.1.2.5  2003/08/21 15:28:58  zvision
 * Fixed double h323-setup-time sent in Acct-Stop
 *
 * Revision 1.1.2.4  2003/08/17 20:05:39  zvision
 * Added h323-setup-time attribute to Acct-Start packets (Cisco compatibility).
 *
 * Revision 1.1.2.3  2003/07/31 22:58:48  zvision
 * Added Framed-IP-Address attribute and improved h323-disconnect-cause handling
 *
 * Revision 1.1.2.2  2003/07/03 15:30:39  zvision
 * Added cvs Log keyword
 *
 */
#if defined(HAS_RADIUS) && defined(HAS_ACCT)

#include <ptlib.h>
#include <h323pdu.h>
#include "h323util.h"
#include "gkacct.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "radacct.h"

// append RADIUS based accounting logger to the global list of loggers
GkAcctFactory<RadAcct> RAD_ACCT("RadAcct");

extern PString GetConferenceIDString( const H225_ConferenceIdentifier& id );

PMutex RadAcct::sessionMutex;
int RadAcct::sessionSeed = PRandom();
int RadAcct::sessionInc = 0;

RadAcct::RadAcct( 
	PConfig& cfg, 
	const PString& acctName,
	const char* cfgSecName
	)
	:
	GkAcctLogger( cfg, acctName, cfgSecName ),
	portBase( 1024 ),
	portMax( 65535 ),
	radiusClient( NULL )
{
	SetSupportedEvents(RadAcctEvents);
	
	const PString& secName = GetConfigSectionName();
	radiusServers = cfg.GetString(secName,"Servers","").Tokenise(";, \t",FALSE); 
	sharedSecret = cfg.GetString(secName,"SharedSecret","");
	acctPort = (WORD)(cfg.GetInteger(secName,"DefaultAcctPort"));
	requestTimeout = cfg.GetInteger(secName,"RequestTimeout");
	idCacheTimeout = cfg.GetInteger(secName,"IdCacheTimeout");
	socketDeleteTimeout = cfg.GetInteger(secName,"SocketDeleteTimeout");
	numRequestRetransmissions = cfg.GetInteger(secName,"RequestRetransmissions");
	roundRobin = cfg.GetBoolean(secName,"RoundRobinServers",TRUE);
	appendCiscoAttributes = cfg.GetBoolean(secName,"AppendCiscoAttributes",TRUE);
	includeFramedIp = cfg.GetBoolean(secName,"IncludeEndpointIP",TRUE);
	localInterface = cfg.GetString(secName,"LocalInterface", "");
	fixedUsername = cfg.GetString(secName,"FixedUsername", "");
	
	if( radiusServers.GetSize() < 1 )
	{
		PTRACE(1,"RADACCT\tCannot build "<<acctName<<" accounting logger"
			" - no RADIUS servers specified in the config"
			);
		return;
	}
	
	if( (!localInterface.IsEmpty()) 
		&& (!PIPSocket::IsLocalHost(localInterface)) )
	{
		PTRACE(1,"RADACCT\tConfigured local interface - "<<localInterface
			<<" - does not belong to this machine, assuming ip:*"
			);
		localInterface = PString::Empty();
	}

	/// build RADIUS client
	radiusClient = new RadiusClient( 
		radiusServers[0],
		(radiusServers.GetSize() > 1) ? radiusServers[1] : PString::Empty(),
		localInterface
		);

	/// if there were specified more than two RADIUS servers, append them
	for( int i = 2; i < radiusServers.GetSize(); i++ )
		radiusClient->AppendServer( radiusServers[i] );	
		
	radiusClient->SetSharedSecret( sharedSecret );
	radiusClient->SetRoundRobinServers( roundRobin );
		
	if( acctPort > 0 )
		radiusClient->SetAcctPort( acctPort );
		
	if( requestTimeout > 0 )
		radiusClient->SetRequestTimeout( requestTimeout );
	if( idCacheTimeout > 0 )
		radiusClient->SetIdCacheTimeout( idCacheTimeout );
	if( socketDeleteTimeout > 0 )
		radiusClient->SetSocketDeleteTimeout( socketDeleteTimeout );
	if( numRequestRetransmissions > 0 )
		radiusClient->SetRetryCount( numRequestRetransmissions );
	
	const PStringArray s = cfg.GetString(
		secName,"RadiusPortRange",""
		).Tokenise( "-" );

	// parse port range		
	if( s.GetSize() >= 2 )
	{ 
		unsigned p1 = s[0].AsUnsigned();
		unsigned p2 = s[1].AsUnsigned();
	
		// swap if base is greater than max
		if( p2 < p1 )
		{
			const unsigned temp = p1;
			p1 = p2;
			p2 = temp;
		}
		
		if( p1 > 65535 )
			p1 = 65535;
		if( p2 > 65535 )
			p2 = 65535;
	
		if( (p1 > 0) && (p2 > 0) )
		{
			portBase = (WORD)p1;
			portMax = (WORD)p2;
		}
	}
	
	radiusClient->SetClientPortRange( portBase, portMax-portBase+1 );
	
	if( localInterface.IsEmpty() )
		localInterfaceAddr = Toolkit::Instance()->GetRouteTable()->GetLocalAddress();
	else
		localInterfaceAddr = PIPSocket::Address(localInterface);

	NASIdentifier = Toolkit::Instance()->GKName();
}

RadAcct::~RadAcct()
{
	delete radiusClient;
}

GkAcctLogger::Status RadAcct::LogAcctEvent(
	GkAcctLogger::AcctEvent evt, 
	callptr& call
	)
{
	if( (evt & GetEnabledEvents() & GetSupportedEvents()) == 0 )
		return Next;
		
	if( radiusClient == NULL ) {
		PTRACE(1,"RADACCT\t"<<GetName()<<" - null RADIUS client instance");
		return Fail;
	}

	if( (evt & (AcctStart | AcctStop | AcctUpdate)) && (!call) ) {
		PTRACE(1,"RADACCT\t"<<GetName()<<" - missing call info for event"<<evt);
		return Fail;
	}
	
	// build RADIUS Accounting-Request
	RadiusPDU* pdu = radiusClient->BuildPDU();
	if( pdu == NULL ) {
		PTRACE(2,"RADACCT\t"<<GetName()<<" - could not build Accounting-Request PDU"
			<<" for event "<<evt<<", call no. "<<(call?call->GetCallNumber():0)
			);
		return Fail;
	}

	pdu->SetCode( RadiusPDU::AccountingRequest );

	*pdu += new RadiusAttr( RadiusAttr::AcctStatusType, 
		(evt & AcctStart) ? RadiusAttr::AcctStatus_Start
		: ((evt & AcctStop) ? RadiusAttr::AcctStatus_Stop
		: ((evt & AcctUpdate) ? RadiusAttr::AcctStatus_InterimUpdate
		: ((evt & AcctOn) ? RadiusAttr::AcctStatus_AccountingOn
		: ((evt & AcctOff) ? RadiusAttr::AcctStatus_AccountingOff : 0)
		))) );
			
	// Gk works as NAS point, so append NAS IP
	*pdu += new RadiusAttr( RadiusAttr::NasIpAddress, localInterfaceAddr );
	*pdu += new RadiusAttr( RadiusAttr::NasIdentifier, NASIdentifier );
	*pdu += new RadiusAttr( RadiusAttr::NasPortType, RadiusAttr::NasPort_Virtual );
		
	if( evt & (AcctStart | AcctStop | AcctUpdate) ) {
		*pdu += new RadiusAttr( RadiusAttr::ServiceType, RadiusAttr::ST_Login );
		
		PString sessionId = call->GetAcctSessionId();
		if( sessionId.IsEmpty() ) {
			sessionId = GetSessionId();
			call->SetAcctSessionId(sessionId);
		}
		
		*pdu += new RadiusAttr( RadiusAttr::AcctSessionId, sessionId );

		PString srcInfo = call->GetSourceInfo();
		if( !(srcInfo.IsEmpty() || srcInfo == "unknown") ) {
			const PINDEX index = srcInfo.FindOneOf(":");
			if( index != P_MAX_INDEX )
				srcInfo = srcInfo.Left(index);
		}
	
		endptr callingEP = call->GetCallingParty();
		PString userName;
		
		PIPSocket::Address callingSigAddr, calledSigAddr;
		WORD callingSigPort = 0, calledSigPort = 0;
			
		call->GetSrcSignalAddr(callingSigAddr,callingSigPort);
		call->GetDestSignalAddr(calledSigAddr,calledSigPort);
		
		if( callingEP && (callingEP->GetAliases().GetSize() > 0) )
			userName = GetBestAliasAddressString(
				callingEP->GetAliases(),
				H225_AliasAddress::e_h323_ID
				);
		else if( !srcInfo.IsEmpty() )
			userName = srcInfo;
		else if( callingSigAddr.IsValid() )
			userName = callingSigAddr.AsString();
		
		if( !(userName.IsEmpty() && fixedUsername.IsEmpty()) )					
			*pdu += new RadiusAttr( RadiusAttr::UserName, 
				fixedUsername.IsEmpty() ? userName : fixedUsername 
				);
		else
			PTRACE(3,"RADACCT\t"<<GetName()<<" could not determine User-Name"
				<<" for the call no. "<<call->GetCallNumber()
				);
		
		if( includeFramedIp && callingSigAddr.IsValid() )
			*pdu += new RadiusAttr( RadiusAttr::FramedIpAddress, callingSigAddr );
		
		if( (evt & AcctStart) == 0 )
			*pdu += new RadiusAttr( 
				RadiusAttr::AcctSessionTime, 
				call->GetDuration() 
				);
	
		PString callingStationId;
	
		callingStationId = call->GetCallingStationId();
		
		if( callingStationId.IsEmpty() && callingEP 
			&& (callingEP->GetAliases().GetSize() > 0) )
			callingStationId = GetBestAliasAddressString(
				callingEP->GetAliases(),
				H225_AliasAddress::e_dialedDigits,
				H225_AliasAddress::e_partyNumber,
				H225_AliasAddress::e_h323_ID
				);
					
		if( callingStationId.IsEmpty() )
			callingStationId = srcInfo;
			
		if( callingStationId.IsEmpty() && callingSigAddr.IsValid() )
			callingStationId = ::AsString(callingSigAddr,callingSigPort);
		
		if( !callingStationId.IsEmpty() )
			*pdu += new RadiusAttr( RadiusAttr::CallingStationId,
				callingStationId
				);
		else
			PTRACE(3,"RADACCT\t"<<GetName()<<" could not determine"
				<<" Calling-Station-Id for the call "<<call->GetCallNumber()
				);

		PString calledStationId;
								
		calledStationId = call->GetCalledStationId();
			
		if( calledStationId.IsEmpty() 
			&& !(call->GetDestInfo().IsEmpty() || call->GetDestInfo() == "unknown") ) {
			const PString dest = call->GetDestInfo();
			const PINDEX index = dest.FindOneOf(":");
				
			calledStationId = (index == P_MAX_INDEX) ? dest : dest.Left(index);
		}
		
		if( calledStationId.IsEmpty() ) {
			endptr calledEP = call->GetCalledParty();
			if( calledEP && (calledEP->GetAliases().GetSize() > 0) )
				calledStationId = GetBestAliasAddressString(
					calledEP->GetAliases(),
					H225_AliasAddress::e_dialedDigits,
					H225_AliasAddress::e_partyNumber,
					H225_AliasAddress::e_h323_ID
					);
		}
	
		if( calledStationId.IsEmpty() && calledSigAddr.IsValid() )
			calledStationId = ::AsString(calledSigAddr,calledSigPort);
		
		if( calledStationId.IsEmpty() )
			PTRACE(3,"RADACCT\t"<<GetName()<<" could not determine"
				<<" Called-Station-Id for the call no. "<<call->GetCallNumber()
				);
		else
			*pdu += new RadiusAttr( RadiusAttr::CalledStationId, calledStationId );
		
		if( appendCiscoAttributes ) {
			*pdu += new RadiusAttr(
				PString("h323-gw-id=") + NASIdentifier,
				CiscoVendorId, 33
				);
			
			*pdu += new RadiusAttr(
				PString("h323-conf-id=") 
					+ GetConferenceIDString(call->GetConferenceIdentifier()),
				CiscoVendorId, 24
				);
						
			*pdu += new RadiusAttr(
				PString("h323-call-origin=proxy"), CiscoVendorId, 26
				);
				
			*pdu += new RadiusAttr(
				PString("h323-call-type=VoIP"), CiscoVendorId, 27
				);
	
			time_t tm = call->GetSetupTime();
			if( tm != 0 ) 					
				*pdu += new RadiusAttr(
					PString("h323-setup-time=") + AsString(tm),
					CiscoVendorId, 25
					);
			
			if( evt & (AcctStop | AcctUpdate) ) {
				tm = call->GetConnectTime();
				if( tm != 0 )		
					*pdu += new RadiusAttr(
						PString("h323-connect-time=") + AsString(tm),
						CiscoVendorId, 28
						);
			}
			
			if( evt & AcctStop ) {
				tm = call->GetDisconnectTime();
				if( tm != 0 )
					*pdu += new RadiusAttr(
						PString("h323-disconnect-time=") + AsString(tm),
						CiscoVendorId, 29
						);
				
				*pdu += new RadiusAttr(
					PString("h323-disconnect-cause=") 
						+ PString( PString::Unsigned, (long)(call->GetDisconnectCause()), 16 ),
					CiscoVendorId, 30
					);
			}					
			
			if( calledSigAddr.IsValid() )
				*pdu += new RadiusAttr(
					PString("h323-remote-address=") + calledSigAddr.AsString(),
					CiscoVendorId, 23
					);
		}
	
		*pdu += new RadiusAttr( RadiusAttr::AcctDelayTime, 0 );
	}
		
	// send request and wait for response
	RadiusPDU* response = NULL;
	BOOL result = OnSendPDU(*pdu,evt,call);
	
	if( result )
		// for acct update we do not wait for the response, so we can
		// do updates with reasonable performance
		if( evt & AcctUpdate )
			result = radiusClient->SendRequest( *pdu );
		else
			result = radiusClient->MakeRequest( *pdu, response ) && (response != NULL);
			
	delete pdu;
			
	if( !result ) {
		delete response;
		return Fail;
	}
				
	if( response ) {
		// check if Accounting-Request has been accepted
		result = (response->GetCode() == RadiusPDU::AccountingResponse);
		if( result )
			result = OnReceivedPDU(*response,evt,call);
		else
			PTRACE(4,"RADACCT\t"<<GetName()<<" - received response is not "
				" an AccountingResponse, event "<<evt<<", call no. "
				<<(call?call->GetCallNumber():0)
				);
		delete response;
	}
			
	return result ? Ok : Fail;
}

BOOL RadAcct::OnSendPDU(
	RadiusPDU& pdu,
	AcctEvent evt,
	callptr& call
	)
{
	return TRUE;
}

BOOL RadAcct::OnReceivedPDU(
	RadiusPDU& pdu,
	AcctEvent evt,
	callptr& call
	)
{
	return TRUE;
}

PString RadAcct::GetSessionId()
{
	PWaitAndSignal lock( sessionMutex );
	
	return psprintf("%04x%04x",sessionSeed&0xffff,(sessionInc++)&0xffff);
}

#endif /* defined(HAS_RADIUS) && defined(HAS_ACCT) */
