/*
     This file is part of GNUnet.
     (C) 2005 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/meta.c
 * @brief code for dealing with meta and keyword tree views
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "helper.h"
#include "meta.h"
#include "fs.h"
#include <extractor.h>

static int publishMetaData(EXTRACTOR_KeywordType type,
			   const char * data,
			   void * cls) {
  GtkListStore * store = cls;
  const char * stype;
  GtkTreeIter iter;

  stype = EXTRACTOR_getKeywordTypeAsString(type);
  GNUNET_ASSERT(stype != NULL);
  gtk_list_store_append(store,
			&iter);
  gtk_list_store_set(store,
		     &iter,
		     META_TYPE, type,
		     META_STYPE, stype,
		     META_VALUE, data,
		     -1);
  return OK;
}

void createMetaDataListTreeView(GladeXML * xml,
				const char * name,
				const char * previewName,
				const struct ECRS_MetaData * init) {
  GtkWidget * metaList;
  GtkListStore * metamodel;
  GtkCellRenderer * renderer;
  GtkWidget * preview;
  GdkPixbuf * pixbuf;
  GdkPixbufLoader * loader;
  unsigned char * thumb;
  size_t ts;
  int col;

  metamodel
    = gtk_list_store_new(META_NUM,
			 G_TYPE_INT,
			 G_TYPE_STRING,
			 G_TYPE_STRING);
  metaList = glade_xml_get_widget(xml,
				  name);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(metaList),
					      -1,
					      _("Category"),
					      renderer,
					      "text", META_STYPE,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(metaList),
                                              col - 1),
                                              TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(metaList),
					      -1,
					      _("Value"),
					      renderer,
					      "text", META_VALUE,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(metaList),
                                              col - 1),
                                              TRUE);
  gtk_tree_view_set_model(GTK_TREE_VIEW(metaList),
			  GTK_TREE_MODEL(metamodel));
  if (init != NULL)
    ECRS_getMetaData(init,
		     &publishMetaData,
		     metamodel);
  if ( (previewName == NULL) ||
       (init == NULL) )
    return;
  preview = glade_xml_get_widget(xml,
				 previewName);

  thumb = NULL;
  ts = ECRS_getThumbnailFromMetaData(init,
				     &thumb);
  preview = glade_xml_get_widget(xml,
				 previewName);	
  if (ts != 0) {
    loader = gdk_pixbuf_loader_new();
    gdk_pixbuf_loader_write(loader,
			    (const guchar*) thumb,
			    ts,
			    NULL);
    pixbuf = gdk_pixbuf_loader_get_pixbuf(loader);
    g_object_ref(pixbuf);
    gdk_pixbuf_loader_close(loader,
			    NULL);
    if (pixbuf != NULL) {
      gtk_image_set_from_pixbuf(GTK_IMAGE(preview),
				pixbuf);
      g_object_unref(pixbuf);
    }
    UNREF(loader);
  }
}

static int publishKeyword(const char * data,
			  void * cls) {
  GtkListStore * store = cls;
  GtkTreeIter iter;

  gtk_list_store_append(store,
			&iter);
  gtk_list_store_set(store,
		     &iter,
		     0, data,
		     -1);
  return OK;
}

void createKeywordListTreeView(GladeXML * xml,
			       const char * name,
			       const struct ECRS_URI * init) {
  GtkWidget * keywordList;
  GtkListStore * keymodel;
  GtkCellRenderer * renderer;

  keymodel
    = gtk_list_store_new(1,
			G_TYPE_STRING);
  keywordList = glade_xml_get_widget(xml,
				     name);
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(keywordList),
					      -1,
					      _("Keyword"),
					      renderer,
					      "text", 0,
					      NULL);
  gtk_tree_view_set_model(GTK_TREE_VIEW(keywordList),
			  GTK_TREE_MODEL(keymodel));
  if (init != NULL)
    ECRS_getKeywordsFromUri(init,
			    &publishKeyword,
			    keymodel);
}

void createMetaTypeComboBox(GladeXML * xml,
			    const char * name) {
  const char * stype;
  GtkWidget * metaType;
  GtkCellRenderer * renderer;
  GtkListStore * keywordTypeModel;
  GtkTreeIter iter;
  EXTRACTOR_KeywordType type;

  keywordTypeModel
    = gtk_list_store_new(KTYPE_NUM,
			 G_TYPE_STRING,
			 G_TYPE_INT);
  metaType = glade_xml_get_widget(xml,
				  name);
  gtk_combo_box_set_model(GTK_COMBO_BOX(metaType),
			  GTK_TREE_MODEL(keywordTypeModel));
  for (type=0;type<EXTRACTOR_getHighestKeywordTypeNumber();type++) {
    stype = EXTRACTOR_getKeywordTypeAsString(type);
    gtk_list_store_append(keywordTypeModel,
			  &iter);
    gtk_list_store_set(keywordTypeModel,
		       &iter,
		       KTYPE_STRING, stype,
		       KTYPE_TYPE, type,
		       -1);
  }
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(metaType),
			     renderer,
			     FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(metaType),
				renderer,
				"text", KTYPE_STRING);
  gtk_combo_box_set_active(GTK_COMBO_BOX(metaType),
			   0);
}

void handleKeywordListUpdate(GladeXML * xml,
			     const char * inputLineName,
			     const char * keywordListName) {
  const char * keyword;
  GtkWidget * keywordList;
  GtkWidget * entryLine;
  GtkListStore * keymodel;
  GtkTreeIter iter;

  keywordList = glade_xml_get_widget(xml,
				     keywordListName);
  keymodel
    = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(keywordList)));

  entryLine = glade_xml_get_widget(xml,
				   inputLineName);
  keyword = gtk_entry_get_text(GTK_ENTRY(entryLine));
  gtk_list_store_append(keymodel,
			&iter);
  gtk_list_store_set(keymodel,
		     &iter,
		     0, keyword,
		     -1);
  gtk_entry_set_text(GTK_ENTRY(entryLine), "");
}

void handleMetaDataListUpdate(GladeXML * xml,			
			      const char * typeInputLineName,
			      const char * valueInputLineName,
			      const char * metaDataListName) {
  const char * value;
  GtkWidget * metaList;
  GtkWidget * entryLine;
  GtkWidget * typeCB;
  EXTRACTOR_KeywordType type;
  GtkListStore * metamodel;
  GtkListStore * typemodel;
  GtkTreeIter iter;
  char * stype;

  metaList = glade_xml_get_widget(xml,
				  metaDataListName);
  metamodel
    = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(metaList)));

  entryLine = glade_xml_get_widget(xml,
				   valueInputLineName);
  value = gtk_entry_get_text(GTK_ENTRY(entryLine));
  if ( (value == NULL) || (strlen(value) == 0) )
    return;
  typeCB = glade_xml_get_widget(xml,
				typeInputLineName);
  typemodel
    = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(typeCB)));
  if (! gtk_combo_box_get_active_iter(GTK_COMBO_BOX(typeCB),
				      &iter))
    return; /* oops */
  gtk_tree_model_get(GTK_TREE_MODEL(typemodel),
		     &iter,
		     KTYPE_STRING, &stype,
		     KTYPE_TYPE, &type,
		     -1);
  gtk_list_store_append(metamodel,
			&iter);
  gtk_list_store_set(metamodel,
		     &iter,
		     META_TYPE, type,
		     META_STYPE, stype,
		     META_VALUE, value,
		     -1);
  gtk_entry_set_text(GTK_ENTRY(entryLine), "");
}

struct ECRS_MetaData * getMetaDataFromList(GladeXML * xml,
					   const char * name,
					   const char * previewName) {
  GtkWidget * metaList;
  GtkWidget * preview;
  GtkTreeModel * metamodel;
  GtkTreeIter iter;
  struct ECRS_MetaData * meta;
  EXTRACTOR_KeywordType type;
  GdkPixbuf * pixbuf;
  char * mvalue;

  metaList = glade_xml_get_widget(xml,
				  name);
  metamodel = gtk_tree_view_get_model(GTK_TREE_VIEW(metaList));
  meta = ECRS_createMetaData();
  if (gtk_tree_model_get_iter_first(metamodel,
				    &iter)) {
    do {	
      gtk_tree_model_get(metamodel,
			 &iter,
			 META_TYPE, &type,
			 META_VALUE, &mvalue,
			 -1);
      ECRS_addToMetaData(meta,
			 type,
			 mvalue);
    } while (gtk_tree_model_iter_next(metamodel,
				      &iter));
  }
  if (previewName == NULL)
    return meta;

  preview = glade_xml_get_widget(xml,
				 previewName);

  if (GTK_IMAGE_PIXBUF ==
      gtk_image_get_storage_type(GTK_IMAGE(preview))) {
    size_t length;
    char * thumb;

    pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(preview));
    if (pixbuf != NULL) {
      thumb = NULL;
      if (gdk_pixbuf_save_to_buffer(pixbuf,
				    &thumb,
				    &length,
				    "png",
				    NULL,
				    NULL)) {
	char * binary;

	binary = EXTRACTOR_binaryEncode((const unsigned char*)thumb,
					length);
	free(thumb);
	ECRS_addToMetaData(meta,
			   EXTRACTOR_THUMBNAIL_DATA,
			   binary);
	FREE(binary);	
      }
    }
  }
  return meta;
}

struct ECRS_URI * getKeywordURIFromList(GladeXML * xml,
					const char * name) {
  GtkTreeModel * keymodel;
  GtkTreeView * keyList;
  GtkTreeIter iter;
  struct ECRS_URI * keywordURI;
  char * mvalue;
  char ** keywords;
  unsigned int kpos;
  unsigned int ksize;

  keyList = GTK_TREE_VIEW(glade_xml_get_widget(xml,
					       name));
  keymodel = gtk_tree_view_get_model(keyList);

  keywords = NULL;
  ksize = 0;
  GROW(keywords,
       ksize,
       64);
  kpos = 0;
  if (gtk_tree_model_get_iter_first(keymodel,
				    &iter)) {
    do {	
      gtk_tree_model_get(keymodel,
			 &iter,
			 0, &mvalue,
			 -1);
      keywords[kpos++] = mvalue;
      if (kpos == ksize)
	GROW(keywords,
	     ksize,
	     kpos*2);
    } while (gtk_tree_model_iter_next(keymodel,
				      &iter));
  }
  keywords[kpos] = NULL;

  keywordURI = ECRS_keywordsToUri((const char**)keywords);
  while (kpos > 0)
    FREE(keywords[--kpos]);
  GROW(keywords,
       ksize,
       0);
  return keywordURI;
}

char * updateIntervalToString(TIME_T interval) {
  if (interval == ECRS_SBLOCK_UPDATE_SPORADIC)
    return strdup(_("--sporadic update--"));
  if (interval == ECRS_SBLOCK_UPDATE_NONE)
    return strdup(_("--no update--"));
  return timeIntervalToFancyString(interval * cronSECONDS);
}

int parseTimeInterval(const char * timeSpec,
		      TIME_T * interval) {
  if (timeSpec == NULL)
    return SYSERR;
  if (0 == strcmp(_("--sporadic update--"),
		  timeSpec))
    *interval = ECRS_SBLOCK_UPDATE_SPORADIC;
  else if (0 == strcmp(_("--no update--"),
		       timeSpec))
    *interval = ECRS_SBLOCK_UPDATE_NONE;
  else
    return parseTime(timeSpec,
		     interval);
  return OK;
}

int tryParseTimeInterval(GladeXML * xml,
			 const char * intervalComboBoxEntry,
			 TIME_T * interval) {
  GtkWidget * update;
  const char * timeSpec;

  update = glade_xml_get_widget(xml,
				intervalComboBoxEntry);
  timeSpec = gtk_entry_get_text(GTK_ENTRY(gtk_bin_get_child(GTK_BIN(update))));
  return parseTimeInterval(timeSpec,
			   interval);
}

unsigned int getAnonymityLevel(GladeXML * xml,
			       const char * spinButtonName) {
  GtkWidget * spin;

  spin = glade_xml_get_widget(xml,
			      spinButtonName);
  return gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
}



/* end of meta.c */
