/*
     This file is part of GNUnet.
     (C) 2005 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/upload.c
 * @brief code for uploading with gnunet-gtk
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "search.h"
#include "upload.h"
#include "fs.h"
#include "meta.h"
#include <extractor.h>

#ifdef MINGW
#include <shlobj.h>
#ifndef BIF_NONEWFOLDERBUTTON
#define BIF_NONEWFOLDERBUTTON 0x200
#endif
#endif


static GtkTreeStore * summary;

/**
 * XML tree for the meta-data dialog of upload.
 * (there can only be one at a time;
 * maybe NULL at times where there is no dialog)
 */
static GladeXML * metaXML;

/**
 */
void displayUploadUpdate(const char * mainName,
			 const char * filename,
			 unsigned long long completed,
			 unsigned long long total) {
  GtkTreeIter iter;
  GtkTreeIter child;
  int progress;
  char * name;

  if (total != 0)
    progress = 100 * completed / total;
  else
    progress = 100;

  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(summary),
				    &iter)) {
    do {	
      gtk_tree_model_get(GTK_TREE_MODEL(summary),
			 &iter,
			 UPLOAD_FILENAME, &name,
			 -1);
      if (0 == strcmp(name, filename)) {
	gtk_tree_store_set(GTK_TREE_STORE(summary),
			   &iter,
			   UPLOAD_PROGRESS, progress,
			   -1);
	return;
      }
      if (0 == strcmp(name, mainName)) {
	if (gtk_tree_model_iter_children(GTK_TREE_MODEL(summary),
					 &child,
					 &iter)) {
	  do {	
	    gtk_tree_model_get(GTK_TREE_MODEL(summary),
			       &child,
			       UPLOAD_FILENAME, &name,
			       -1);
	    if (0 == strcmp(name, filename)) {
	      gtk_tree_store_set(GTK_TREE_STORE(summary),
				 &child,
				 UPLOAD_PROGRESS, progress,
				 -1);
	      return;
	    }
	  } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
					    &child));	
	}
	gtk_tree_store_append(GTK_TREE_STORE(summary),
			      &child,
			      &iter);
	gtk_tree_store_set(GTK_TREE_STORE(summary),
			   &child,
			   UPLOAD_FILENAME, filename,
			   UPLOAD_PROGRESS, progress,
			   UPLOAD_URISTRING, NULL,
			   -1);
	return;
      }
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
				      &iter));
  }
  if (0 != strcmp(mainName,
		  filename)) {
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &iter,
			  NULL);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &iter,
		       UPLOAD_FILENAME, mainName,
		       UPLOAD_PROGRESS, 0,
		       UPLOAD_URISTRING, NULL,
		       -1);
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &child,
			  &iter);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &child,
		       UPLOAD_FILENAME, filename,
		       UPLOAD_PROGRESS, progress,
		       UPLOAD_URISTRING, NULL,
		       -1);
  } else {
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &iter,
			  NULL);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &iter,
		       UPLOAD_FILENAME, filename,
		       UPLOAD_PROGRESS, progress,
		       UPLOAD_URISTRING, NULL,
		       -1);
  }
}

/**
 */
void displayUploadComplete(const char * mainName,
			   const char * filename,
			   const struct ECRS_URI * uri) {
  GtkTreeIter iter;
  GtkTreeIter child;
  char * name;
  char * us;

  us = ECRS_uriToString(uri);

  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(summary),
				    &iter)) {
    do {	
      gtk_tree_model_get(GTK_TREE_MODEL(summary),
			 &iter,
			 UPLOAD_FILENAME, &name,
			 -1);
      if (0 == strcmp(name, filename)) {
	gtk_tree_store_set(GTK_TREE_STORE(summary),
			   &iter,
			   UPLOAD_PROGRESS, 100,
			   UPLOAD_URISTRING, us,
			   -1);
	FREE(us);
	return;
      }
      if (0 == strcmp(name, mainName)) {
	if (gtk_tree_model_iter_children(GTK_TREE_MODEL(summary),
					 &child,
					 &iter)) {
	  do {	
	    gtk_tree_model_get(GTK_TREE_MODEL(summary),
			       &child,
			       UPLOAD_FILENAME, &name,
			       -1);
	    if (0 == strcmp(name, filename)) {
	      gtk_tree_store_set(GTK_TREE_STORE(summary),
				 &child,
				 UPLOAD_PROGRESS, 100,
				 UPLOAD_URISTRING, us,
				 -1);
	      FREE(us);
	      return;
	    }
	  } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
					    &child));	
	}
	gtk_tree_store_append(GTK_TREE_STORE(summary),
			      &child,
			      &iter);
	gtk_tree_store_set(GTK_TREE_STORE(summary),
			   &child,
			   UPLOAD_FILENAME, filename,
			   UPLOAD_PROGRESS, 100,
			   UPLOAD_URISTRING, us,
			   -1);
	FREE(us);
	return;
      }
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
				      &iter));
  }
  if (0 != strcmp(mainName,
		  filename)) {
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &iter,
			  NULL);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &iter,
		       UPLOAD_FILENAME, mainName,
		       UPLOAD_PROGRESS, 0,
		       UPLOAD_URISTRING, NULL,
		       -1);
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &child,
			  &iter);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &iter,
		       UPLOAD_FILENAME, filename,
		       UPLOAD_PROGRESS, 100,
		       UPLOAD_URISTRING, NULL,
		       -1);
  } else {
    gtk_tree_store_append(GTK_TREE_STORE(summary),
			  &iter,
			  NULL);
    gtk_tree_store_set(GTK_TREE_STORE(summary),
		       &iter,
		       UPLOAD_FILENAME, filename,
		       UPLOAD_PROGRESS, 100,
		       UPLOAD_URISTRING, us,
		       -1);
  }
  FREE(us);
}


void on_selectAlternativePreviewButton_selection_changed(GtkWidget * preview,
							 GtkWidget * fileChooser) {
  char * fn;

  fn = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fileChooser));
  if (fn == NULL) {
    gtk_image_set_from_pixbuf(GTK_IMAGE(preview),
			      NULL);
  } else {
    GdkPixbuf * buf;

    buf = gdk_pixbuf_new_from_file(fn,
				   NULL);
    gtk_image_set_from_pixbuf(GTK_IMAGE(preview),
			      buf);
    g_object_unref(buf);
    free(fn);
  }
}

void on_metaDataDialogKeywordAddButton_clicked(gpointer dummy,
					       GtkWidget * uploadButton) {
  handleKeywordListUpdate(metaXML,
			  "fileInformationKeywordEntry",
			  "metaDataDialogKeywordList");
}

void on_metaDataDialogMetaDataAddButton_clicked(gpointer dummy,
						GtkWidget * uploadButton) {
  handleMetaDataListUpdate(metaXML,
			   "metaDataDialogMetaTypeComboBox",
			   "metaDataDialogValueEntry",
			   "metaDataDialogMetaDataList");
}

void on_fsinsertuploadbutton_clicked(gpointer dummy,
				     GtkWidget * uploadButton) {
  const char * filename;
  const char * filenamerest;
  GtkWidget * uploadLine;
  GtkWidget * entry;
  GtkWidget * dialog;
  GtkWidget * recBut;
  GtkWidget * idxBut;
  GtkWidget * deepIndex;
  struct ECRS_MetaData * meta;
  EXTRACTOR_ExtractorList * extractors;
  char * config;
  struct ECRS_URI * keywordURI;
  struct ECRS_URI * gkeywordURI;

  extractors = EXTRACTOR_loadDefaultLibraries();
  config = getConfigurationString("FS",
				  "EXTRACTORS");
  if (config != NULL) {
    extractors = EXTRACTOR_loadConfigLibraries(extractors,
					       config);
    FREE(config);
  }

  uploadLine = glade_xml_get_widget(getMainXML(),
				    "uploadFilenameComboBoxEntry");
  entry = gtk_bin_get_child(GTK_BIN(uploadLine));
  filename = gtk_entry_get_text(GTK_ENTRY(entry));

  metaXML
    = glade_xml_new(getGladeFileName(),
		    "metaDataDialog",
		    PACKAGE_NAME);
  connectGladeWithPlugins(metaXML);
  dialog = glade_xml_get_widget(metaXML,
				"metaDataDialog");
  meta = ECRS_createMetaData();
  ECRS_extractMetaData(meta,
		       filename,
		       extractors);
  EXTRACTOR_removeAll(extractors);
  filenamerest = &filename[strlen(filename)-1];
  while ( (filenamerest > filename) &&
	  (filenamerest[-1] != DIR_SEPARATOR) )
    filenamerest--;
  ECRS_addToMetaData(meta,
		     EXTRACTOR_FILENAME,
		     filenamerest);
  createMetaDataListTreeView(metaXML,
			     "metaDataDialogMetaDataList",
			     "previewImage",
			     meta);
  keywordURI = ECRS_metaDataToUri(meta);
  ECRS_freeMetaData(meta);
  createKeywordListTreeView(metaXML,
			    "metaDataDialogKeywordList",
			    keywordURI);
  ECRS_freeUri(keywordURI);
  createMetaTypeComboBox(metaXML,
			 "metaDataDialogMetaTypeComboBox");
  gtk_dialog_set_default_response(GTK_DIALOG(dialog),
				  GTK_RESPONSE_OK);
  if (gtk_dialog_run(GTK_DIALOG(dialog)) != GTK_RESPONSE_CANCEL) {
    int doIndex;
    int doRec;

    addLogEntry(_("Uploading `%s'"), filenamerest);

    recBut = glade_xml_get_widget(getMainXML(),
				  "scopeRecursiveButton");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(recBut)))
      doRec = YES;
    else
      doRec = NO;
    idxBut = glade_xml_get_widget(getMainXML(),
				  "indexbutton");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(idxBut)))
      doIndex = YES;
    else
      doIndex = NO;

    meta = getMetaDataFromList(metaXML,
			       "metaDataDialogMetaDataList",
			       "previewImage");
    keywordURI = getKeywordURIFromList(metaXML,
				       "metaDataDialogKeywordList");
    if (doRec) {
      deepIndex = glade_xml_get_widget(getMainXML(),
				       "deepIndexCheckButton");
      gkeywordURI = ECRS_stringToUri(ECRS_URI_PREFIX
				     ECRS_SEARCH_INFIX);
      FSUI_uploadAll(ctx,
		     filename,
		     getAnonymityLevel(getMainXML(),
				       "uploadAnonymityLevelSpinButton"),
		     doIndex,
		     (TRUE == gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(deepIndex)))
		     ? YES : NO,
		     meta,
		     gkeywordURI,
		     keywordURI);
    } else {
      FSUI_upload(ctx,
		  filename,
		  getAnonymityLevel(getMainXML(),
				    "uploadAnonymityLevelSpinButton"),
		  doIndex,
		  NO,
		  meta,
		  keywordURI);
    }
    ECRS_freeMetaData(meta);
    ECRS_freeUri(keywordURI);
  }
  gtk_widget_destroy (dialog);
  UNREF(metaXML);
  metaXML = NULL;
}

static char * selectFile() {
#ifndef MINGW
  GtkWidget *dialog;
  GladeXML * uploadXML;
  GtkWidget * recBut;
  char *ret;

  uploadXML
    = glade_xml_new(getGladeFileName(),
		    "uploadfilechooserdialog",
		    PACKAGE_NAME);
  connectGladeWithPlugins(uploadXML);
  dialog = glade_xml_get_widget(uploadXML,
				"uploadfilechooserdialog");

  recBut = glade_xml_get_widget(getMainXML(),
				"scopeRecursiveButton");

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(recBut)))
    gtk_file_chooser_set_action(GTK_FILE_CHOOSER(dialog),
				GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) != GTK_RESPONSE_CANCEL) {
    char * filename;

    filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
    ret = STRDUP(filename);
    g_free(filename);
  } else
    ret = NULL;

  gtk_widget_destroy (dialog);
  UNREF(uploadXML);
  return ret;
#else /* MINGW */
  GtkWidget *fileBut;
  char *fn;

  fileBut = glade_xml_get_widget(getMainXML(),
        "scopeFileOnlyButton");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(fileBut)))
    fn = plibc_ChooseFile(_("Choose the file you want to publish."),
      OFN_FILEMUSTEXIST | OFN_SHAREAWARE);
  else
    fn = plibc_ChooseDir(_("Choose the directory you want to publish."),
			 BIF_USENEWUI | BIF_SHAREABLE | BIF_NONEWFOLDERBUTTON);

  return fn;
#endif /* MINGW */
}

void on_mainFileSharingInsertBrowseButton_clicked(GtkWidget * browseButton,
						  gpointer dummy) {
  char *filename;
  GtkWidget * uploadLine;
  GtkWidget * entry;
  GtkListStore * model;
  GtkTreeIter iter;

  if ((filename = selectFile())) {
    uploadLine = glade_xml_get_widget(getMainXML(),
				      "uploadFilenameComboBoxEntry");
    entry = gtk_bin_get_child(GTK_BIN(uploadLine));
    gtk_entry_set_text(GTK_ENTRY(entry),
		       filename);
    model = GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(uploadLine)));
    gtk_list_store_prepend(model,
			   &iter);
    gtk_list_store_set(model,
		       &iter,
		       0,
		       filename,
		       -1);
    free(filename);
  }
}


void fs_upload_start() {
  GtkWidget * uploadList;
  GtkWidget * uploadEntry;
  GtkCellRenderer * renderer;
  GtkListStore * model;
  int col;

  uploadList = glade_xml_get_widget(getMainXML(),
				    "activeUploadsList");
  summary =
    gtk_tree_store_new(UPLOAD_NUM,
		       G_TYPE_STRING, /* filename */
		       G_TYPE_INT,    /* progress */
		       G_TYPE_STRING);  /* URI (as string) - after completion */
  gtk_tree_view_set_model(GTK_TREE_VIEW(uploadList),
			  GTK_TREE_MODEL(summary));
  renderer = gtk_cell_renderer_progress_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(uploadList),
					      -1,
					      _("Filename"),
					      renderer,
					      "text", UPLOAD_FILENAME,
					      "value", UPLOAD_PROGRESS,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(uploadList),
                                              col - 1),
                                              TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(uploadList),
					      -1,
					      _("URI"),
					      renderer,
					      "text", UPLOAD_URISTRING,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(uploadList),
                                              col - 1),
                                              TRUE);

  uploadEntry
    = glade_xml_get_widget(getMainXML(),
			   "uploadFilenameComboBoxEntry");

  model = gtk_list_store_new(1, G_TYPE_STRING);
  gtk_combo_box_set_model(GTK_COMBO_BOX(uploadEntry),
			  GTK_TREE_MODEL(model));
  gtk_combo_box_entry_set_text_column(GTK_COMBO_BOX_ENTRY(uploadEntry),
				      0);
}

void fs_upload_stop() {
  /* nothing to be done */
}

/* end of upload.c */
