/*
     This file is part of GNUnet
     (C) 2004, 2005 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
     Boston, MA 02110-1301, USA.

*/

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_stats_lib.h>
#include <GNUnet/gnunet_getoption_lib.h>
#include <GNUnet/gnunet_protocols.h>
#include "functions.h"

#define FUNCTIONS_DEBUG NO

static StatPair * lastStatValues;

static unsigned int lsv_size;

static GNUNET_TCP_SOCKET * sock;

static Mutex lock;

static long connectionGoal;

static long long banddown;

static long long bandup;

static int getStatValue(long long * value,
			long long * lvalue,
			cron_t * dtime,
			const char * optName) {			
  unsigned int i;

  *value = 0;
  if (lvalue != NULL)
    *lvalue = 0;
  for (i=0;i<lsv_size;i++) {
    if (0 == strcmp(optName,
		    lastStatValues[i].statName)) {
      *value = lastStatValues[i].value;
      if (lvalue != NULL)
	*lvalue = lastStatValues[i].lvalue;
      if (dtime != NULL)
	*dtime = lastStatValues[i].delta;
      return OK;
    }
  }
#if FUNCTIONS_DEBUG
  LOG(LOG_DEBUG,
      "Statistic not found: `%s'\n",
      optName);
#endif
  return SYSERR;
}

static void updateConnectionGoal(void * unused) {
  char * cmh;
  char * availableDown;
  char * availableUp;

  MUTEX_LOCK(&lock);
  cmh = getConfigurationOptionValue(sock,
				    "gnunetd",
				    "connection-max-hosts");
  availableDown = getConfigurationOptionValue(sock,
					      "LOAD",
					      "MAXNETDOWNBPSTOTAL");
  availableUp = getConfigurationOptionValue(sock,
					    "LOAD",
					    "MAXNETUPBPSTOTAL");
  MUTEX_UNLOCK(&lock);
  if (cmh == NULL)
    connectionGoal = 0;
  else
    connectionGoal = atol(cmh);
  if (availableDown == NULL)
    banddown = 0;
  else
    banddown = atol(availableDown);
  if (availableUp == NULL)
    bandup = 0;
  else
    bandup = atol(availableUp);

  FREENONNULL(cmh);
  FREENONNULL(availableDown);
  FREENONNULL(availableUp);
}

static int getConnectedNodesStat(const void * closure,
				 gfloat ** data) {
  long long val;

  if (connectionGoal == 0)
    return SYSERR;
  if (OK != getStatValue(&val,
			 NULL,
			 NULL,
            		 "# of connected peers"))
    return SYSERR;
  data[0][0] = ((gfloat) val) / connectionGoal;
  return OK;
}

static int getCPULoadStat(const void * closure,
			  gfloat ** data) {
  long long val;

  if (OK != getStatValue(&val,
			 NULL,
			 NULL,
		        "% of allowed cpu load"))
    return SYSERR;
  data[0][0] = (gfloat) val / 100.0;
  return OK;
}

static int getTrafficRecvStats(const void * closure,
			       gfloat ** data) {
  long long total;
  long long noise;
  long long content;
  long long queries;
  long long ltotal;
  long long lnoise;
  long long lcontent;
  long long lqueries;
  cron_t dtime;
  char * buffer;
  int i;

  if (OK != getStatValue(&total,	
			 &ltotal,
			 &dtime,
	  	         "# bytes decrypted"))
    return SYSERR;
  if (OK != getStatValue(&noise,
			 &lnoise,
			 NULL,
			 "# bytes of noise received"))
    return SYSERR;
  i = 0;
  buffer = MALLOC(512);
  SNPRINTF(buffer,
	   512,
	   "# bytes received of type %d",
	   P2P_PROTO_gap_RESULT);
  if (OK != getStatValue(&content,
			 &lcontent,
			 NULL,
			 buffer)) {
    content += 0;
    lcontent += 0;
  }
  i = 0;
  SNPRINTF(buffer,
	   512,
           "# bytes received of type %d",
	   P2P_PROTO_gap_QUERY);
  if (OK != getStatValue(&queries,
			 &lqueries,
			 NULL,
			 buffer)) {
    queries = 0;
    lqueries = 0;
  }
  FREE(buffer);
  if (banddown == 0)
    return SYSERR;

  total -= ltotal;
  noise -= lnoise;
  queries -= lqueries;
  content -= lcontent;
  if (banddown < 0) {
    data[0][0] = 0.0;
    data[0][1] = 0.0;
    data[0][2] = 0.0;
    data[0][3] = 0.0;
    return OK;
  }
  data[0][0] = ((gfloat) noise) / (banddown * dtime / cronSECONDS); /* red */
  data[0][1] = ((gfloat) (content+noise)) / (banddown * dtime / cronSECONDS); /* green */
  data[0][2] = ((gfloat)(queries+content+noise)) / (banddown * dtime / cronSECONDS); /* yellow */
  data[0][3] = ((gfloat) total) / (banddown * dtime / cronSECONDS); /* blue */
  /*printf("I: %f %f %f\n",
	 data[0][0],
	 data[0][1],
	 data[0][2]);*/

  return OK;
}

static int getTrafficSendStats(const void * closure,
			       gfloat ** data) {
  long long total;
  long long noise;
  long long content;
  long long queries;
  long long ltotal;
  long long lnoise;
  long long lcontent;
  long long lqueries;
  cron_t dtime;
  char * buffer;
  int i;

  if (OK != getStatValue(&total,	
			 &ltotal,
			 &dtime,
			  "# bytes encrypted"))
    return SYSERR;
  if (OK != getStatValue(&noise,
			 &lnoise,
			 NULL,
			  "# bytes noise sent"))
    return SYSERR;
  i = 0;
  buffer = MALLOC(512);
  SNPRINTF(buffer,
	   512,
	    "# bytes transmitted of type %d",
	   P2P_PROTO_gap_RESULT);
  if (OK != getStatValue(&content,
			 &lcontent,
			 NULL,
			 buffer)) {
    content = 0;
    lcontent = 0;
  }
  i = 0;
  SNPRINTF(buffer,
	   512,
	    "# bytes transmitted of type %d",
	   P2P_PROTO_gap_QUERY);
  if (OK != getStatValue(&queries,
			 &lqueries,
			 NULL,
			 buffer)) {
    queries = 0;
    lqueries = 0;
  }
  FREE(buffer);
  if (bandup == 0)
    return SYSERR;
  total -= ltotal;
  noise -= lnoise;
  queries -= lqueries;
  content -= lcontent;
  if (bandup < 0) {
    data[0][0] = 0.0;
    data[0][1] = 0.0;
    data[0][2] = 0.0;
    data[0][3] = 0.0;
    return OK;
  }
  data[0][0] = ((gfloat) noise) / (bandup * dtime / cronSECONDS); /* red */
  data[0][1] = ((gfloat) (noise + content)) / (bandup*dtime / cronSECONDS); /* green */
  data[0][2] = ((gfloat) (noise + content + queries)) / (bandup*dtime / cronSECONDS); /* yellow */
  data[0][3] = ((gfloat) total) / (bandup*dtime / cronSECONDS); /* blue */
  /* printf("O: %f %f %f\n",
     data[0][0],
     data[0][1],
	 data[0][2]);*/
  return OK;
}


static int statsProcessor(const char * optName,
			  unsigned long long value,
			  void * data) {
  cron_t * delta = data;
  int j;
  int found;

  found = -1;
  for (j=0;j<lsv_size;j++)
    if (0 == strcmp(optName,
		    lastStatValues[j].statName))
      found = j;
  if (found == -1) {
    found = lsv_size;
    GROW(lastStatValues,
	 lsv_size,
	 lsv_size+1);
    lastStatValues[found].statName
      = STRDUP(optName);
  }
  lastStatValues[found].lvalue
    = lastStatValues[found].value;
  lastStatValues[found].value
    = value;
  lastStatValues[found].delta
    = *delta;
  return OK;
}

/**
 * Cron-job that updates all stat values.
 */
static void updateStatValues(void * unused) {
  static cron_t lastUpdate;
  cron_t now;
  cron_t delta;

  cronTime(&now);
  delta = now - lastUpdate;
  MUTEX_LOCK(&lock);
  if (OK == requestStatistics(sock,
			      &statsProcessor,
			      &delta))
    lastUpdate = now;
  MUTEX_UNLOCK(&lock);
}


StatEntry stats[] = {
 {
    gettext_noop("Connectivity"),
    gettext_noop("# connected nodes (100% = connection table size)"),
    &getConnectedNodesStat,
    NULL,
    1,
    NO,
  },
  {
    gettext_noop("CPU load"),
    gettext_noop("CPU load (in percent of allowed load)"),
    &getCPULoadStat,
    NULL,
    1,
    NO,
  },
  {
    gettext_noop("Inbound Traffic"),
    gettext_noop("Noise (red), Content (green), Queries (yellow), other (blue)"),
    &getTrafficRecvStats,
    NULL,
    4,
    YES,
  },
  {
    gettext_noop("Outbound Traffic"),
    gettext_noop("Noise (red), Content (green), Queries (yellow), other (blue)"),
    &getTrafficSendStats,
    NULL,
    4,
    YES,
  },
  {
    NULL,
    NULL,
    NULL,
    NULL,
    1,
    NO,
  },
};

int UPDATE_INTERVAL;

void init_functions() {
  UPDATE_INTERVAL
    = getConfigurationInt("GNUNET-GTK",
			  "STATS-INTERVAL") * cronSECONDS;
  if (UPDATE_INTERVAL == 0)
    UPDATE_INTERVAL = 30 * cronSECONDS;
  sock = getClientSocket();
  MUTEX_CREATE(&lock);
  updateStatValues(NULL);
  updateConnectionGoal(NULL);
  addCronJob(&updateStatValues,
	     UPDATE_INTERVAL,
	     UPDATE_INTERVAL,
	     NULL);
  addCronJob(&updateConnectionGoal,
	     5 * cronMINUTES,
	     5 * cronMINUTES,
	     NULL);
}

void done_functions() {
  int i;
  delCronJob(&updateConnectionGoal,
	     5 * cronMINUTES,
	     NULL);
  delCronJob(&updateStatValues,
	     UPDATE_INTERVAL,
	     NULL);
  MUTEX_DESTROY(&lock);
  releaseClientSocket(sock);
  for (i=0;i<lsv_size;i++)
    FREE(lastStatValues[i].statName);
  GROW(lastStatValues,
       lsv_size,
       0);
  sock = NULL;
}


/* end of functions.c */
