/*
     This file is part of GNUnet.
     (C) 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/daemon/daemon.c
 * @brief code for gnunet-gtk gnunetd interaction
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_getoption_lib.h>
#include <GNUnet/gnunet_util_config_impl.h>
#include <GNUnet/gnunet_util_cron.h>
#include <GNUnet/gnunet_util_network_client.h>
#include <gtk/gtk.h>

static struct CronManager * cron;

static struct GE_Context * ectx;

static struct GC_Configuration * cfg;

/**
 * cron job that periodically updates the model for the
 * application list.
 */
static void * updateAppModelSafe(void * unused) {
  struct ClientServerConnection * sock;
  GtkWidget * w;
  GtkListStore * model;
  GtkTreeIter iter;
  char * apps;
  char * next;
  char * pos;
  char * desc;

  model = gtk_list_store_new(2,
			     G_TYPE_STRING,
			     G_TYPE_STRING);
  apps = NULL;
  sock = client_connection_create(ectx, cfg);
  if (sock != NULL)
    apps = getConfigurationOptionValue(sock,
				       "GNUNETD",
				       "APPLICATIONS");
  if (apps != NULL) {
    next = apps;
    do {
      while (*next == ' ')
	next++;
      pos = next;
      while ( (*next != '\0') &&
	      (*next != ' ') )
	next++;
      if (*next == '\0') {
	next = NULL; /* terminate! */
      } else {
	*next = '\0'; /* add 0-termination for pos */
	next++;
      }
      if (strlen(pos) > 0) {
	desc
	  = getConfigurationOptionValue(sock,
					"ABOUT",
					pos);
	
	gtk_list_store_append(model,
			      &iter);
	gtk_list_store_set(model,
			   &iter,
			   0, pos,
			   1, dgettext("GNUnet", desc),
			   -1);
	FREENONNULL(desc);
      }
    } while (next != NULL);
    FREE(apps);
  }
  w = glade_xml_get_widget(getMainXML(),
			   "applicationList");
  gtk_tree_view_set_model(GTK_TREE_VIEW(w),
			  GTK_TREE_MODEL(model));
  gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(w)),
			      GTK_SELECTION_NONE);
  connection_destroy(sock);
  return NULL;
}

static void updateAppModel(void * dummy) {
  gtkSaveCall(&updateAppModelSafe, NULL);
}


static void * doUpdateMenus(void * arg) {
  int ret = *(int*) arg;
  static GtkWidget * killEntry = NULL;
  static GtkWidget * launchEntry = NULL;
  static GtkWidget * statsEntryYes = NULL;
  static GtkWidget * statsEntryNo = NULL;
  static GtkWidget * statsEntryError = NULL;
  static GtkWidget * chooser = NULL;
  static int once = 1;
  static int isLocal;
  char * host;
  int canStart;
  struct GC_Configuration * dcfg;
  char * fn;
  char * user;
  struct passwd * pws;

  if (once) {
    once = 0;
    killEntry
      = glade_xml_get_widget(getMainXML(),
			     "stopDaemon");
    launchEntry
      = glade_xml_get_widget(getMainXML(),
			     "startDaemon");
    statsEntryYes
      = glade_xml_get_widget(getMainXML(),
			     "statusPixmapYes");
    statsEntryNo
      = glade_xml_get_widget(getMainXML(),
			     "statusPixmapNo");
    statsEntryError
      = glade_xml_get_widget(getMainXML(),
			     "statusPixmapError");
    chooser
      = glade_xml_get_widget(getMainXML(),
			     "gnunetdconfigFileChooserButton");
    GC_get_configuration_value_string(cfg,
				      "NETWORK",
				      "HOST",
				      "localhost",
				      &host);
    if (strcmp(host, "localhost")==0)
      isLocal = TRUE;
    else
      isLocal = FALSE;
    FREE(host);
  }
  updateAppModelSafe(NULL);
  if (ret == 0) {
    canStart = 0;
    if ( (isLocal) &&
	 (NULL != (fn = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser)))) ) {
      if (disk_file_test(ectx, fn) == YES) {
	dcfg = GC_create_C_impl();
	canStart = 1;
	if (0 != GC_parse_configuration(dcfg, fn))
	  canStart = 0;
	user = NULL;
	GC_get_configuration_value_string(dcfg,
					  "GNUNETD",
					  "USER",
					  "",
					  &user);
	if (strlen(user) > 0) {
#ifndef WINDOWS
	  if (NULL == (pws = getpwnam(user))) {
	    canStart = 0;
	  } else {
	    if (pws->pw_uid != getuid())
	      canStart = (geteuid() == 0);
	    else
	      canStart = 1;
	  }
#else
	canStart = 1;
#endif
	}
	FREE(user);
	GC_free(dcfg);
      }
      g_free(fn);
    }
    gtk_widget_hide(statsEntryYes);
    gtk_widget_set_sensitive(killEntry, FALSE);
    if (canStart && isLocal) {
      gtk_widget_set_sensitive(launchEntry, TRUE);
      gtk_widget_show_all(statsEntryNo);
      gtk_widget_hide(statsEntryError);
    } else {
      gtk_widget_set_sensitive(launchEntry, FALSE);
      gtk_widget_show_all(statsEntryError);
      gtk_widget_hide(statsEntryNo);
    }
  } else {
    gtk_widget_hide(statsEntryNo);
    gtk_widget_hide(statsEntryError);
    gtk_widget_show_all(statsEntryYes);
    gtk_widget_set_sensitive(killEntry, TRUE);
    gtk_widget_set_sensitive(launchEntry, FALSE);
  }
  return NULL;
}

static void cronCheckDaemon(void * dummy) {
  int ret;

  if (OK == connection_test_running(ectx,
				    cfg))
    ret = 1;
  else
    ret = 0;
  gtkSaveCall(&doUpdateMenus, &ret);
}


/**
 * Launch gnunetd w/ checks
 */
void on_startDaemon_clicked_daemon(GtkWidget * widget,
				   gpointer data) {
  GtkWidget * launchEntry;
  GtkWidget * chooser;
  char * fn;

  launchEntry
    = glade_xml_get_widget(getMainXML(),
			   "startDaemon");
  gtk_widget_set_sensitive(launchEntry, FALSE);
  if (OK == connection_test_running(ectx,
				    cfg) ) {
    cronCheckDaemon(NULL);
    return;
  } else {	
    addLogEntry(_("Launching gnunetd..."));
    chooser
      = glade_xml_get_widget(getMainXML(),
			     "gnunetdconfigFileChooserButton");
    fn = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
    GE_BREAK(ectx, fn != NULL);
    if (SYSERR != os_daemon_start(ectx,
				  cfg,
				  fn,
				  YES)) {
      addLogEntry(_("Launched gnunetd"));
    } else {
      addLogEntry(_("Launching gnunetd failed"));
    }
    if (fn != NULL)
      g_free(fn);
  }
}

/**
 * Kill gnunetd
 */
void on_stopDaemon_clicked_daemon(GtkWidget * widget,
				  gpointer data) {
  GtkWidget * killEntry = NULL;
  struct ClientServerConnection * sock;

  killEntry
    = glade_xml_get_widget(getMainXML(),
			   "stopDaemon");
  gtk_widget_set_sensitive(killEntry, FALSE);

  if (OK == connection_test_running(ectx,
				    cfg) ) {
    sock = client_connection_create(ectx,
				    cfg);
    if (OK != connection_request_shutdown(sock)) {
      GtkWidget * dialog;

      dialog = gtk_message_dialog_new
	(NULL,
	 GTK_DIALOG_MODAL,
	 GTK_MESSAGE_ERROR,
	 GTK_BUTTONS_CLOSE,
	 _("Error requesting shutdown of gnunetd."));
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    } else {
      addLogEntry(_("Terminating gnunetd..."));
    }
    connection_destroy(sock);
  }
  cronCheckDaemon(NULL);
}

void init_daemon(struct GE_Context * e,
		 struct GC_Configuration * c) {
  GtkWidget * tab;
  GtkWidget * apps;
  GtkListStore * model;
  GtkCellRenderer * renderer;
  int col;
  char * daemon_config;

  ectx = e;
  cfg = c;
  apps
    = glade_xml_get_widget(getMainXML(),
			   "applicationList");
  model = gtk_list_store_new(1, G_TYPE_STRING);
  gtk_tree_view_set_model(GTK_TREE_VIEW(apps),
			  GTK_TREE_MODEL(model));
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(apps),
					      -1,
					      _("Application"),
					      renderer,
					      "text", 0,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(apps),
                                              col - 1),
                                              TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(apps),
					      -1,
					      _("Description"),
					      renderer,
					      "text", 1,
					      NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(apps),
                                              col - 1),
                                              TRUE);

  updateAppModelSafe(NULL);
  tab
    = glade_xml_get_widget(getMainXML(),
			   "daemonScrolledWindow");
  gtk_widget_show(tab);
  daemon_config = NULL;
  GC_get_configuration_value_filename(cfg,
				      "DAEMON",
				      "CONFIGFILE",
				      DEFAULT_DAEMON_CONFIG_FILE,
				      &daemon_config);
  if (YES == disk_file_test(NULL, daemon_config)) {
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(glade_xml_get_widget(getMainXML(),
									"gnunetdconfigFileChooserButton")),
				  daemon_config);
  } else {
    addLogEntry(_("Configuration file for GNUnet daemon `%s' does not exist! Run `gnunet-setup -d'."),
		daemon_config);
  }
  FREE(daemon_config);

  cron = cron_create(ectx);
  cron_add_job(cron,
	       &cronCheckDaemon,
	       0,
	       15 * cronSECONDS,
	       NULL);
  cron_add_job(cron,
	       &updateAppModel,
	       5 * cronMINUTES,
	       5 * cronMINUTES,
	       NULL);
  cron_start(cron);
}

void done_daemon() {
  cron_stop(cron);
  cron_del_job(cron,
	       &cronCheckDaemon,
	       15 * cronSECONDS,
	       NULL);
  cron_del_job(cron,
	       &updateAppModel,
	       5 * cronMINUTES,
	       NULL);
  cron_destroy(cron);
}

/**
* Launch 'gnunet-setup -d wizard-gtk' with needed rights
*/
void on_startDaemonConfWizard_clicked_daemon(GtkWidget * widget,
					     gpointer data) {
  startGNUnetSetup(TRUE);
}

/**
* Launch 'gnunet-setup -d gconfig' with needed rights
*/
void on_startDaemonConfTool_clicked_daemon(GtkWidget * widget,
					   gpointer data) {
  startGNUnetSetup(FALSE);
}
/* end of daemon.c */
