/*
     This file is part of GNUnet
     (C) 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/
/**
 * @file src/common/logging.c
 * @brief This file contains GUI functions related to logging
 * @author Igor Wronsky
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <glib.h>
#include <gmodule.h>

static GtkWidget *infoWindowTextView;

static void
init ()
{
  if (infoWindowTextView != NULL)
    return;
  infoWindowTextView
    = glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (),
                            "messageWindowTextView");
}

/**
 * Do not track more than MAX messages
 */
#define MAX_LINES 1000

/**
 * Callback for GNUNET_GTK_show_info_message()
 */
static void *
doInfoMessage (void *args)
{
  char *note = args;
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  GtkTextIter end;

  init ();
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (infoWindowTextView));
  gtk_text_buffer_get_iter_at_offset (buffer, &iter, 0);
  gtk_text_buffer_insert (buffer, &iter, note, -1);
  if (gtk_text_buffer_get_line_count (buffer) >= MAX_LINES)
    {
      gtk_text_buffer_get_iter_at_line (buffer, &iter, MAX_LINES - 1);
      gtk_text_buffer_get_iter_at_line (buffer, &end, -1);
      gtk_text_buffer_delete (buffer, &iter, &end);
    }
  return NULL;
}

/**
 * Appends a message to the info window
 *
 * @param doPopup do we open the window, GNUNET_YES or GNUNET_NO
 */
void
GNUNET_GTK_show_info_message (const char *format, ...)
{
  va_list args;
  gchar *note;

  va_start (args, format);
  note = g_strdup_vprintf (format, args);
  va_end (args);
  GNUNET_GTK_save_call (&doInfoMessage, note);
  g_free (note);
}

static void *
saveAddLogEntry (void *args)
{
  static GtkWidget *s = NULL;
  static int once = 1;
  static guint id;
  char *val;

  init ();
  if (once)
    {
      once = 0;
      s =
        glade_xml_get_widget (GNUNET_GTK_get_main_glade_XML (), "statusbar");
      id = gtk_statusbar_get_context_id (GTK_STATUSBAR (s), "LOG");
    }
  else
    gtk_statusbar_pop (GTK_STATUSBAR (s), id);
  val = GNUNET_strdup ((const char *) args);
  if (strstr (val, "\n") != NULL)
    strstr (val, "\n")[0] = '\0';
  gtk_statusbar_push (GTK_STATUSBAR (s), id, val);
  GNUNET_free (val);
  return NULL;
}

/**
 * Appends a log entry to the info window
 *
 * @param txt the log entry
 *
 */
void
GNUNET_GTK_add_log_entry (const char *txt, ...)
{
  va_list args;
  gchar *note;

  va_start (args, txt);
  note = g_strdup_vprintf (txt, args);
  va_end (args);
  GNUNET_GTK_save_call (&doInfoMessage, note);
  GNUNET_GTK_save_call (&saveAddLogEntry, note);
  g_free (note);
}

static void
ge_gtk_log_handler (void *ctx,
                    GNUNET_GE_KIND kind, const char *date, const char *msg)
{
  GNUNET_GTK_show_info_message ("%s: %s", date, msg);
}

struct GNUNET_GE_Context *
GNUNET_GTK_create_gtk_logger (GNUNET_GE_KIND mask)
{
  struct GNUNET_GE_Context *myLog;

  myLog = GNUNET_GE_create_context_callback (mask,
                                             &ge_gtk_log_handler, NULL, NULL,
                                             NULL);
  return myLog;
}

/* end of logging.c */
