/* -*- Mode: c++ -*- */
/*
 * Copyright 2001,2002,2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//
// This is a demo of narrowband FM reception using the microtune 4937
// cable modem tuner module as the front end.
//

#include <GrFIRfilterCCF.h>
#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <VrFixOffset.h>
#include <VrComplexFIRfilter.h>
#include <GrFreqXlatingFIRfilterSCF.h>
#include <GrFreqXlatingFIRfilterFCF.h>
#include <VrQuadratureDemod.h>
#include <GrFIRfilterFFF.h>
#include <VrAudioSink.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>
#include <gr_firdes.h>
#include <gr_fir_builder.h>
#include <VrNullSink.h>
#include <GrIIRfilter.h>
#include <getopt.h>
#include <GrSimpleScopeSink.h>

const int inputRate = 20000000;		// input sample rate from PCI-DAS4020/12
const float IF_Freq = 5.75e6;
//const float IF_Freq = 5.735e6;
//const float IF_Freq = 5.755e6;
//const float IF_Freq = 5.745e6;

//const float channelBW = 20e3;
const float channelBW = 25e3;
const float channelSpacing = 50e3;
const float demodBW = 3e3;
const float TAU = 75e-6;  // 75us in US, 50us in EUR  

const int CFIRdecimate1 = 125;
const int CFIRdecimate2 = 4;
const int RFIRdecimate = 5;
const int CTCSSdecimate = 20;  // This gives us 200 samples per second

const float CTCSS_HIGH = 250.3;
const float CTCSS_LOW = 67;

const float CTCSS_IF = (CTCSS_HIGH + CTCSS_LOW)/2;
const float CTCSS_BW = CTCSS_HIGH - CTCSS_LOW;

const int quadRate = inputRate / CFIRdecimate1 / CFIRdecimate2;
const int audioRate = quadRate / RFIRdecimate;

static void 
usage (const char *name)
{
  cerr << "usage: " << name << " [-g] [-N]\n";
  cerr << "  -g : no gui\n";
  cerr << "  -N : use null sink instead of audio sink\n";
  exit (1);
}

int main (int argc, char **argv)
{
  try {
    
    bool use_gui_p = true;
    bool use_null_sink_p = false;

    int c;
    while ((c = getopt (argc, argv, "gN")) != EOF){
      switch (c){
      case 'g':	use_gui_p = false;    break;
      case 'N': use_null_sink_p = true; break;
      default:
	usage (argv[0]);
      }
    }
    if (optind != argc)
      usage (argv[0]);

    VrGUI *guimain = 0;
    VrGUILayout *horiz = 0;
    VrGUILayout *vert = 0;
    VrGUILayout *vert2 = 0;

    if (use_gui_p){
      guimain = new VrGUI(argc, argv);
      horiz = guimain->top->horizontal();
      vert = horiz->vertical();
      vert2 = horiz->horizontal()->vertical();
   }

    cerr << "Input Sampling Rate: " << inputRate << endl;
    cerr << "Complex1 FIR decimation factor: " << CFIRdecimate1 << endl;
    cerr << "Complex2 FIR decimation factor: " << CFIRdecimate2 << endl;
    cerr << "QuadDemod Sampling Rate: " << quadRate << endl;
    cerr << "Real FIR decimation factor: " << RFIRdecimate << endl;
    cerr << "Audio Sampling Rate: " << audioRate << endl;

    // --> short
    VrSource<short> *source = 
      make_GrMC4020SourceS(inputRate, MCC_CH3_EN | MCC_ALL_1V);

    // short --> short 
    VrFixOffset<short,short> *offset_fixer =
      new VrFixOffset<short,short>();

    // build channel filter
    //

    vector<float> channel_coeffs =
      gr_firdes::low_pass (1.0,				// gain
			   inputRate,			// sampling rate
			   channelBW/2,			// low-pass cutoff freq
			   (inputRate/CFIRdecimate1-channelBW),	// width of transition band
			   // We make this twice as wide as you would think.  This allows some
			   // aliasing, but it's outside our final bandwidth, determined by the next filter
			   gr_firdes::WIN_HANN);

    cerr << "Number of channel_coeffs: " << channel_coeffs.size () << endl;

    vector<float> channel2_coeffs =
      gr_firdes::low_pass (1.0,
			   inputRate/CFIRdecimate1,
			   channelBW/2,
			   (channelSpacing - channelBW)/2,
			   gr_firdes::WIN_HANN);

    cerr << "Number of channel2_coeffs: " << channel2_coeffs.size () << endl;
  
    // short --> VrComplex
    GrFreqXlatingFIRfilterSCF* chan_filter =
      new GrFreqXlatingFIRfilterSCF(CFIRdecimate1, channel_coeffs, IF_Freq);
    
    GrFIRfilterCCF *chan2_filter =
      new GrFIRfilterCCF (CFIRdecimate2,channel2_coeffs);



    // VrComplex --> float
    VrQuadratureDemod<float> *fm_demod =
      new VrQuadratureDemod<float>(1);

    //   vector<float> audio_coeffs =
    //    gr_firdes::low_pass (10000,				// gain
			 // 		   quadRate,			// sampling rate
    //	   demodBW,
    //   min(3e3F,audioRate/2-demodBW),
    //	   gr_firdes::WIN_HAMMING);

    vector<float> audio_coeffs =
      gr_firdes::band_pass (10000,				// gain
			    quadRate,			// sampling rate
			    300,
			    demodBW,
			    250,
			    gr_firdes::WIN_HAMMING);

  
    cerr << "Number of audio_coeffs: " << audio_coeffs.size () << endl;
  
    GrFIRfilterFFF *audio_filter = 
      new GrFIRfilterFFF(RFIRdecimate, audio_coeffs);

    vector<double> fftaps = vector<double>(2);
    vector <double> fbtaps = vector<double>(2);
    
    fftaps[0] = 1 - exp(-1/(TAU*audioRate));
    fftaps[1] = 0;
    fbtaps[0] = 0;
    fbtaps[1] = exp(-1/TAU/inputRate*50);;
    
    GrIIRfilter<float,short,double> *deemph =
      new GrIIRfilter<float,short,double> (1,fftaps,fbtaps); 

    vector<float> ctcss_coeffs =
      gr_firdes::low_pass (1.0,
			   audioRate/CTCSSdecimate,
			   CTCSS_BW/2,
			   CTCSS_BW/10,
			   gr_firdes::WIN_HANN);

    cerr << "CTCSS Rate " << audioRate/CTCSSdecimate << endl;
    cerr << "CTCSS BW " << CTCSS_BW << endl;


    GrFreqXlatingFIRfilterFCF* ctcss_filt =
      new GrFreqXlatingFIRfilterFCF(CTCSSdecimate,ctcss_coeffs, CTCSS_IF);
    
    VrQuadratureDemod<float> *ctcss_demod =
      new VrQuadratureDemod<float>(1);

    VrSink<VrComplex> *fft_sink1 = 0;
    VrSink<VrComplex> *fft_sink2 = 0;
    VrSink<short> *fft_sink3 = 0;
    VrSink<VrComplex> *fft_sink4 = 0;
    VrSink<float> *plot_sink5 = 0;

    if (use_gui_p){
      // sink1 is channel filter output
      fft_sink1 = new GrFFTSink<VrComplex>(vert, 0, 100, 512);

      // sink2 is fm demod output
      fft_sink2 = new GrFFTSink<VrComplex>(vert, 0, 100, 512);

      // sink3 is audio output
      fft_sink3 = new GrFFTSink<short>(vert, 0, 100, 512);

      fft_sink4 = new GrFFTSink<VrComplex>(vert2, 60, 160, 128);
      plot_sink5 = new GrSimpleScopeSink<float>(vert2, -10  , 10, 200);
    }

    VrSink<short> *final_sink;
    if (use_null_sink_p)
      final_sink = new VrNullSink<short>();
    else
      final_sink = new VrAudioSink<short>();

    //connect the modules together

    NWO_CONNECT (source, offset_fixer);
    NWO_CONNECT (offset_fixer, chan_filter);
    NWO_CONNECT (chan_filter, chan2_filter);
    NWO_CONNECT (chan2_filter, fm_demod);

    if (use_gui_p) {
      NWO_CONNECT (chan_filter, fft_sink1);
      NWO_CONNECT (chan2_filter, fft_sink2);
      NWO_CONNECT (deemph, fft_sink3);
      NWO_CONNECT (audio_filter, ctcss_filt);
      NWO_CONNECT (ctcss_filt, fft_sink4);
      NWO_CONNECT (ctcss_filt, ctcss_demod);
      NWO_CONNECT (ctcss_demod, plot_sink5);
    }

    NWO_CONNECT (fm_demod, audio_filter);
    NWO_CONNECT (audio_filter, deemph);
    NWO_CONNECT (deemph, final_sink);

    VrMultiTask *m = new VrMultiTask ();
    if (use_gui_p){
      m->add (fft_sink1);
      m->add (fft_sink2);
      m->add (fft_sink3);
      m->add (fft_sink4);
      m->add (plot_sink5);
    }
    
    m->add (final_sink);
    
    m->start ();
    if (use_gui_p)
      guimain->start ();

    // *((int *) 0) = 1;	// cause segv
    
    while (1){
      if (use_gui_p)
	guimain->processEvents(10 /*ms*/);
      m->process();
    }	

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
