/* -*- Mode: c++ -*- 
 *
 *  Copyright 1999 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VrSYNCCORR_H_
#define _VrSYNCCORR_H_

#define SIZE 4096
#define MAX_RESULTS_SIZE  10000

#include <VrDecimatingSigProc.h>
#include <fstream>
#if defined (ENABLE_MMX)
#include <VrMMX.h>
#endif


// ********************************************************

struct corr_result {
  float value;
  VrSampleIndex index;
};


template<class iType> 
class VrSyncCorr : public VrDecimatingSigProc<iType,char> {
protected:

  float result, old_result, *sqr, weight_sign;
  u_int first, first_high, first_low, current, mask;
  int step_size, skip, differential_offset;
  int low_interval, high_interval, best_period;
  int corr_count, pattern_length, pattern_count, corr_init;
  int window_size, excess_samples, numTaps, struct_size;
  float freq;
  virtual void get_filter_output(iType*,int,int,VrComplex*,int); 
  VrComplex *filter_results;
  VrComplex phase_corr_incr, phase_bump, phase_offset;
  VrComplex* filter_taps;
#if defined (ENABLE_MMX)
  mmxTaps* processedTaps; //Precomputed constants, shifted four times
#endif
  
public: 
  virtual void set_window(int w) {window_size = w;}
  virtual void set_period(int p) {best_period = p;}
  virtual void increment_input_ptr(int i) {differential_offset+=i;}
  virtual const char *name() { return "VrSyncCorr"; }
  virtual int forecast(VrSampleRange output,
		       VrSampleRange inputs[]);
  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]);
  virtual void initialize();
  VrSyncCorr(int, int, int, float);
  ~VrSyncCorr();
};



  /*   Correlator that looks for "low-high-low" shape in symbol sequence 
   *   
   *                                       |------------|
   *                                       |            |
   *                                       |            |
   *                                       |            |
   *      |--------------------------------|            |----------|
   *
   *      ^                 ^                                      ^
   *      |    Initialize   |       Produce output values          |  OFF
   *        
   *      |--pattern length-| <---- desired number of outputs ---->|
   *      
   *
   *      |<-- corr_init -->|
   *      |<-----  corr_on --------------------------------------->|

   */

template<class iType> int
VrSyncCorr<iType>::work(VrSampleRange output, void *ao[],
				   VrSampleRange inputs[], void *ai[])
{
  iType **i = (iType **)ai;
  char **o = (char **)ao; 
  struct corr_result best, *out_ptr;
  int size = output.size;
  float x, y, sign, *sqr;
  int current, output_time, step_size, number_outputs;
  float max, min; 
  VrSampleIndex max_loc, min_loc;
  VrSampleIndex offset = inputs[0].index;

  iType *input_ptr;
  VrComplex *f_results;

  input_ptr = i[0];
  out_ptr = (struct corr_result*) o[0];
  f_results = filter_results;


  output_time = 0; // I think that this only affects phase -> doesn't matter
  number_outputs = window_size * size / step_size + 5;
  get_filter_output(input_ptr, output_time, number_outputs, f_results, step_size);

  while(size-->0) {

    x = real(*f_results);
    y = imag(*f_results++);
    sign = (float) (2 * (x>0.0))-1;
    
    sqr[(current)&mask] =  sign*(x*x + y*y);

    if (corr_init) {    // initialize array used for correlation
	
      pattern_count++;
      if (pattern_count == 1){
	old_result = 0.0; 
	weight_sign = -1.0;
	max = 0.0;
	min = 0.0;
      }
      if (pattern_count == low_interval + 1){
	weight_sign =  1.0;
      }
      if (pattern_count == low_interval + high_interval + 1) { 
	weight_sign = -1.0;
      }
      
      old_result += weight_sign * sign * (x*x + y*y);
      current++;
      
      if (pattern_count == pattern_length){ 
	pattern_count = 0;
	corr_init = 0;
	first = current-pattern_length;
	first_high = first + low_interval;
	first_low = first_high + high_interval;
      }
    } else {           // now start producing output values
      
      result = old_result;
      
      result += sqr[(first++)&mask];
      result -= (2 * sqr[(first_high++)&mask]);
      result += (2 * sqr[(first_low++)&mask]);
      result -= sqr[(current)&mask];
      
      if (result > max) {
	max = result;
	max_loc = (current - (pattern_length/2)) * step_size;// middle of footprint
      }
      if (result < min) {
	min = result;
	min_loc = (current - (pattern_length/2)) * step_size;// middle of footprint
      }

      current++;

      old_result = result;
      }
    if (max >= -min) { //highest magnitude was positive
      best.value = max;
      best.index = max_loc + offset;
    } 
    else {
      best.value = min;
      best.index = min_loc + offset; 
    }
  } 
  *out_ptr++ =  best; 
  return output.size;
}

template<class iType> void
VrSyncCorr<iType>::get_filter_output(iType *inputArray, int output_time, 
			       int num_outputs, VrComplex *result_array, int decimation)
{
  VrComplex result = 0;
  unsigned int size = num_outputs;

  VrComplex phase_correction;
    phase_correction=pow(phase_corr_incr, output_time) * phase_offset;

  for (;size>0;size--,inputArray+=decimation) {
    result = 0;

#if defined (ENABLE_MMX)
    if(processedTaps->mmxReady())
      result = processedTaps->mmxCVDProduct(inputArray);
    else
#endif
    { 
      VrComplex *taps_tmp = filter_taps;
      for (int j=0; j < numTaps; j++)
	result += taps_tmp[j] * inputArray[j];
    }
    
    phase_correction *= phase_corr_incr;
    result *= phase_correction;
    *result_array++ = result;
  }
}

template<class iType> 
VrSyncCorr<iType>::VrSyncCorr(int dec, int step, int num_taps, float center_freq)
  :VrDecimatingSigProc<iType,char>(1,dec)
{
  numTaps = num_taps;
  freq = center_freq;
  step_size = step;
  struct_size = sizeof(struct corr_result);

}

template<class iType> int
VrSyncCorr<iType>::forecast(VrSampleRange output,
					   VrSampleRange inputs[]) {
  /* dec:1 ratio with history */
  for(unsigned int i=0;i<numberInputs;i++) {
    inputs[i].index=output.index*best_period+ differential_offset; /* ! do not subtract history ! */
    inputs[i].size=output.size*decimation + window_size + 2*excess_samples;
//jca printf ("VrSyncCorr forecast[i] index %lld size %ld output.index %lld dec %d off %d\n", inputs[i].index, inputs[i].size,
	//jca output.index, decimation, differential_offset);
  }
  return 0;
}  


template<class iType> 
void VrSyncCorr<iType>::initialize()
{

  sqr = new float[SIZE];
  mask = (u_int) (SIZE-1);
  differential_offset = 0;
  excess_samples = 0;
  

  /* template for sync pulse is [2x low + 4x high + 2x low] symbols */
  
  first = 0;
  first_high = (int) (2.0 * nominal_samples_per_symbol / (float) step_size);
  first_low = (int) (6.0 * nominal_samples_per_symbol / (float) step_size);

  current = 0;
  corr_init = 1;
  corr_count = 0;

  pattern_length = (int) (8.0 * nominal_samples_per_symbol / (float) step_size)/2*2;//even
  excess_samples = pattern_length / 2 * step_size;
  pattern_count = 0;

  filter_results = new VrComplex[MAX_RESULTS_SIZE];
  history = excess_samples;


  window_size = source_rate / sync_rate / 10;
  best_period = (int) (nominal_samples_per_symbol * symbols_per_period);

}

template<class iType> 
VrSyncCorr<iType>::~VrSyncCorr()
{
  delete sqr;
  delete filter_results;
}

#endif









