/*  GNU Robbo
 *  Copyright (C) notes:
 *  An Idea and Atari version: LK Avalon, Janusz Pelc, 1989
 *                 Linux Code: Arkadiusz Lipiec, 2002-2009
 *                                 <arkadiusz.lipiec@gmail.com>
 *                             Thunor 2007-2009
 *                                 <thunorsif@hotmail.com>
 *
 *  GNU Robbo is free software - you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  GNU Robbo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the impled warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with GNU CC; see the file COPYING. If not, write to the
 *  Free Software Foundation, 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 *
 */

#include "game.h"

/* Defines */
/*
#define DEBUG_VIDEO
#define DEBUG_MESSAGE_BOX
*/
#define BANNER_COLOURS 22 * 3
#define DELAY_BANNER 2

/* Variables */
SDL_Color bgcolor, fgcolor;	/* General purpose colours */
SDL_Surface *image;			/* General purpose surface */

/* Used for animating banners */
int bannerstate;
int bannerframe;
int bannercolour[BANNER_COLOURS] = {
	0x00ff00, 0x00f010, 0x00e020, 0x00d030, 0x00c040, 0x00b050, 0x00a060, 0x009070, 0x008080, 0x007090, 0x0060a0, 0x0050b0, 0x0060a0, 0x007090, 0x008080, 0x009070, 0x00a060, 0x00b050, 0x00c040, 0x00d030, 0x00e020, 0x00f010,
	0x00ff00, 0x10f000, 0x20e000, 0x30d000, 0x40c000, 0x50b000, 0x60a000, 0x709000, 0x808000, 0x907000, 0xa06000, 0xb05000, 0xa06000, 0x907000, 0x808000, 0x709000, 0x60a000, 0x50b000, 0x40c000, 0x30d000, 0x20e000, 0x10f000,
	0x00ff00, 0x10f010, 0x20e020, 0x30d030, 0x40c040, 0x50b050, 0x60a060, 0x709070, 0x808080, 0x907090, 0xa060a0, 0xb050b0, 0xa060a0, 0x907090, 0x808080, 0x709070, 0x60a060, 0x50b050, 0x40c040, 0x30d030, 0x20e020, 0x10f010
};

/* Used for animating icons */
int capsulestate;
int capsuleframe;
int capsuleoffsets[2 * 2] = {86, 18, 103, 18};
int teleportstate;
int teleportframe;
int teleportoffsets[2 * 2] = {1, 69, 18, 69};
int barrierstate;
int barrierframe;
int barrieroffsets[2 * 2] = {154, 52, 154, 69};
int butterflystate;
int butterflyframe;
int butterflyoffsets[2 * 2] = {137, 35, 154, 35};
int bearstate;
int bearframe;
int bearoffsets[2 * 2] = {18, 18, 35, 18};
int blackbearstate;
int blackbearframe;
int blackbearoffsets[2 * 2] = {103, 35, 120, 35};
int birdstate;
int birdframe;
int birdoffsets[2 * 2] = {52, 18, 69, 18};
int gunstate;
int gunframe;
int gunoffsets[2 * 5 * 5] = {
	1, 52, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 18, 52, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 52, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 18, 52, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 1, 52
};
int laserstate;
int laserframe;
int laseroffsets[2 * 5 * 14] = {
	1, 35, 0, 0, 0, 0, 0, 0, 0, 0, 
	1, 35, 1, 35, 0, 0, 0, 0, 0, 0, 
	1, 35, 1, 35, 1, 35, 0, 0, 0, 0, 
	1, 35, 1, 35, 1, 35, 1, 35, 0, 0, 
	1, 35, 1, 35, 1, 35, 1, 35, 1, 35, 
	1, 35, 1, 35, 1, 35, 1, 35, 0, 0, 
	1, 35, 1, 35, 1, 35, 0, 0, 0, 0, 
	1, 35, 1, 35, 0, 0, 0, 0, 0, 0, 
	1, 35, 0, 0, 0, 0, 0, 0, 0, 0, 
	35, 35, 0, 0, 0, 0, 0, 0, 0, 0, 
	18, 35, 0, 0, 0, 0, 0, 0, 0, 0, 
	35, 35, 0, 0, 0, 0, 0, 0, 0, 0, 
	52, 35, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
int blasterstate;
int blasterframe;
int blasteroffsets[2 * 5 * 9] = {
	35, 69, 0, 0, 0, 0, 0, 0, 0, 0, 
	52, 69, 35, 69, 0, 0, 0, 0, 0, 0, 
	69, 69, 52, 69, 35, 69, 0, 0, 0, 0, 
	52, 69, 69, 69, 52, 69, 35, 69, 0, 0, 
	35, 69, 52, 69, 69, 69, 52, 69, 35, 69,
	0, 0, 35, 69, 52, 69, 69, 69, 52, 69,
	0, 0, 0, 0, 35, 69, 52, 69, 69, 69,
	0, 0, 0, 0, 0, 0, 35, 69, 52, 69,
	0, 0, 0, 0, 0, 0, 0, 0, 35, 69
};
int robbostate;
int robboframe;
int robbooffsets[2 * 2] = {35, 86, 52, 86};

/* Screen banners */
char *banner[8] = {
	".OOOOO...OO...OOO..OOO..OOO.....oO#$oO..............oO#.......#$o...............",	/* 80 wide */
	"OOOOOOO..OOO..OOO..OOO..OOO.....O#$..#$.....O#$o....O#$.......$oO.........$oO#..",
	"OOO..OO..OOO..OOO..OOO..OOO.....#$o...oO...O#$.O#...#$oO#$....oO#$oO.....$oO.$o.",
	"OOO......OOOO.OOO..OOO..OOO.....$oO...O#..O#$...$o..$oO..oO...O#$..#$...$oO...O#",
	"OOO.OOO..OOOOOOOO..OOO..OOO.....oO#..O#...#$o...oO..oO#...#$..#$o...oO..oO#...#$",
	"OOO..OO..OOO.OOOO..OOO..OOO.....O#$oO#....$oO...O#..O#$...$o..$oO...O#..O#$...$o",
	"OOOOOOO..OOO..OOO..OOOOOOOO.....#$o..$o...oO#...#$..#$o...oO..oO#...#$..#$o...oO",
	".OOOOO...OOO..OOO...OOOOOO......$oO...O#...#$oO#$...$oO#$oO...O#$oO#$....oO#$oO."};

char *helpscreenbanner[10] = {
	"OOO..OOO............oO#..........",	/* 33 wide */
	"OOO..OOO....O#$o....O#$..........",
	"OOO..OOO...O#$.O#...#$o..OOOOOO..",
	"OOO..OOO..O#$...$o..$oO..OOO..OO.",
	"OOOOOOOO..#$oOOOoO..oO#..OOO...OO",
	"OOO..OOO..$oO.......$oO..OOO...OO",
	"OOO..OOO..oO#...#$..oO#..OOO..OO.",
	"OOO..OOO...#$oO#$...O#$..OOOOOO..",
	".........................OOO.....",
	".........................OOO....."};

char *optionsscreenbanner[10] = {
	".OOOOOO.............oO#....OOO..............................",	/* 60 wide */
	"OOOOOOOO............O#$....OOO....O#$o....ooo.........O#$o..",
	"OOO..OOO..OOOOOO....#$ooo........O#$.O#...#$oO#$.....O#$.O#.",
	"OOO..OOO..OOO..OO...$oO....OOO..O#$...$o..$oO..oO...O#$.....",
	"OOO..OOO..OOO...OO..oO#....OOO..#$o...oO..oO#...#$...O$oooO.",
	"OOO..OOO..OOO...OO..$oO....OOO..$oO...O#..O#$...$o........O#",
	"OOOOOOOO..OOO..OO...oO#....OOO..oO#...#$..#$o...oO..OO#...#$",
	".OOOOOO...OOOOOO.....#$oo..OOO...#$oO#$...$oO...oO...OO$o#$.",
	"..........OOO...............................................",
	"..........OOO..............................................."};

/* Function prototypes */
void show_scoreline(void);
Uint32 getpixel(SDL_Surface *surface, int x, int y);
void clear_screen(void);
void show_banner(char *banner[], int xpos, int ypos, int width, int height);
void write_lcd_char(char lcdchar, int xpos, int ypos);


/***************************************************************************
 * Show Game Area                                                          *
 ***************************************************************************/
/* Rewritten to use a viewport that can reposition itself at Robbo's
   destination in as many cycles as requested.

   On entry: redrawlevel = REDRAW_INITIALISE to initialise */

void show_game_area(int redrawlevel) {
	int x, y, xoffset = viewport.xoffset, yoffset = viewport.yoffset;
	int vpepx, vpepy;
	SDL_Rect srcrect, destrect;

	if (redrawlevel >= REDRAW_INITIALISE) {
		viewport_needs_redrawing();
		clear_screen();
		scoreline.redraw = 0xff;

		/* Maximum key/button repeat for Robbo navigation keys */
		user_controls[ACTION_UP].delay = 0;
		user_controls[ACTION_UP].interval = 0;
		user_controls[ACTION_DOWN].delay = 0;
		user_controls[ACTION_DOWN].interval = 0;
		user_controls[ACTION_LEFT].delay = 0;
		user_controls[ACTION_LEFT].interval = 0;
		user_controls[ACTION_RIGHT].delay = 0;
		user_controls[ACTION_RIGHT].interval = 0;
		/* Set a useable repeat for Robbo shoot keys */
		user_controls[ACTION_SHOOT_UP].delay = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_UP].interval = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_DOWN].delay = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_DOWN].interval = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_LEFT].delay = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_LEFT].interval = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_RIGHT].delay = game_cycle_limit / 2;
		user_controls[ACTION_SHOOT_RIGHT].interval = game_cycle_limit / 2;
	}

	/* Calculate the valid expected position of the viewport */
	vpepx = robbo.x - viewport.w / 2;
	if (vpepx < 0) {
		vpepx = 0;
	} else if (vpepx + viewport.w > level.w) {
		vpepx = level.w - viewport.w;
	}
	vpepy = robbo.y - viewport.h / 2;
	if (vpepy < 0) {
		vpepy = 0;
	} else if (vpepy + viewport.h > level.h) {
		vpepy = level.h - viewport.h;
	}
	/* If viewport.cycles_to_dest > 0 then move the viewport towards the expected position.
	   viewport.cycles_to_dest tells us how many cycles we have to get the viewport over to the expected position
	   and this is set to 1 on exit from this function. It is possible to lock the viewport at a set location
	   by setting viewport.cycles_to_dest to 0 before calling this function. It's also possible to instantly
	   place the viewport at its expected position by setting viewport.cycles_to_dest to 1 */
	if ((viewport.x != vpepx || viewport.y != vpepy) && viewport.cycles_to_dest > 0) {
		if (viewport.x != vpepx) {
			if (viewport.x < vpepx) {
				viewport.x += ceil((float)(vpepx - viewport.x) / viewport.cycles_to_dest);
			} else {
				viewport.x -= ceil((float)(viewport.x - vpepx) / viewport.cycles_to_dest);
			}
		}
		if (viewport.y != vpepy) {
			if (viewport.y < vpepy) {
				viewport.y += ceil((float)(vpepy - viewport.y) / viewport.cycles_to_dest);
			} else {
				viewport.y -= ceil((float)(viewport.y - vpepy) / viewport.cycles_to_dest);
			}
		}
		viewport.cycles_to_dest--;
		viewport_needs_redrawing();
	} else {
		viewport.cycles_to_dest = 1;
	}

	/* Iterate through the board locations in the viewport and draw the objects if required */
	for (y = viewport.y; y < viewport.y + viewport.h; y++) {
		for (x = viewport.x; x < viewport.x + viewport.w; x++) {
			destrect = set_rect(xoffset + (x - viewport.x) * video.field_size, yoffset + (y - viewport.y) * video.field_size, video.field_size, video.field_size);
			/* If the location is outside the dimensions of the level then draw a BACKGROUND_COLOUR tile instead */
			if (x < 0 || y < 0 || x >= level.w || y >= level.h) {
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));
			} else {
				/* The location is valid but only draw locations that have been marked for redrawing */
				if (board[x][y].redraw) {
					board[x][y].redraw = FALSE;
					if (level.now_is_blinking > 1) {
						/* If the level is blinking then draw a white tile */
						SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, 0xff, 0xff, 0xff));
					} else {
						if (level.colour_override != UNDEFINED) {
							SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, level.colour_override >> 16 & 0xff, level.colour_override >> 8 & 0xff, level.colour_override & 0xff));
						} else if (level.colour != UNDEFINED) {
							SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, level.colour >> 16 & 0xff, level.colour >> 8 & 0xff, level.colour & 0xff));
						} else {
							SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, DEFAULT_LEVEL_COLOUR >> 16 & 0xff, DEFAULT_LEVEL_COLOUR >> 8 & 0xff, DEFAULT_LEVEL_COLOUR & 0xff));
						}
					}
					/* Get icon surface offsets for object's current animation frame */
					if (board[x][y].type != EMPTY_FIELD) {
						srcrect = set_rect(board[x][y].icon[board[x][y].state].x, board[x][y].icon[board[x][y].state].y, video.field_size, video.field_size);
						if (SDL_BlitSurface(icons, &srcrect, screen, &destrect) < 0) {
							fprintf(stdout,"BlitSurface error: %s\n", SDL_GetError());
						}
					}
					/* Are we drawing Robbo? */
					if (robbo.x == x && robbo.y == y && robbo.alive && /* Is this his location and is he alive? */
						(!robbo.teleporting || (robbo.teleporting && robbo.moved <= DELAY_TELEPORTING))) {	/* Is he either not teleporting or just about to materialise from teleporting? */
						srcrect = set_rect(robbo_images_srcrect[robbo.direction + robbo.state].x, robbo_images_srcrect[robbo.direction + robbo.state].y, robbo_images_srcrect[robbo.direction + robbo.state].w, robbo_images_srcrect[robbo.direction + robbo.state].h);
						if (SDL_BlitSurface(icons, &srcrect, screen, &destrect) < 0) {
							fprintf(stdout,"BlitSurface error: %s\n", SDL_GetError());
						}
					}
				}
			}
		}
	}

	/* If the level is blinking then mark the viewport for redrawing */
	if (level.now_is_blinking > 1) {
		level.now_is_blinking--;
		if (level.now_is_blinking == 1) {
			viewport_needs_redrawing();
		}
	}

	/* Redraw the parts of the scoreline that have been requested */
	if (scoreline.redraw) show_scoreline();

}

/***************************************************************************
 * Show Game Area Fade                                                     *
 ***************************************************************************/
/* Simulates the screen fading in. I think fades can become annoying, so I
   don't fade when manually changing levels and packs.
   
   On entry: redrawlevel = REDRAW_INITIALISE to initialise with
				type = 1 to 17 for a specific fade type or
			 	type = 0 to kill an existing fade
			 redrawlevel = REDRAW_EVERYTHING for everything */

void show_game_area_fade(int redrawlevel, int type) {
	SDL_Rect destrect;
	static int fadesize = 0, fadetype;
	int xoffset = viewport.xoffset, yoffset = viewport.yoffset, x, y;

	if (redrawlevel == REDRAW_INITIALISE) {

		/* Initialise a fade */
		fadetype = type;
		if (fadetype == 0) {
			fadesize = 0;
		} else if (fadetype == 16) {
			fadesize = viewport.w;
		} else {
			fadesize = video.field_size;
		}

	} else if (redrawlevel == REDRAW_EVERYTHING) {

		if (fadesize > 0) {
	
			for (y = 0; y < viewport.h; y++) {
				for (x = 0; x < viewport.w; x++) {

					if (fadetype == 0) {
						/* No fade */
					} else if (fadetype == 1) {
						/* Fade middle dot */
						destrect = set_rect(x * video.field_size + xoffset + (video.field_size - fadesize) / 2, y * video.field_size + yoffset + (video.field_size - fadesize) / 2, fadesize, fadesize);
					} else if (fadetype == 2) {
						/* Fade bottom right */
						destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
					} else if (fadetype == 3) {
						/* Fade top left and bottom right */
						if (x % 2 == 0) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, fadesize, fadesize);
						} else if (x % 2 == 1) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						}
					} else if (fadetype == 4) {
						/* Fade bottom left and top right */
						if (x % 2 == 0) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						} else if (x % 2 == 1) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset, fadesize, fadesize);
						}
					} else if (fadetype == 5) {
						/* Fade bottom */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + video.field_size - fadesize, video.field_size, fadesize);
					} else if (fadetype == 6) {
						/* Fade top left */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, fadesize, fadesize);
					} else if (fadetype == 7) {
						/* Fade top right */
						destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset, fadesize, fadesize);
					} else if (fadetype == 8) {
						/* Fade bottom left */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
					} else if (fadetype == 9) {
						/* Intermittent fade top left and bottom right */
						if (fadesize % 4 == 0) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, fadesize, fadesize);
						} else if (fadesize % 4 == 2) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						}
					} else if (fadetype == 10) {
						/* Fade left */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, fadesize, video.field_size);
					} else if (fadetype == 11) {
						/* Fade middle tall */
						destrect = set_rect(x * video.field_size + xoffset + (video.field_size - fadesize) / 2, y * video.field_size + yoffset, fadesize, video.field_size);
					} else if (fadetype == 12) {
						/* Fade middle wide */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + (video.field_size - fadesize) / 2, video.field_size, fadesize);
					} else if (fadetype == 13) {
						/* Fade top */
						destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, video.field_size, fadesize);
					} else if (fadetype == 14) {
						/* Fade right */
						destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset, fadesize, video.field_size);
					} else if (fadetype == 15) {
						/* Intermittent fade bottom left and top right */
						if (fadesize % 4 == 0) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						} else if (fadesize % 4 == 2) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset, fadesize, fadesize);
						}
					} else if (fadetype == 16) {
						/* Board fades from the right */
						if (x < fadesize) destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, video.field_size, video.field_size);
					} else if (fadetype == 17) {
						/* Fade top left, top right, bottom left and bottom right */
						if (x % 2 == 0 && y % 2 == 0) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset, fadesize, fadesize);
						} else if (x % 2 == 1 && y % 2 == 0) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset, fadesize, fadesize);
						} else if (x % 2 == 0 && y % 2 == 1) {
							destrect = set_rect(x * video.field_size + xoffset, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						} else if (x % 2 == 1 && y % 2 == 1) {
							destrect = set_rect(x * video.field_size + xoffset + video.field_size - fadesize, y * video.field_size + yoffset + video.field_size - fadesize, fadesize, fadesize);
						}
					}

					SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].fade_colour >> 16 & 0xff, skins[selected_skin].fade_colour >> 8 & 0xff, skins[selected_skin].fade_colour & 0xff));

				}
			}
			
			if (fadetype == 16) {
				fadesize--;
			} else {
				fadesize -= video.field_size / 16;
			}
			
			viewport_needs_redrawing();
			
		}
	}
}

/***************************************************************************
 * Show Scoreline                                                          *
 ***************************************************************************/
/* Now each item can be individually redrawn by the setting of bits in
   the global variable scoreline.redraw prior to calling this function */
  
void show_scoreline(void) {
	int xoffset, count, number;
	int yoffset = scoreline.yoffset;
	char tempstring[256];
	SDL_Rect destrect;

	if (scoreline.redraw & SCORELINE_ICONS) {
		xoffset = scoreline.xoffset;

		/* Draw screw icon */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		if (SDL_BlitSurface(icons, &score_screw_srcrect, screen, &destrect) < 0) {
			fprintf(stdout, "BlitSurface error: %s\n", SDL_GetError());   
		}
		
		xoffset = xoffset + video.field_size * 2 + video.field_size / 4;

		/* Draw key icon */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		if (SDL_BlitSurface(icons, &score_key_srcrect, screen, &destrect) < 0) {
			fprintf(stdout, "BlitSurface error: %s\n", SDL_GetError());
		}

		xoffset = xoffset + video.field_size * 2 + video.field_size / 2;

		/* Draw bullet icon */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		if (SDL_BlitSurface(icons, &score_bullet_srcrect, screen, &destrect) < 0) {
			fprintf(stdout, "BlitSurface error: %s\n", SDL_GetError());
		}

		xoffset = xoffset + video.field_size * 3;

		/* Draw level icon */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		if (SDL_BlitSurface(icons, &score_level_srcrect, screen, &destrect) < 0) {
			fprintf(stdout, "BlitSurface error: %s\n", SDL_GetError());
		}
	}

	if (scoreline.redraw & SCORELINE_SCREWS) {
		xoffset = scoreline.xoffset + video.field_size;

		/* First erase the area occupied by the digits */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));

		/* Draw screw count */
		number = 10;
		for (count = 0; count < 2; count++) {
			sprintf(tempstring, "%i", (robbo.screws / number) % 10);
			write_lcd_char(tempstring[0], xoffset, yoffset);
			xoffset = xoffset + video.field_size / 2;
			number = number / 10;
		}
	}

	if (scoreline.redraw & SCORELINE_KEYS) {
		xoffset = scoreline.xoffset + video.field_size * 3 + video.field_size / 2;

		/* First erase the area occupied by the digits */
		destrect = set_rect(xoffset, yoffset, video.field_size, video.field_size);
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));

		/* Draw key count */
		number = 10;
		for (count = 0; count < 2; count++) {
			sprintf(tempstring, "%i", (robbo.keys / number) % 10);
			write_lcd_char(tempstring[0], xoffset, yoffset);
			xoffset = xoffset + video.field_size / 2;
			number = number / 10;
		}
	}

	if (scoreline.redraw & SCORELINE_BULLETS) {
		xoffset = scoreline.xoffset + video.field_size * 6;

		/* First erase the area occupied by the digits */
		destrect = set_rect(xoffset, yoffset, video.field_size + video.field_size / 2, video.field_size);
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));

		/* Draw bullet count */
		number = 100;
		for (count = 0; count < 3; count++) {
			sprintf(tempstring, "%i", (robbo.bullets / number) % 10);
			write_lcd_char(tempstring[0], xoffset, yoffset);
			xoffset = xoffset + video.field_size / 2;
			number = number / 10;
		}
	}

	if (scoreline.redraw & SCORELINE_LEVEL) {
		xoffset = scoreline.xoffset + video.field_size * 9;

		/* First erase the area occupied by the digits */
		destrect = set_rect(xoffset, yoffset, video.field_size + video.field_size / 2, video.field_size);
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));

		/* Draw level count */
		number = 100;
		for (count = 0; count < 3; count++) {
			sprintf(tempstring, "%i", (level_packs[selected_pack].level_selected / number) % 10);
			write_lcd_char(tempstring[0], xoffset, yoffset);
			xoffset = xoffset + video.field_size / 2;
			number = number / 10;
		}
	}

	if (scoreline.redraw & SCORELINE_PACK) {
		xoffset = scoreline.xoffset + video.field_size * 10 + video.field_size / 2;

		/* First erase the area occupied by the digits */
		destrect = set_rect(xoffset, yoffset, video.field_size * 5 + video.field_size / 2, video.field_size);
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));

		/* Draw "-" */
		tempstring[0] = '-';
		write_lcd_char(tempstring[0], xoffset, yoffset);

		/* Draw pack name */
		for (count = 0; count < strlen(level_packs[selected_pack].name); count++) {
			write_lcd_char(level_packs[selected_pack].name[count], xoffset + (count + 1) * video.field_size / 2, yoffset);
		}
	}

	if (scoreline.redraw & SCORELINE_AUTHOR) {
		fgcolor.r = skins[selected_skin].author_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].author_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].author_text_colour & 0xff;
		bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;

		/* Draw level author */
		if (level.author[0] == '_') {
			sprintf(tempstring,"%s", &level.author[1]);
		} else if (strlen(level.author) > 0) {
			sprintf(tempstring,"%s: %s", txt_Level_Author, level.author);
		} else {
			strcpy(tempstring, "");
		}
		if (strlen(tempstring) > 0) {
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(authorline.xoffset, authorline.yoffset, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}
	}

	scoreline.redraw = 0;
}

/*************************************************************/
/* Set Rect **************************************************/
/*************************************************************/

SDL_Rect set_rect(int x, int y, int w, int h)
{
	SDL_Rect rect;
	rect.x = x;
	rect.y = y;
	rect.w = w;
	rect.h = h;
	return rect;
}

/*************************************************************/
/* Get Pixel *************************************************/
/*************************************************************/
/* Not used in this project anymore */

Uint32 getpixel(SDL_Surface *surface, int x, int y) 
{
	int bpp = surface->format->BytesPerPixel;
	/* Here p is the address to the pixel we want to retrieve */
	Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * bpp;
            
	switch(bpp) {
		case 1:
			return *p;
		case 2:
			return *(Uint16 *)p;
		case 3:
			if(SDL_BYTEORDER == SDL_BIG_ENDIAN)
				return p[0] << 16 | p[1] << 8 | p[2];
			else
				return p[0] | p[1] << 8 | p[2] << 16;
		case 4:
			return *(Uint32 *)p;
		default:
			return 0;       /* shouldn't happen, but avoids warnings */
	}
}

/*************************************************************/
/* Clear Screen***********************************************/
/*************************************************************/

void clear_screen(void)
{
	SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));
}

/***************************************************************************
 * Set Video Mode                                                          *
 ***************************************************************************/
/* Now allows the setting of any resolution with a choice of field_sizes */

/*	On exit: returns 1 on error */   

int set_video_mode(void) {
	SDL_Rect **modes;
	int hwsurface = FALSE, count, flags;
	#ifdef DEBUG_VIDEO	
		const SDL_VideoInfo *videoinfo;
	#endif
	char description[256];

	/* Validate the screen resolution and field_size that the user may have requested */
	if (video.xres == UNDEFINED || video.yres == UNDEFINED) {
		video.xres = 640; video.yres = 480;
		if (video.field_size == UNDEFINED) video.field_size = 32;
	} else {
		if (video.field_size == UNDEFINED) video.field_size = 32;
		if (video.xres < 480 || video.yres < 480) video.field_size = 16;
	}
	/* Force GP2X screen resolution as there is no option */
	#if defined(PLATFORM_GP2X)
		video.xres = 320; video.yres = 240; video.field_size = 16;
	#endif
	/* Validate fullscreen that the user may have requested */
	if (video.fullscreen == UNDEFINED) video.fullscreen = 0;
	#if defined(PLATFORM_GP2X)
		video.fullscreen = SDL_FULLSCREEN;
	#elif defined(PLATFORM_ZAURUS)
		video.fullscreen = SDL_FULLSCREEN;
	#endif
	/* Validate the viewport dimensions */
	if (viewport.max_w == UNDEFINED || viewport.max_h == UNDEFINED) {
		viewport.max_w = DEFAULT_VIEWPORT_WIDTH;
		viewport.max_h = DEFAULT_VIEWPORT_HEIGHT;
	}
	if (viewport.maximise == UNDEFINED) viewport.maximise = FALSE;
	
	#ifdef DEBUG_VIDEO	
		printf("*** Start %s ***\n", __func__);
		printf ("Fullscreen hwsurfaces :-\n");
	#endif
	/* Query available fullscreen hwsurface modes */
	modes = SDL_ListModes(NULL, SDL_FULLSCREEN | SDL_HWSURFACE);
	if (modes == NULL) {
		#ifdef DEBUG_VIDEO	
			printf ("None found\n");
		#endif
	} else if (modes == (SDL_Rect **) -1) {
		#ifdef DEBUG_VIDEO	
			printf("Any dimension is okay\n");
		#endif
		hwsurface = TRUE;
	} else {
		for (count = 0; modes[count]; ++count) {
			#ifdef DEBUG_VIDEO	
				printf("%dx%d\n", modes[count]->w, modes[count]->h);
			#endif
			if (modes[count]->w == video.xres && modes[count]->h == video.yres) {
				hwsurface = TRUE;
			}
		}
	}
	#ifdef DEBUG_VIDEO	
		printf ("Fullscreen swsurfaces :-\n");
	#endif
	/* Query available fullscreen swsurface modes */
	modes = SDL_ListModes(NULL, SDL_FULLSCREEN | SDL_SWSURFACE);
	if (modes == NULL) {
		#ifdef DEBUG_VIDEO	
			printf ("None found\n");
		#endif
	} else if (modes == (SDL_Rect **) -1) {
		#ifdef DEBUG_VIDEO	
			printf("Any dimension is okay\n");
		#endif
	} else {
		for (count = 0; modes[count]; ++count) {
			#ifdef DEBUG_VIDEO	
				printf("%dx%d\n", modes[count]->w, modes[count]->h);
			#endif
		}
	}
	#ifdef DEBUG_VIDEO	
		videoinfo = SDL_GetVideoInfo();
		printf ("Best video mode info :-\n");
		printf ("hw_available: %u\n", videoinfo->hw_available);
		printf ("wm_available: %u\n", videoinfo->wm_available);
		printf ("blit_hw: %u\n", videoinfo->blit_hw);
		printf ("blit_hw_CC: %u\n", videoinfo->blit_hw_CC);
		printf ("blit_hw_A: %u\n", videoinfo->blit_hw_A);
		printf ("blit_sw: %u\n", videoinfo->blit_sw);
		printf ("blit_sw_CC: %u\n", videoinfo->blit_sw_CC);
		printf ("blit_sw_A: %u\n", videoinfo->blit_sw_A);
		printf ("blit_fill: %u\n", videoinfo->blit_fill);
		printf ("video_mem: %uKB\n", videoinfo->video_mem);
		printf ("BitsPerPixel: %u\n", videoinfo->vfmt->BitsPerPixel);
		printf ("BytesPerPixel: %u\n", videoinfo->vfmt->BytesPerPixel);
	#endif

	/* Well after all that clever video mode detection code the GP2X
	   GPH SDL states it has hwsurface support and then seg faults ;) */
	#if defined(PLATFORM_GP2X)
		hwsurface = FALSE;
	#endif
	
	/* We have enough info to set the correct mode now */
	if (hwsurface) {
		flags = SDL_DOUBLEBUF | SDL_HWSURFACE | SDL_ANYFORMAT | video.fullscreen;
		strcpy(description, "double buffered hardware");
	} else {
		flags = SDL_SWSURFACE | SDL_ANYFORMAT | video.fullscreen;
		strcpy(description, "software");
	}
	/* Attempt to set the requested video mode or fall back to some defaults */
	for (count = 0; count < 3; count++) {
		if (count == 0) {
			fprintf(stdout, "Setting video mode %ix%i 16bpp %s surface.\n", video.xres, video.yres, description);
		} else if (count == 1) {
			video.xres = 640; video.yres = 480; video.field_size = 32;
			fprintf(stdout, "Trying default hi-res video mode %ix%i 16bpp %s surface.\n", video.xres, video.yres, description);
		} else if (count == 2) {
			video.xres = 320; video.yres = 240; video.field_size = 16;
			fprintf(stdout, "Trying default low-res video mode %ix%i 16bpp %s surface.\n", video.xres, video.yres, description);
		}
		screen = SDL_SetVideoMode(video.xres, video.yres, 16, flags);
		if (screen != NULL) {
			break;
		} else if (count < 2) {
			fprintf(stdout, "Failed: %s\n", SDL_GetError());
		} else {
			return 1;
		}
	}

	/* We won't be needing the mouse pointer if full screen */
	if (video.fullscreen == SDL_FULLSCREEN) SDL_ShowCursor(SDL_DISABLE);

	#ifdef DEBUG_VIDEO	
		videoinfo = SDL_GetVideoInfo();
		printf ("Current video mode info :-\n");
		printf ("hw_available: %u\n", videoinfo->hw_available);
		printf ("wm_available: %u\n", videoinfo->wm_available);
		printf ("blit_hw: %u\n", videoinfo->blit_hw);
		printf ("blit_hw_CC: %u\n", videoinfo->blit_hw_CC);
		printf ("blit_hw_A: %u\n", videoinfo->blit_hw_A);
		printf ("blit_sw: %u\n", videoinfo->blit_sw);
		printf ("blit_sw_CC: %u\n", videoinfo->blit_sw_CC);
		printf ("blit_sw_A: %u\n", videoinfo->blit_sw_A);
		printf ("blit_fill: %u\n", videoinfo->blit_fill);
		printf ("video_mem: %uKB\n", videoinfo->video_mem);
		printf ("BitsPerPixel: %u\n", videoinfo->vfmt->BitsPerPixel);
		printf ("BytesPerPixel: %u\n", videoinfo->vfmt->BytesPerPixel);
		printf("*** Stop %s ***\n", __func__);
	#endif

	return 0;
} 

/***************************************************************************
 * Toggle Fullscreen                                                       *
 ***************************************************************************/
/* Toggles between a window and fullscreen on supported platforms */   

void toggle_fullscreen(int *fullscreen) {
	if (!(SDL_WM_ToggleFullScreen(screen))) {
		fprintf(stdout, "Unable to toggle fullscreen: %s\n", SDL_GetError());
	}

	/* We won't be needing the mouse pointer if full screen */
	/* Be careful to record the current fullscreen state */
	if (*fullscreen != SDL_FULLSCREEN) {
		*fullscreen = SDL_FULLSCREEN;
		SDL_ShowCursor(SDL_DISABLE);
	} else {
		*fullscreen = 0;
		SDL_ShowCursor(SDL_ENABLE);
	}

}

/***************************************************************************
 * Show Level Colour                                                       *
 ***************************************************************************/
/* Used for debugging */   

void show_level_colour(int debug_colour_select_r, int debug_colour_select_g, int debug_colour_select_b, int debug_colour_select_component) {
	char tempstring[60];
	int xoffset, yoffset;
	SDL_Rect destrect;
	
	xoffset = authorline.xoffset + video.field_size * 13.5;
	yoffset = authorline.yoffset;

	/* Display red component */
	if (debug_colour_select_component == 0) {
		bgcolor.r = 0xff; bgcolor.g = 0; bgcolor.b = 0;
		fgcolor.r = 0; fgcolor.g = 0; fgcolor.b = 0;
	} else {
		bgcolor.r = 0; bgcolor.g = 0; bgcolor.b = 0;
		fgcolor.r = 0xff; fgcolor.g = 0xff; fgcolor.b = 0xff;
	}
	sprintf(tempstring, "%02X", debug_colour_select_r);
	image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
	destrect = set_rect(xoffset, yoffset, image->w, image->h);
	SDL_BlitSurface(image, NULL, screen, &destrect);
	xoffset += image->w;
	SDL_FreeSurface(image);
	/* Display green component */
	if (debug_colour_select_component == 1) {
		bgcolor.r = 0; bgcolor.g = 0xff; bgcolor.b = 0;
		fgcolor.r = 0; fgcolor.g = 0; fgcolor.b = 0;
	} else {
		bgcolor.r = 0; bgcolor.g = 0; bgcolor.b = 0;
		fgcolor.r = 0xff; fgcolor.g = 0xff; fgcolor.b = 0xff;
	}
	sprintf(tempstring, "%02X", debug_colour_select_g);
	image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
	destrect = set_rect(xoffset, yoffset, image->w, image->h);
	SDL_BlitSurface(image, NULL, screen, &destrect);
	xoffset += image->w;
	SDL_FreeSurface(image);
	/* Display blue component */
	if (debug_colour_select_component == 2) {
		bgcolor.r = 0; bgcolor.g = 0; bgcolor.b = 0xff;
		fgcolor.r = 0; fgcolor.g = 0; fgcolor.b = 0;
	} else {
		bgcolor.r = 0; bgcolor.g = 0; bgcolor.b = 0;
		fgcolor.r = 0xff; fgcolor.g = 0xff; fgcolor.b = 0xff;
	}
	sprintf(tempstring, "%02X", debug_colour_select_b);
	image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
	destrect = set_rect(xoffset, yoffset, image->w, image->h);
	SDL_BlitSurface(image, NULL, screen, &destrect);
	SDL_FreeSurface(image);
}

/***************************************************************************
 * Inc Colour Component                                                    *
 ***************************************************************************/
/* Used for debugging */   

void inc_colour_component(int *colour_component) {
	if (*colour_component == 0xf8) {
		*colour_component = 0xff;
	} else if (*colour_component == 0xff) {
		*colour_component = 0x0;
	} else {
		*colour_component += 0x8;
	}
}

/***************************************************************************
 * Dec Colour Component                                                    *
 ***************************************************************************/
/* Used for debugging */   

void dec_colour_component(int *colour_component) {
	if (*colour_component == 0x0) {
		*colour_component = 0xff;
	} else if (*colour_component == 0xff) {
		*colour_component = 0xf8;
	} else {
		*colour_component -= 0x8;
	}
}

/***************************************************************************
 * Show Intro Screen                                                       *
 ***************************************************************************/
/* On entry: redrawlevel = REDRAW_INITIALISE to initialise
			 redrawlevel = REDRAW_EVERYTHING for everything
			 redrawlevel = REDRAW_INTERMEDIATE for menu
			 redrawlevel = REDRAW_ANIMATED for animated objects only */

void show_introscreen(int redrawlevel) {
	SDL_Rect destrect;
	int count, xoffset, yoffset;
	char tempstring[256];
	
	xoffset = (screen->w - video.field_size * 15) / 2;
	yoffset = (screen->h - video.field_size * 15) / 2;
	
	bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;

	if (redrawlevel >= REDRAW_INITIALISE) {
		/* Reinitialise some variables */
		bannerframe = 0;
		
		/* Restore normal key/button repeat values for these keys */
		user_controls[ACTION_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_RIGHT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_RIGHT].interval = KEY_REPEAT_INTERVAL;
	}
	
	if (redrawlevel >= REDRAW_EVERYTHING) {
		/* Reinitialise some variables */
		bannerstate = 0;

		/* Set entire screen to a colour */
		clear_screen();
		
		/* Draw the version */
		fgcolor.r = skins[selected_skin].version_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].version_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].version_text_colour & 0xff;
		image = TTF_RenderUTF8_Shaded(font, VERSION, fgcolor, bgcolor);
		if (video.field_size == 16) {
			destrect = set_rect(xoffset + 217, yoffset + 37, image->w, image->h);
		} else if (video.field_size == 32) {
			destrect = set_rect(xoffset + 438, yoffset + 55, image->w, image->h);
		}
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);

		/* Draw the intro story */
		fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
		for (count = 0; count < TXT_INTRO_STORY_ROWS; count++) {
			image = TTF_RenderUTF8_Shaded(font, &txt_intro_story[count * TXT_MAX_COLS], fgcolor, bgcolor);
			SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
			if (video.field_size == 16) {
				destrect = set_rect(xoffset + 2, yoffset + 48 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
			} else if (video.field_size == 32) {
				destrect = set_rect(xoffset + 4, yoffset + 96 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
			}
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}

		/* Draw the intro keys */
		fgcolor.r = skins[selected_skin].default_controls_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].default_controls_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].default_controls_text_colour & 0xff;
		for (count = 0; count < TXT_INTRO_KEYS_ROWS; count++) {
			image = TTF_RenderUTF8_Shaded(font, &txt_intro_keys[count * TXT_MAX_COLS], fgcolor, bgcolor);
			SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
			if (video.field_size == 16) {
				destrect = set_rect(xoffset + 6, yoffset + 105 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
			} else if (video.field_size == 32) {
				destrect = set_rect(xoffset + 12, yoffset + 210 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
			}
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}

		/* Draw the intro keys description */
		fgcolor.r = skins[selected_skin].default_controls_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].default_controls_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].default_controls_text_colour & 0xff;
		for (count = 0; count < TXT_INTRO_KEYS_DESCRIPTION_ROWS; count++) {
			image = TTF_RenderUTF8_Shaded(font, &txt_intro_keys_description[count * TXT_MAX_COLS], fgcolor, bgcolor);
			SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
			if (video.field_size == 16) {
				destrect = set_rect(xoffset + 79, yoffset + 105 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
			} else if (video.field_size == 32) {
				destrect = set_rect(xoffset + 158, yoffset + 210 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
			}
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}

		/* Draw the intro credits */
		fgcolor.r = skins[selected_skin].credits_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].credits_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].credits_text_colour & 0xff;
		for (count = 0; count < TXT_INTRO_CREDITS_ROWS; count++) {
			image = TTF_RenderUTF8_Shaded(font, &txt_intro_credits[count * TXT_MAX_COLS], fgcolor, bgcolor);
			SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
			if (video.field_size == 16) {
				destrect = set_rect(screen->w / 2 - image->w / 2, yoffset + 199 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
			} else if (video.field_size == 32) {
				destrect = set_rect(screen->w / 2 - image->w / 2, yoffset + 398 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
			}
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}
	}
	
	if (redrawlevel >= REDRAW_INTERMEDIATE) {
		/* Erase the menu text area only */
		if (video.field_size == 16) {
			destrect = set_rect(xoffset + 126, yoffset + 105, 112, 86);
		} else if (video.field_size == 32) {
			destrect = set_rect(xoffset + 252, yoffset + 210, 224, 172);
		}
		SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, skins[selected_skin].background_colour >> 16 & 0xff, skins[selected_skin].background_colour >> 8 & 0xff, skins[selected_skin].background_colour & 0xff));
	
		/* Draw the intro menu text */
		for (count = 0; count < TXT_INTRO_MENU_ROWS; count++) {
			if (count - 1 == introscreenmenuposition) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, &txt_intro_menu[count * TXT_MAX_COLS], fgcolor, bgcolor);
			SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
			if (video.field_size == 16) {
				destrect = set_rect(xoffset + 129, yoffset + 105 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
			} else if (video.field_size == 32) {
				destrect = set_rect(xoffset + 258, yoffset + 210 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
			}
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}

		/* Draw the current level */
		/* < */
		strcpy(tempstring, "< ");
		if (level_packs[selected_pack].level_selected > 1) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		if (video.field_size == 16) {
			destrect = set_rect(xoffset + 164, yoffset + 129, image->w, image->h);
		} else if (video.field_size == 32) {
			destrect = set_rect(xoffset + 330, yoffset + 258, image->w, image->h);
		}
		SDL_BlitSurface(image, NULL, screen, &destrect);
		destrect.x += image->w;
		SDL_FreeSurface(image);
		/* 999 */
		sprintf(tempstring, "%i %s %i", level_packs[selected_pack].level_selected, txt_of, level_packs[selected_pack].last_level);
		fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		destrect.w = image->w; destrect.h = image->h;
		SDL_BlitSurface(image, NULL, screen, &destrect);
		destrect.x += image->w;
		SDL_FreeSurface(image);
		/* > */
		strcpy(tempstring, " >");
		if (level_packs[selected_pack].level_selected < level_packs[selected_pack].level_reached) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		destrect.w = image->w; destrect.h = image->h;
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);

		/* Draw the current pack */
		/* < */
		strcpy(tempstring, "< ");
		if (selected_pack > 0) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		if (video.field_size == 16) {
			destrect = set_rect(xoffset + 165, yoffset + 141, image->w, image->h);
		} else if (video.field_size == 32) {
			destrect = set_rect(xoffset + 330, yoffset + 282, image->w, image->h);
		}
		SDL_BlitSurface(image, NULL, screen, &destrect);
		destrect.x += image->w;
		SDL_FreeSurface(image);
		/* "Original" */
		fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
		image = TTF_RenderUTF8_Shaded(font, level_packs[selected_pack].name, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		destrect.w = image->w; destrect.h = image->h;
		SDL_BlitSurface(image, NULL, screen, &destrect);
		destrect.x += image->w;
		SDL_FreeSurface(image);
		/* > */
		strcpy(tempstring, " >");
		if (selected_pack < found_pack_count - 1) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
		destrect.w = image->w; destrect.h = image->h;
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);
	}
	
	if (redrawlevel >= REDRAW_ANIMATED) {
		bannerstate--;
		if (bannerstate <= 0) {
			/* Draw the horizontal line under the GNU Robbo banner */
			if (video.field_size == 16) {
				destrect = set_rect(xoffset + 0, yoffset + 43, 217, video.field_size / 8);
			} else if (video.field_size == 32) {
				destrect = set_rect(xoffset + 0, yoffset + 86, 480, video.field_size / 8);
			}
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			/* Draw the horizontal line above the controls area */
			destrect = set_rect(xoffset + 0, yoffset + 102 * video.field_size / 16, 240 * video.field_size / 16, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			/* Draw the horizontal line below the controls area */
			destrect = set_rect(xoffset + 0, yoffset + 194 * video.field_size / 16, 240 * video.field_size / 16, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			/* Draw the vertical line to the left of the controls area */
			destrect = set_rect(xoffset + 0, yoffset + 102 * video.field_size / 16, video.field_size / 8, 92 * video.field_size / 16);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			/* Draw the vertical line in the middle of the controls area */
			destrect = set_rect(xoffset + 123 * video.field_size / 16, yoffset + 102 * video.field_size / 16, video.field_size / 8, 92 * video.field_size / 16);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			/* Draw the vertical line to the right of the controls area */
			destrect = set_rect(xoffset + 238 * video.field_size / 16, yoffset + 102 * video.field_size / 16, video.field_size / 8, 92 * video.field_size / 16);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));

			/* Draw the GNU Robbo banner */
			if (video.field_size == 16) {
				show_banner(banner, xoffset + 0, yoffset + 6, 80, 8);
			} else if (video.field_size == 32) {
				show_banner(banner, xoffset + 24, yoffset + 19, 80, 8);
			}

			bannerstate = DELAY_BANNER;
		}
	}
}

/***************************************************************************
 * Show Help Screen                                                        *
 ***************************************************************************/
/* On entry: redrawlevel = REDRAW_INITIALISE to initialise
			 redrawlevel = REDRAW_EVERYTHING for everything
			 redrawlevel = REDRAW_INTERMEDIATE for menu
			 redrawlevel = REDRAW_ANIMATED for animated objects only */

void show_helpscreen(int redrawlevel) {
	SDL_Rect srcrect, destrect;
	int xoffset, yoffset, yincrement = video.field_size + video.field_size / 4;
	Uint32 tilecolour = skins[selected_skin].help_tile_colour;	/* 0x709070 0x60a000 0x907090 0x961B2B */
	int iconoffsets1[7 * 2] = {52, 1, 103, 1, 137, 18, 69, 35, 120, 1, 137, 1, 1, 18};
	int iconoffsets2[7 * 2] = {86, 1, 1, 1, 69, 1, 0, 0, 0, 0, 0, 0, 0, 0};
	int iconoffsets3[7 * 2] = {0, 0, 0, 0, 0, 0, 86, 69, 86, 69, 86, 69, 0, 0};
	char tempstring[256];
	int count;
	
	xoffset = (screen->w - video.field_size * 15) / 2;
	yoffset = (screen->h - video.field_size * 15) / 2;
	
	bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
	fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;

	if (redrawlevel >= REDRAW_INITIALISE) {
		/* Reinitialise some variables */
		bannerframe = capsuleframe = teleportframe = barrierframe = butterflyframe = bearframe = 0;
		blackbearframe = birdframe = gunframe = laserframe = blasterframe = robboframe = 0;

		/* Restore normal key/button repeat values for these keys */
		user_controls[ACTION_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_RIGHT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_RIGHT].interval = KEY_REPEAT_INTERVAL;
	}
	
	if (redrawlevel >= REDRAW_EVERYTHING) {
		/* Reinitialise some variables */
		bannerstate = capsulestate = teleportstate = barrierstate = butterflystate = bearstate = 0;
		blackbearstate = birdstate = gunstate = laserstate = blasterstate = robbostate = 0;

		/* Set entire screen to a colour */
		clear_screen();
		
		if (helpscreenpage == 0) {

			/* Draw the help text */
			for (count = 0; count < TXT_HELP_PAGE0_ROWS; count++) {
				image = TTF_RenderUTF8_Shaded(font, &txt_help_page0[count * TXT_MAX_COLS], fgcolor, bgcolor);
				SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
				if (video.field_size == 16) {
					destrect = set_rect(xoffset + 2, yoffset + 57 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
				} else if (video.field_size == 32) {
					destrect = set_rect(xoffset + 4, yoffset + 104 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
				}
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);
			}

		} else if (helpscreenpage == 1) {
			
			for (count = 0; count < TXT_HELP_PAGE1_ROWS; count++) {
				/* Draw the background tiles for the icons */
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				/* Draw the icons */
				srcrect = set_rect(iconoffsets1[count * 2] * video.field_size / 16, iconoffsets1[count * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				if (count == 4) {
					/* Draw the background tile for the door icon */
					destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size * 8, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
					SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
					/* Draw the door icon */
					srcrect = set_rect(154 * video.field_size / 16, 1 * video.field_size / 16, video.field_size, video.field_size);
					SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				}
				/* Draw the text for the icons */
				image = TTF_RenderUTF8_Shaded(font, &txt_help_page1[count * TXT_MAX_COLS], fgcolor, bgcolor);
				destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size * 1.5, yoffset + 57 * video.field_size / 16 + count * yincrement, image->w, image->h);
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);
			}

		} else if (helpscreenpage == 2) {

			for (count = 0; count < TXT_HELP_PAGE2_ROWS; count++) {
				if (count < 3) {
					/* Draw the background tiles for the icons */
					destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
					SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
					/* Draw the icons */
					srcrect = set_rect(iconoffsets2[count * 2] * video.field_size / 16, iconoffsets2[count * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
					destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
					SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				}
				/* Draw the text for the icons */
				image = TTF_RenderUTF8_Shaded(font, &txt_help_page2[count * TXT_MAX_COLS], fgcolor, bgcolor);
				destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size * 1.5, yoffset + 57 * video.field_size / 16 + count * yincrement, image->w, image->h);
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);
			}

		} else if (helpscreenpage == 3) {

			for (count = 0; count < TXT_HELP_PAGE3_ROWS; count++) {
				if (count > 2 && count < 6) {
					/* Draw the background tiles for the icons */
					destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
					SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
					/* Draw the icons */
					srcrect = set_rect(iconoffsets3[count * 2] * video.field_size / 16, iconoffsets3[count * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
					destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, video.field_size, video.field_size);
					SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				}
				/* Draw the text for the icons */
				image = TTF_RenderUTF8_Shaded(font, &txt_help_page3[count * TXT_MAX_COLS], fgcolor, bgcolor);
				if (count >= 3 && count <= 5) {
					destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size * 6.5, yoffset + 57 * video.field_size / 16 + count * yincrement, image->w, image->h);
				} else {
					destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size * 1.5, yoffset + 57 * video.field_size / 16 + count * yincrement, image->w, image->h);
				}
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);
			}
			
		}

	}
	
	if (redrawlevel >= REDRAW_INTERMEDIATE) {

		/* Exit */
		if (helpscreenmenuposition == 1) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
		}
		sprintf(tempstring, "%s", txt_Exit);
		image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
		destrect = set_rect(xoffset + 240 * video.field_size / 16 / 2 - image->w / 2, yoffset + 220 * video.field_size / 16, image->w, image->h);
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);

		if (helpscreenpage == 0 || helpscreenpage == 1 || helpscreenpage == 2) {

			/* Next > */
			if (helpscreenmenuposition == 2) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			sprintf(tempstring, "%s >", txt_Next);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 238 * video.field_size / 16 - image->w, yoffset + 220 * video.field_size / 16, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

		}

		if (helpscreenpage == 1 || helpscreenpage == 2 || helpscreenpage == 3) {

			/* < Back */
			if (helpscreenmenuposition == 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			sprintf(tempstring, "< %s", txt_Back);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 220 * video.field_size / 16, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}
		
	}

	if (redrawlevel >= REDRAW_ANIMATED) {
		
		if (helpscreenpage == 2) {
			/* Animate the capsule */
			capsulestate--;
			if (capsulestate <= 0) {
				srcrect = set_rect(capsuleoffsets[capsuleframe * 2] * video.field_size / 16, capsuleoffsets[capsuleframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 3 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				capsuleframe++; if (capsuleframe > 1) capsuleframe = 0;
				capsulestate = DELAY_CAPSULE;
			}

			/* Animate the teleport */
			teleportstate--;
			if (teleportstate <= 0) {
				srcrect = set_rect(teleportoffsets[teleportframe * 2] * video.field_size / 16, teleportoffsets[teleportframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 4 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				teleportframe++; if (teleportframe > 1) teleportframe = 0;
				teleportstate = DELAY_TELEPORT;
			}

			/* Animate the barrier */
			barrierstate--;
			if (barrierstate <= 0) {
				srcrect = set_rect(barrieroffsets[barrierframe * 2] * video.field_size / 16, barrieroffsets[barrierframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 5 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				barrierframe++; if (barrierframe > 1) barrierframe = 0;
				barrierstate = DELAY_BARRIER;
			}

			/* Animate the butterfly */
			butterflystate--;
			if (butterflystate <= 0) {
				srcrect = set_rect(butterflyoffsets[butterflyframe * 2] * video.field_size / 16, butterflyoffsets[butterflyframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 6 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				butterflyframe++; if (butterflyframe > 1) butterflyframe = 0;
				butterflystate = DELAY_BUTTERFLY;
			}
			
		} else if (helpscreenpage == 3) {
			
			/* Animate the bear */
			bearstate--;
			if (bearstate <= 0) {
				srcrect = set_rect(bearoffsets[bearframe * 2] * video.field_size / 16, bearoffsets[bearframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				bearframe++; if (bearframe > 1) bearframe = 0;
				bearstate = DELAY_BEAR;
			}

			/* Animate the blackbear */
			blackbearstate--;
			if (blackbearstate <= 0) {
				srcrect = set_rect(blackbearoffsets[blackbearframe * 2] * video.field_size / 16, blackbearoffsets[blackbearframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 1 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				blackbearframe++; if (blackbearframe > 1) blackbearframe = 0;
				blackbearstate = DELAY_BEAR;
			}

			/* Animate the bird */
			birdstate--;
			if (birdstate <= 0) {
				srcrect = set_rect(birdoffsets[birdframe * 2] * video.field_size / 16, birdoffsets[birdframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 2 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				birdframe++; if (birdframe > 1) birdframe = 0;
				birdstate = DELAY_BIRD;
			}
			
			/* Animate the gun */
			gunstate--;
			if (gunstate <= 0) {
				destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size, yoffset + 57 * video.field_size / 16 + 3 * yincrement, video.field_size * 5, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				for (count = 0; count < 5; count++) {
					if (gunoffsets[gunframe * 2] != 0 && gunoffsets[gunframe * 2 + 1] != 0) {
						srcrect = set_rect(gunoffsets[gunframe * 2] * video.field_size / 16, gunoffsets[gunframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
						destrect = set_rect(xoffset + 2 * video.field_size / 16 + (count + 1) * video.field_size, yoffset + 57 * video.field_size / 16 + 3 * yincrement, video.field_size, video.field_size);
						SDL_BlitSurface(icons, &srcrect, screen, &destrect);
					}
					gunframe++; if (gunframe >= 25) gunframe = 0;
				}
				gunstate = DELAY_LASER;
			}

			/* Animate the laser */
			laserstate--;
			if (laserstate <= 0) {
				destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size, yoffset + 57 * video.field_size / 16 + 4 * yincrement, video.field_size * 5, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				for (count = 0; count < 5; count++) {
					if (laseroffsets[laserframe * 2] != 0 && laseroffsets[laserframe * 2 + 1] != 0) {
						srcrect = set_rect(laseroffsets[laserframe * 2] * video.field_size / 16, laseroffsets[laserframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
						destrect = set_rect(xoffset + 2 * video.field_size / 16 + (count + 1) * video.field_size, yoffset + 57 * video.field_size / 16 + 4 * yincrement, video.field_size, video.field_size);
						SDL_BlitSurface(icons, &srcrect, screen, &destrect);
					}
					laserframe++; if (laserframe >= 70) laserframe = 0;
				}
				if (laserframe < 45) {
					laserstate = DELAY_LASER;
				} else {
					laserstate = DELAY_LITTLE_BOOM;
				}
			}

			/* Animate the blaster */
			blasterstate--;
			if (blasterstate <= 0) {
				destrect = set_rect(xoffset + 2 * video.field_size / 16 + video.field_size, yoffset + 57 * video.field_size / 16 + 5 * yincrement, video.field_size * 5, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				for (count = 0; count < 5; count++) {
					if (blasteroffsets[blasterframe * 2] != 0 && blasteroffsets[blasterframe * 2 + 1] != 0) {
						srcrect = set_rect(blasteroffsets[blasterframe * 2] * video.field_size / 16, blasteroffsets[blasterframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
						destrect = set_rect(xoffset + 2 * video.field_size / 16 + (count + 1) * video.field_size, yoffset + 57 * video.field_size / 16 + 5 * yincrement, video.field_size, video.field_size);
						SDL_BlitSurface(icons, &srcrect, screen, &destrect);
					}
					blasterframe++; if (blasterframe >= 45) blasterframe = 0;
				}
				blasterstate = DELAY_LASER;
			}

			/* Animate Robbo */
			robbostate--;
			if (robbostate <= 0) {
				srcrect = set_rect(robbooffsets[robboframe * 2] * video.field_size / 16, robbooffsets[robboframe * 2 + 1] * video.field_size / 16, video.field_size, video.field_size);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 6 * yincrement, video.field_size, video.field_size);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, tilecolour >> 16 & 0xff, tilecolour >> 8 & 0xff, tilecolour & 0xff));
				SDL_BlitSurface(icons, &srcrect, screen, &destrect);
				robboframe++; if (robboframe > 1) robboframe = 0;
				robbostate = DELAY_ROBBO;
			}
		}
		
		/* Animate the Help banner */
		bannerstate--;
		if (bannerstate <= 0) {
			/* Draw the horizontal line under the banner */
			destrect = set_rect(xoffset + 0, yoffset + 43 * video.field_size / 16, 240 * video.field_size / 16, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));

			/* Draw the Help banner */
			if (video.field_size == 16) {
				show_banner(helpscreenbanner, screen->w / 2 - 33 * 3 / 2, yoffset + 6, 33, 10);
			} else if (video.field_size == 32) {
				show_banner(helpscreenbanner, screen->w / 2 - 33 * 5 / 2, yoffset + 19, 33, 10);
			}
			
			bannerstate = DELAY_BANNER;
		}
	}
}

/***************************************************************************
 * Show Options Screen                                                     *
 ***************************************************************************/
/* On entry: redrawlevel = REDRAW_INITIALISE to initialise
			 redrawlevel = REDRAW_EVERYTHING for everything
			 redrawlevel = REDRAW_INTERMEDIATE for menu
			 redrawlevel = REDRAW_ANIMATED for animated objects only */

void show_optionsscreen(int redrawlevel) {
	int xoffset, yoffset, yincrement = video.field_size;
	SDL_Surface *icons_preview;
	char tempstring[256];
	SDL_Rect destrect;
	int count;
	
	xoffset = (screen->w - video.field_size * 15) / 2;
	yoffset = (screen->h - video.field_size * 15) / 2;
	
	bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
	fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;

	if (redrawlevel >= REDRAW_INITIALISE) {
		/* Reinitialise some variables */
		bannerframe = 0;
		temp_game_cycle_limit = game_cycle_limit;
		temp_key_repeat_delay = key_repeat_delay;
		temp_key_repeat_interval = key_repeat_interval;
		temp_default_joystick = default_joystick;
		temp_joystick_dead_zone = joystick_dead_zone;
		temp_selected_skin = selected_skin;
		temp_selected_locale = selected_locale;
		for (count = 0; count < USER_CONTROLS; count++) {
			temp_user_controls[count].device = user_controls[count].device;
			temp_user_controls[count].id = user_controls[count].id;
			temp_user_controls[count].mod = user_controls[count].mod;
		}
		
		/* Restore normal key/button repeat values for these keys */
		user_controls[ACTION_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_RIGHT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_UP].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_UP].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_DOWN].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_DOWN].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_LEFT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_LEFT].interval = KEY_REPEAT_INTERVAL;
		user_controls[ACTION_SHOOT_RIGHT].delay = KEY_REPEAT_DELAY;
		user_controls[ACTION_SHOOT_RIGHT].interval = KEY_REPEAT_INTERVAL;
	}
	
	if (redrawlevel >= REDRAW_EVERYTHING) {
		/* Reinitialise some variables */
		bannerstate = 0;

		/* Set entire screen to a colour */
		clear_screen();
	}
	
	if (redrawlevel >= REDRAW_INTERMEDIATE) {

		if (optionsscreenpage == 0) {

			/* Game Speed */
			if (optionsscreenmenuposition[optionsscreenpage] == 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Game_Speed, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* +< */
			if (temp_game_cycle_limit < GAME_CYCLE_LIMIT_MAX) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " +< ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* n */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			if (temp_game_cycle_limit == 33) {
				sprintf(tempstring, "%s", txt_Fast);
			} else if (temp_game_cycle_limit == 25) {
				sprintf(tempstring, "%s", txt_Normal);
			} else if (temp_game_cycle_limit == 20) {
				sprintf(tempstring, "%s", txt_Slow);
			} else {
				sprintf(tempstring, "%iHz", temp_game_cycle_limit);
			}
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);
		
			/* >- */
			if (temp_game_cycle_limit > GAME_CYCLE_LIMIT_MIN) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >-       ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		
			/* Default Joystick */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (optionsscreenmenuposition[optionsscreenpage] == 1) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Default_Joystick, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 1 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (temp_default_joystick > 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " < ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* Joystick name */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			if (joystick_count == 0) {
				strcpy(tempstring, txt_No_joystick_found);
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (strlen(joystick_list[temp_default_joystick]) <= 28) {
				sprintf(tempstring, "%i:", temp_default_joystick);
				strcat(tempstring, joystick_list[temp_default_joystick]);
			} else {
				sprintf(tempstring, "%i:", temp_default_joystick);
				strncat(tempstring, joystick_list[temp_default_joystick], 28 - 3);
				tempstring[28 + 2 - 4] = '\0';	/* If max characters were copied then the string won't be null terminated */
				strcat(tempstring, "...");
			}
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect.w = 240 * video.field_size / 16; destrect.h = video.field_size;
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* > */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (temp_default_joystick < joystick_count - 1) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >  ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* Joystick Axes Dead Zone */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (optionsscreenmenuposition[optionsscreenpage] == 2) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Joystick_Axes_Dead_Zone, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 2 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (temp_joystick_dead_zone > 1) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " < ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* n% */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			}
			sprintf(tempstring, "%i%%", temp_joystick_dead_zone);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* > */
			if (joystick_count == 0) {
				fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
			} else if (temp_joystick_dead_zone < 99) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >  ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* Key Repeat: */
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, txt_Key_Repeat, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 3 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* Key Repeat: Delay */
			if (optionsscreenmenuposition[optionsscreenpage] == 3) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Delay, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 12 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 4 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (temp_key_repeat_delay > KEY_REPEAT_MIN) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " +< ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* n */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			sprintf(tempstring, "%ims", temp_key_repeat_delay);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* > */
			if (temp_key_repeat_delay < KEY_REPEAT_MAX) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >-      ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
			
			/* Key Repeat: Interval */
			if (optionsscreenmenuposition[optionsscreenpage] == 4) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Interval, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 12 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 5 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (temp_key_repeat_interval > KEY_REPEAT_MIN) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " +< ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* n */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			sprintf(tempstring, "%ims", temp_key_repeat_interval);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* > */
			if (temp_key_repeat_interval < KEY_REPEAT_MAX) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >-      ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* Language */
			if (optionsscreenmenuposition[optionsscreenpage] == 5) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Language, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 6 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (temp_selected_locale > 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " < ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* Locale name */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, locales[temp_selected_locale].name, fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);
		
			/* > */
			if (temp_selected_locale < locale_count - 1) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >       ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* By */
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			sprintf(tempstring, "%s ", txt_Translation_by);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 7 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* Locale author */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, locales[temp_selected_locale].author, fgcolor, bgcolor);
			destrect.w = 240 * video.field_size / 16; destrect.h = video.field_size;
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

		} else if (optionsscreenpage == 1) {

			/* Draw the control reconfiguration help text */
			for (count = 0; count < TXT_OPTIONS_PAGE1_ROWS; count++) {
				fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
				image = TTF_RenderUTF8_Shaded(font, &txt_options_page1[count * TXT_MAX_COLS], fgcolor, bgcolor);
				SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
				if (video.field_size == 16) {
					destrect = set_rect(xoffset + 2, yoffset + 57 + count * (FONTSIZE16 + video.field_size / 8), image->w, image->h);
				} else if (video.field_size == 32) {
					destrect = set_rect(xoffset + 4, yoffset + 104 + count * (FONTSIZE32 + video.field_size / 8), image->w, image->h);
				}
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);
			}

		} else if (optionsscreenpage >= 2 && optionsscreenpage <= 4) {
			
			for (count = 0; count < 9; count++) {
				
				/* Action */
				if (optionsscreenmenuposition[optionsscreenpage] == count) {
					fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
				} else {
					fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
				}
				if (count + (optionsscreenpage - 2) * 9 == ACTION_UP) {
					sprintf(tempstring, "%s ", txt_Move_Up);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_DOWN) {
					sprintf(tempstring, "%s ", txt_Move_Down);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_LEFT) {
					sprintf(tempstring, "%s ", txt_Move_Left);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_RIGHT) {
					sprintf(tempstring, "%s ", txt_Move_Right);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_SHOOT_UP) {
					sprintf(tempstring, "%s ", txt_Shoot_Up);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_SHOOT_DOWN) {
					sprintf(tempstring, "%s ", txt_Shoot_Down);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_SHOOT_LEFT) {
					sprintf(tempstring, "%s ", txt_Shoot_Left);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_SHOOT_RIGHT) {
					sprintf(tempstring, "%s ", txt_Shoot_Right);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_RESTART) {
					sprintf(tempstring, "%s ", txt_Restart);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_SELECT) {
					sprintf(tempstring, "%s ", txt_Confirm);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_EXIT) {
					sprintf(tempstring, "%s ", txt_Exit);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_HELP) {
					sprintf(tempstring, "%s ", txt_Help);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_OPTIONS) {
					sprintf(tempstring, "%s ", txt_Options);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_PREVIOUS_LEVEL) {
					sprintf(tempstring, "%s ", txt_Previous_Level);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_NEXT_LEVEL) {
					sprintf(tempstring, "%s ", txt_Next_Level);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_PREVIOUS_PACK) {
					sprintf(tempstring, "%s ", txt_Previous_Pack);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_NEXT_PACK) {
					sprintf(tempstring, "%s ", txt_Next_Pack);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_TOGGLE_FULLSCREEN) {
					sprintf(tempstring, "%s ", txt_Toggle_Fullscreen);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_MODIFIER1) {
					sprintf(tempstring, "%s1 ", txt_Modifier);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_MODIFIER2) {
					sprintf(tempstring, "%s2 ", txt_Modifier);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_HOME) {
					sprintf(tempstring, "%s ", txt_Home);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_END) {
					sprintf(tempstring, "%s ", txt_End);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_PAGEUP) {
					sprintf(tempstring, "%s ", txt_Page_Up);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_PAGEDOWN) {
					sprintf(tempstring, "%s ", txt_Page_Down);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_VOLUP) {
					sprintf(tempstring, "%s ", txt_Volume_Up);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_VOLDOWN) {
					sprintf(tempstring, "%s ", txt_Volume_Down);
				} else if (count + (optionsscreenpage - 2) * 9 == ACTION_NOT_USED1) {
					sprintf(tempstring, "%s ", txt_Restore_Default_Controls);
				}
				image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
				destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + count * yincrement, 240 * video.field_size / 16, image->h);
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
				destrect.w = image->w; destrect.h = image->h;
				SDL_BlitSurface(image, NULL, screen, &destrect);
				SDL_FreeSurface(image);

				destrect.x = xoffset + 110 * video.field_size / 16;	/* Was 88 */
				
				/* device */
				if (count + (optionsscreenpage - 2) * 9 != ACTION_NOT_USED1) {
					fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
					if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == UNDEFINED) {
						sprintf(tempstring, "%s", "-");
					} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == DEVICE_KEYBOARD) {
						sprintf(tempstring, "%s: ", txt_Key);
					} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == DEVICE_JOYSTICK) {
						sprintf(tempstring, "%s: ", txt_Joy);
					}
					image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
					destrect.w = image->w; destrect.h = image->h;
					SDL_BlitSurface(image, NULL, screen, &destrect);
					destrect.x += image->w;
					SDL_FreeSurface(image);
					
					/* id */
					if (temp_user_controls[count + (optionsscreenpage - 2) * 9].id == UNDEFINED) {
						sprintf(tempstring, "%s ", "-");
					} else {
						if (joystick_count > 0) {
							get_keyjbtn_text(temp_user_controls[count + (optionsscreenpage - 2) * 9].device, temp_user_controls[count + (optionsscreenpage - 2) * 9].id, joystick_list[temp_default_joystick], tempstring);
						} else {
							get_keyjbtn_text(temp_user_controls[count + (optionsscreenpage - 2) * 9].device, temp_user_controls[count + (optionsscreenpage - 2) * 9].id, "", tempstring);
						}
						sprintf(tempstring, "%s ", tempstring);
					}
					image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
					destrect.w = image->w; destrect.h = image->h;
					SDL_BlitSurface(image, NULL, screen, &destrect);
					SDL_FreeSurface(image);

					destrect.x = xoffset + 190 * video.field_size / 16;	/* Was 169 */
					
					/* < */
					if (count + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER1 && count + (optionsscreenpage - 2) * 9 != ACTION_MODIFIER2) {
						if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == UNDEFINED) {
							fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
						} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].mod != UNDEFINED) {
							fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
						} else {
							fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
						}
						image = TTF_RenderUTF8_Shaded(font, "<", fgcolor, bgcolor);
						destrect.w = image->w; destrect.h = image->h;
						SDL_BlitSurface(image, NULL, screen, &destrect);
						destrect.x += image->w;
						SDL_FreeSurface(image);

						/* mod */
						if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == UNDEFINED) {
							fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
						} else {
							fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
						}
						if (temp_user_controls[count + (optionsscreenpage - 2) * 9].mod == UNDEFINED) {
							sprintf(tempstring, "%s", "--");
						} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].mod == ACTION_MODIFIER1) {
							sprintf(tempstring, " %s1 ", txt_Mod);
						} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].mod == ACTION_MODIFIER2) {
							sprintf(tempstring, " %s2 ", txt_Mod);
						}
						image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
						destrect.w = image->w; destrect.h = image->h;
						SDL_BlitSurface(image, NULL, screen, &destrect);
						destrect.x += image->w;
						SDL_FreeSurface(image);

						/* > */
						if (temp_user_controls[count + (optionsscreenpage - 2) * 9].device == UNDEFINED) {
							fgcolor.r = skins[selected_skin].menu_greyed_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_greyed_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_greyed_text_colour & 0xff;
						} else if (temp_user_controls[count + (optionsscreenpage - 2) * 9].mod != ACTION_MODIFIER2) {
							fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
						} else {
							fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
						}
						image = TTF_RenderUTF8_Shaded(font, "> ", fgcolor, bgcolor);
						destrect.w = image->w; destrect.h = image->h;
						SDL_BlitSurface(image, NULL, screen, &destrect);
						SDL_FreeSurface(image);

					}
				}
			}
		} else if (optionsscreenpage == 5) {

			/* Skin */
			if (optionsscreenmenuposition[optionsscreenpage] == 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, txt_Skin, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 0 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* < */
			if (temp_selected_skin > 0) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " < ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* Skin name */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, skins[temp_selected_skin].name, fgcolor, bgcolor);
			destrect.w = 240 * video.field_size / 16; destrect.h = video.field_size;
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* > */
			if (temp_selected_skin < skin_count - 1) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_selected_limit_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_limit_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_limit_text_colour & 0xff;
			}
			image = TTF_RenderUTF8_Shaded(font, " >  ", fgcolor, bgcolor);
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* By */
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			sprintf(tempstring, "%s ", txt_By);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 1 * yincrement, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			destrect.x += image->w;
			SDL_FreeSurface(image);

			/* Skin author */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, skins[temp_selected_skin].author, fgcolor, bgcolor);
			destrect.w = 240 * video.field_size / 16; destrect.h = video.field_size;
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			destrect.w = image->w; destrect.h = image->h;
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

			/* Skin about */
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 57 * video.field_size / 16 + 2 * yincrement, 240 * video.field_size / 16, 3 * video.field_size);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			for (count = 0; count < 3; count++) {
				image = TTF_RenderUTF8_Shaded(font, skins[temp_selected_skin].about[count], fgcolor, bgcolor);
				SDL_SetColorKey(image, SDL_SRCCOLORKEY, SDL_MapRGB(image->format, bgcolor.r, bgcolor.g, bgcolor.b));
				destrect.w = image->w; destrect.h = image->h;
				SDL_BlitSurface(image, NULL, screen, &destrect);
				if (video.field_size == 16) {
					destrect.y += FONTSIZE16 + video.field_size / 8;
				} else if (video.field_size == 32) {
					destrect.y += FONTSIZE32 + video.field_size / 8;
				}
				SDL_FreeSurface(image);
			}

			/* Draw the icons preview image */
			icons_preview = create_skin_preview();
			destrect = set_rect(screen->w / 2 - icons_preview->w / 2, yoffset + 57 * video.field_size / 16 + 5 * video.field_size, video.field_size, video.field_size);
			SDL_BlitSurface(icons_preview, NULL, screen, &destrect);
			SDL_FreeSurface(icons_preview);
		}

		/* Page navigation */
		if (optionsscreenpage > 0) {

			/* < Back */
			if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_LEFT) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			sprintf(tempstring, "< %s", txt_Back);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 2 * video.field_size / 16, yoffset + 220 * video.field_size / 16, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);
		}

		/* Save */
		if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_SAVE) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, txt_Save, fgcolor, bgcolor);
		destrect = set_rect(xoffset + 90 * video.field_size / 16 - image->w / 2, yoffset + 220 * video.field_size / 16, image->w, image->h);
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);

		/* Exit */
		if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_EXIT) {
			fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
		} else {
			fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
		}
		image = TTF_RenderUTF8_Shaded(font, txt_Exit, fgcolor, bgcolor);
		destrect = set_rect(xoffset + 150 * video.field_size / 16 - image->w / 2, yoffset + 220 * video.field_size / 16, image->w, image->h);
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);

		if (optionsscreenpage < OPTIONS_SCREEN_PAGES - 1) {
			/* Page > */
			if (optionsscreenmenuposition[optionsscreenpage] == OPTIONS_MENUPOS_RIGHT) {
				fgcolor.r = skins[selected_skin].menu_selected_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_selected_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_selected_text_colour & 0xff;
			} else {
				fgcolor.r = skins[selected_skin].menu_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].menu_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].menu_text_colour & 0xff;
			}
			sprintf(tempstring, "%s >", txt_Next);
			image = TTF_RenderUTF8_Shaded(font, tempstring, fgcolor, bgcolor);
			destrect = set_rect(xoffset + 238 * video.field_size / 16 - image->w, yoffset + 220 * video.field_size / 16, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			SDL_FreeSurface(image);

		}
	}
	
	if (redrawlevel >= REDRAW_ANIMATED) {
		/* Animate the Options banner */
		if (bannerstate <= 0) {
			/* Draw the horizontal line under the banner */
			destrect = set_rect(xoffset + 0, yoffset + 43 * video.field_size / 16, 240 * video.field_size / 16, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));

			/* Draw the Options banner */
			if (video.field_size == 16) {
				show_banner(optionsscreenbanner, screen->w / 2 - 60 * 3 / 2, yoffset + 6, 60, 10);
			} else if (video.field_size == 32) {
				show_banner(optionsscreenbanner, screen->w / 2 - 60 * 5 / 2, yoffset + 19, 60, 10);
			}
			
			bannerstate = DELAY_BANNER;
		} else {
			bannerstate--;
		}
	}
}

/***************************************************************************
 * Show Message Box                                                        *
 ***************************************************************************/
 /* This displays one very simple message box over the screen at once. If there
    currently exists a message box with a different name then it is destroyed and
    the screen rebuilt, else a message box is drawn over the existing one.
    
    Examples :-
    To initialise: show_message_box(REDRAW_INITIALISE, "bla", 50, "Boo!", TRUE, 0, 0);
    To redraw after you've rebuilt your screen: show_message_box(REDRAW_EVERYTHING, "", 0, "", 0, 0, 0);
    To kill an existing box: show_message_box(REDRAW_INITIALISE, "", -1, "", 0, 0, 0);
    
    On entry: redrawlevel = REDRAW_INITIALISE to initialise a message box
			  name = unique identifier
			  timeout = units of TICK_INTERVAL
			  message = message
			  dynamic = TRUE to fit to the size of the text
			  dynamic = FALSE for an exact size with
			    width = box width
			    height = box height
			  redrawlevel = REDRAW_EVERYTHING to show it (all the other values are ignored)
			  (the timeout is decremented here and so is the rebuilding of the screen) */

void show_message_box(int redrawlevel, char *name, int timeout, char *message, int dynamic, int width, int height) {
	SDL_Rect destrect;
	static char boxname[256] = "", boxmessage[256];
	static int boxtimeout = -1, boxdynamic, boxwidth, boxheight, bannerstate, bannerframe;
	static int boxx, boxy, boxw, boxh, bartlx, bartly, barbrx, barbry;

	if (redrawlevel >= REDRAW_INITIALISE) {

		/* Restore/rebuild the screen if there is an existing
		   message box showing which is not name */
		if ((boxtimeout >= 0) && (strcmp(boxname, name) != 0)) {
			if (game_mode == INTRO_SCREEN) {
				show_introscreen(REDRAW_EVERYTHING);
			} else if (game_mode == GAME_ON) {
				show_game_area(REDRAW_INITIALISE);
			} else if (game_mode == END_SCREEN) {
				show_endscreen();
			} else if (game_mode == HELP_SCREEN) {
				show_helpscreen(REDRAW_EVERYTHING);
			} else if (game_mode == OPTIONS_SCREEN) {
				show_optionsscreen(REDRAW_EVERYTHING);
			}
			bannerframe = 0;	/* This retains the animation frame through updates */
		}

		/* Store passed values */
		strcpy(boxname, name);
		boxtimeout = timeout;
		if ((strlen(message)) == 0) {
			strcpy(boxmessage, " ");	/* This stops SDL_ttf from seg faulting on NULL strings >( */
		} else {
			strcpy(boxmessage, message);
		}
		boxdynamic = dynamic;
		boxwidth = width;
		boxheight = height;

		/* Render the text so we can get its size in image->w and image->h */
		bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
		fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
		image = TTF_RenderUTF8_Shaded(font, boxmessage, fgcolor, bgcolor);

		/* Calculate some dimensions */
		if (boxdynamic) {
			boxw = image->w + video.field_size;
			boxh = image->h + video.field_size;
		} else {
			boxw = boxwidth;
			boxh = boxheight;
		}
		SDL_FreeSurface(image);
		#ifdef DEBUG_MESSAGE_BOX
			printf("*** Start %s ***\n", __func__);
			printf("boxw=%i boxh=%i\n", boxw, boxh);
			printf("*** Stop %s ***\n", __func__);
		#endif
		boxx = (screen->w - boxw) / 2;
		boxy = (screen->h - boxh) / 2;
		bartlx = boxx + video.field_size / 8;
		bartly = boxy + video.field_size / 8;
		barbrx = boxx + boxw - video.field_size / 4;
		barbry = boxy + boxh - video.field_size / 4;

		/* Initialise some variables */
		bannerstate = 0;
		
	}

	if (redrawlevel >= REDRAW_EVERYTHING) {

		if (boxtimeout > 0) {
			boxtimeout--;
			
			/* Draw box background */
			bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
			destrect = set_rect(boxx, boxy, boxw, boxh);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bgcolor.r, bgcolor.g, bgcolor.b));
			
			/* Draw top bar */
			destrect = set_rect(bartlx, bartly, barbrx - bartlx, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			
			/* Draw bottom bar */
			destrect = set_rect(bartlx, barbry, barbrx - bartlx, video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			
			/* Draw left bar */
			destrect = set_rect(bartlx, bartly, video.field_size / 8, barbry - bartly);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			
			/* Draw right bar */
			destrect = set_rect(barbrx, bartly, video.field_size / 8, barbry - bartly + video.field_size / 8);
			SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[bannerframe] >> 16 & 0xff, bannercolour[bannerframe] >> 8 & 0xff, bannercolour[bannerframe] & 0xff));
			
			/* Colour cycle the bars */
			if (bannerstate <= 0) {
				bannerstate = DELAY_BANNER;
				bannerframe++; if (bannerframe >= BANNER_COLOURS) bannerframe = 0;
			} else {
				bannerstate--;
			}
			
			/* Render the text so we can write it */
			bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
			fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;
			image = TTF_RenderUTF8_Shaded(font, boxmessage, fgcolor, bgcolor);

			/* Write message */
			destrect = set_rect(boxx + (boxw - image->w) / 2, boxy + (boxh - image->h) / 2, image->w, image->h);
			SDL_BlitSurface(image, NULL, screen, &destrect);
			
			SDL_FreeSurface(image);

		} else if (boxtimeout == 0) {
			boxtimeout--;

			/* Restore/rebuild the screen */
			if (game_mode == INTRO_SCREEN) {
				show_introscreen(REDRAW_EVERYTHING);
			} else if (game_mode == GAME_ON) {
				show_game_area(REDRAW_INITIALISE);
			} else if (game_mode == END_SCREEN) {
				show_endscreen();
			} else if (game_mode == HELP_SCREEN) {
				show_helpscreen(REDRAW_EVERYTHING);
			} else if (game_mode == OPTIONS_SCREEN) {
				show_optionsscreen(REDRAW_EVERYTHING);
			}
		}
	}
}

/***************************************************************************
 * Show Banner                                                             *
 ***************************************************************************/
/* On entry: banner points to the banner array
			 xpos and ypos are the screen offsets to draw it
			 width and height are the dimensions of the banner array */

void show_banner(char *banner[], int xpos, int ypos, int width, int height) {
	SDL_Rect destrect;
	int x, y, frame = bannerframe;
	
	for(y = 0; y < height; y++) {
		for(x = 0; x < width; x++) {
			if (banner[y][x] != '.') {
				if (video.field_size == 16) {
					destrect = set_rect(x * 3 + xpos, y * 4 + ypos, 2, 3);
				} else {
					destrect = set_rect(x * 5 + xpos, y * 7 + ypos, 4, 6);
				}
				SDL_FillRect(screen, &destrect, SDL_MapRGB(screen->format, bannercolour[frame] >> 16 & 0xff, bannercolour[frame] >> 8 & 0xff, bannercolour[frame] & 0xff));
			}
		}
		frame++; if (frame >= BANNER_COLOURS) frame = 0;
	}

	bannerframe++; if (bannerframe >= BANNER_COLOURS) bannerframe = 0;
}

/***************************************************************************
 * Write LCD Char                                                          *
 ***************************************************************************/
/* As we only have uppercase letters, numbers and a hyphon spread across
   multiple surfaces, this function will manage them.
   On entry: lcdchar = char to write
			 xpos and ypos are the screen offsets to draw it */

void write_lcd_char(char lcdchar, int xpos, int ypos) {
	SDL_Rect srcrect, destrect;

	destrect = set_rect(xpos, ypos, video.field_size / 2, video.field_size);

	/* Force lowercase uppercase */
	if (lcdchar >= 'a' && lcdchar <= 'z') lcdchar -= 0x20;
	
	if (lcdchar >= 'A' && lcdchar <= 'Z') {
		srcrect = set_rect((lcdchar - 'A') * (video.field_size / 2 + video.field_size / 16), 0, video.field_size / 2, video.field_size);
		SDL_BlitSurface(alpha, &srcrect, screen, &destrect);
	} else if (lcdchar >= '0' && lcdchar <= '9') {
		srcrect = set_rect((lcdchar - '0') * (video.field_size / 2 + video.field_size / 16), 0, video.field_size / 2, video.field_size);
		SDL_BlitSurface(ciphers, &srcrect, screen, &destrect);
	} else if (lcdchar == '-') {
		srcrect = set_rect(26 * (video.field_size / 2 + video.field_size / 16), 0, video.field_size / 2, video.field_size);
		SDL_BlitSurface(alpha, &srcrect, screen, &destrect);
	} else {
		/* Drop everything else */
	}
}

/***************************************************************************
 * Show End Screen                                                         *
 ***************************************************************************/
/* Currently this only shows a congratulatory message although I may add
   some funny animations such as Robbo being chased across the screen by bears
   and the capsule zooming up screen etc. */

void show_endscreen(void) {
	SDL_Rect destrect;
	int count, yoffset = screen->h / 2 - video.field_size * 3 / 2;

	clear_screen();

	bgcolor.r = skins[selected_skin].background_colour >> 16 & 0xff; bgcolor.g = skins[selected_skin].background_colour >> 8 & 0xff; bgcolor.b = skins[selected_skin].background_colour & 0xff;
	fgcolor.r = skins[selected_skin].general_text_colour >> 16 & 0xff; fgcolor.g = skins[selected_skin].general_text_colour >> 8 & 0xff; fgcolor.b = skins[selected_skin].general_text_colour & 0xff;

	for (count = 0; count < TXT_ENDSCREEN_CONGRATS_ROWS; count++) {
		image = TTF_RenderUTF8_Shaded(font, &txt_endscreen_congrats[count * TXT_MAX_COLS], fgcolor, bgcolor);
		destrect = set_rect(screen->w / 2 - image->w / 2, yoffset + count * video.field_size, image->w, image->h);
		SDL_BlitSurface(image, NULL, screen, &destrect);
		SDL_FreeSurface(image);	
	}

}








